/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.Frame;
import com.datastax.driver.core.Message;
import com.datastax.driver.core.Segment;
import com.datastax.shaded.netty.buffer.ByteBuf;
import com.datastax.shaded.netty.buffer.ByteBufAllocator;
import com.datastax.shaded.netty.channel.ChannelHandlerContext;
import com.datastax.shaded.netty.channel.ChannelPromise;
import com.datastax.shaded.netty.util.concurrent.Future;
import com.datastax.shaded.netty.util.concurrent.GenericFutureListener;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SegmentBuilder {
    private static final Logger logger = LoggerFactory.getLogger(SegmentBuilder.class);
    private final ChannelHandlerContext context;
    private final ByteBufAllocator allocator;
    private final int maxPayloadLength;
    private final Message.ProtocolEncoder requestEncoder;
    private final List<Frame.Header> currentPayloadHeaders = new ArrayList<Frame.Header>();
    private final List<Message.Request> currentPayloadBodies = new ArrayList<Message.Request>();
    private final List<ChannelPromise> currentPayloadPromises = new ArrayList<ChannelPromise>();
    private int currentPayloadLength;

    SegmentBuilder(ChannelHandlerContext context, ByteBufAllocator allocator, Message.ProtocolEncoder requestEncoder) {
        this(context, allocator, requestEncoder, Segment.MAX_PAYLOAD_LENGTH);
    }

    @VisibleForTesting
    SegmentBuilder(ChannelHandlerContext context, ByteBufAllocator allocator, Message.ProtocolEncoder requestEncoder, int maxPayloadLength) {
        this.context = context;
        this.allocator = allocator;
        this.requestEncoder = requestEncoder;
        this.maxPayloadLength = maxPayloadLength;
    }

    public void addRequest(Message.Request request, ChannelPromise promise) {
        int frameHeaderLength = Frame.Header.lengthFor(this.requestEncoder.protocolVersion);
        int frameBodyLength = this.requestEncoder.encodedSize(request);
        int frameLength = frameHeaderLength + frameBodyLength;
        Frame.Header header = new Frame.Header(this.requestEncoder.protocolVersion, this.requestEncoder.computeFlags(request), request.getStreamId(), request.type.opcode, frameBodyLength);
        if (frameLength > this.maxPayloadLength) {
            ByteBuf frame = this.allocator.ioBuffer(frameLength);
            header.encodeInto(frame);
            this.requestEncoder.encode(request, frame);
            int sliceCount = frameLength / this.maxPayloadLength + (frameLength % this.maxPayloadLength == 0 ? 0 : 1);
            logger.trace("Splitting large request ({} bytes) into {} segments: {}", new Object[]{frameLength, sliceCount, request});
            List<ChannelPromise> segmentPromises = this.split(promise, sliceCount);
            int i = 0;
            do {
                ByteBuf part = frame.readSlice(Math.min(this.maxPayloadLength, frame.readableBytes()));
                part.retain();
                this.process(part, false, segmentPromises.get(i++));
            } while (frame.isReadable());
            frame.release();
        } else {
            if (this.currentPayloadLength + frameLength > this.maxPayloadLength) {
                this.processCurrentPayload();
                this.resetCurrentPayload();
            }
            logger.trace("Adding {}th request to self-contained segment: {}", (Object)(this.currentPayloadHeaders.size() + 1), (Object)request);
            this.currentPayloadHeaders.add(header);
            this.currentPayloadBodies.add(request);
            this.currentPayloadPromises.add(promise);
            this.currentPayloadLength += frameLength;
        }
    }

    public void flush() {
        if (this.currentPayloadLength > 0) {
            this.processCurrentPayload();
            this.resetCurrentPayload();
        }
    }

    protected void processSegment(Segment segment, ChannelPromise segmentPromise) {
        this.context.write(segment, segmentPromise);
    }

    private void process(ByteBuf payload, boolean isSelfContained, ChannelPromise segmentPromise) {
        this.processSegment(new Segment(payload, isSelfContained), segmentPromise);
    }

    private void processCurrentPayload() {
        int requestCount = this.currentPayloadHeaders.size();
        assert (this.currentPayloadBodies.size() == requestCount && this.currentPayloadPromises.size() == requestCount);
        logger.trace("Emitting new self-contained segment with {} frame(s)", (Object)requestCount);
        ByteBuf payload = this.allocator.ioBuffer(this.currentPayloadLength);
        for (int i = 0; i < requestCount; ++i) {
            Frame.Header header = this.currentPayloadHeaders.get(i);
            Message.Request request = this.currentPayloadBodies.get(i);
            header.encodeInto(payload);
            this.requestEncoder.encode(request, payload);
        }
        this.process(payload, true, this.merge(this.currentPayloadPromises));
    }

    private void resetCurrentPayload() {
        this.currentPayloadHeaders.clear();
        this.currentPayloadBodies.clear();
        this.currentPayloadPromises.clear();
        this.currentPayloadLength = 0;
    }

    private ChannelPromise merge(List<ChannelPromise> framePromises) {
        if (framePromises.size() == 1) {
            return framePromises.get(0);
        }
        ChannelPromise segmentPromise = this.context.newPromise();
        final ImmutableList dependents = ImmutableList.copyOf(framePromises);
        segmentPromise.addListener((GenericFutureListener<? extends Future<? super Void>>)new GenericFutureListener<Future<? super Void>>(){

            @Override
            public void operationComplete(Future<? super Void> future) throws Exception {
                if (future.isSuccess()) {
                    for (ChannelPromise framePromise : dependents) {
                        framePromise.setSuccess();
                    }
                } else {
                    Throwable cause = future.cause();
                    for (ChannelPromise framePromise : dependents) {
                        framePromise.setFailure(cause);
                    }
                }
            }
        });
        return segmentPromise;
    }

    private List<ChannelPromise> split(ChannelPromise framePromise, int sliceCount) {
        ArrayList<ChannelPromise> slicePromises = new ArrayList<ChannelPromise>(sliceCount);
        for (int i = 0; i < sliceCount; ++i) {
            slicePromises.add(this.context.newPromise());
        }
        SliceWriteListener sliceListener = new SliceWriteListener(framePromise, slicePromises);
        for (int i = 0; i < sliceCount; ++i) {
            ((ChannelPromise)slicePromises.get(i)).addListener(sliceListener);
        }
        return slicePromises;
    }

    static class SliceWriteListener
    implements GenericFutureListener<Future<Void>> {
        private final ChannelPromise parentPromise;
        private final List<ChannelPromise> slicePromises;
        private int remainingSlices;

        SliceWriteListener(ChannelPromise parentPromise, List<ChannelPromise> slicePromises) {
            this.parentPromise = parentPromise;
            this.slicePromises = slicePromises;
            this.remainingSlices = slicePromises.size();
        }

        @Override
        public void operationComplete(Future<Void> future) {
            if (!this.parentPromise.isDone()) {
                if (future.isSuccess()) {
                    --this.remainingSlices;
                    if (this.remainingSlices == 0) {
                        this.parentPromise.setSuccess();
                    }
                } else {
                    this.parentPromise.setFailure(future.cause());
                    for (ChannelPromise slicePromise : this.slicePromises) {
                        slicePromise.cancel(false);
                    }
                }
            }
        }
    }
}

