'use strict'

const net = require('net');
const zlib = require('zlib');
const moment = require('moment');
const EventEmitter = require('events');

const logger = require('../logger');
const appUtils = require('../app-utils');
const config = require('../default_config');
const constant = require('../app-constants');
const safeStringify = require('../utils/stringify');

const NEXT_LINE = '\r\n';
let MetricsSchema = null;
const appendMsg = 'Socket:';
class Event extends EventEmitter { };

/**
 * Socket is responsible for send the datas to the agent
 * as a string.
 */
function Socket() {
  this.ip = null;
  this.port = null;
  this.socket = null;
  this.worerInfo = '';
  this.triedPortNo = [];
  this.runTimeInfo = null;
  this.event = new Event();
  this.isConnected = false;
  this.stopedTracing = false;
  this.isComponentVerified = false;
  this.noOfConnectionAttemptMade = 0;
  this.lastConnectionVerified = new Date();
}

async function loadProtobufSchema() {
  try {
    const protobuf = require("protobufjs");
    const root = await protobuf.load(__dirname + '/../../protobuf/metrics.proto');
    MetricsSchema = root.lookupType("Metrics");
  } catch (e) {
    logger.error(appendMsg, `can't load schema`, e);
  }
}

Socket.prototype.start = async function (config, runTimeInfo) {
  this.config = config;
  this.runTimeInfo = runTimeInfo;
  this.ip = config && config.agent_host;
  this.port = config && config.agent_port;
  this.ip = this.ip === 'localhost' ? '127.0.0.1' : this.ip;
  this.reservedPort = config.agent_reserved_ports.split(',').map(e => parseInt(e));

  if (runTimeInfo.mode === 'CLUSTER' && runTimeInfo.workerCount > 0) {
    this.worerInfo = constant.MESSAGE_DELIMITER + runTimeInfo.workerId + ',' + runTimeInfo.workerCount;
  }

  if (!this.ip || !this.port) {
    logger.error(appendMsg, 'not a valid socket ip or port');
    return;
  }

  if (this.config.pay_load_message_type === 'PROTOBUF') {
    await loadProtobufSchema();
  }

  if (this.reservedPort.indexOf(this.port) === -1) this.reservedPort.push(this.port);
  _startConnectionWatcher.call(this);
  _triggerHeartBeat.call(this);
}

Socket.prototype.stop = function () {
  _stopConnectionWatcher.call(this);
  _destroyClient.call(this);
}

Socket.prototype.isReadyToTransmit = function () {
  return this.socket && this.isConnected && this.isComponentVerified && !this.dataSending && this.config.unique_component_id;
}

function _startConnectionWatcher() {
  const _self = this;
  _stopConnectionWatcher.call(this);
  this.differentPortCheckInterval = _getTimeout.call(this) * 2;
  const componentCheck = this.config.component_check_interval || config.component_check_interval;
  _setConnCheckInterval.call(this);
  _attemptConnection.call(this);

  /** check whether the component is avaliable in agent for every 60 sec if not verified with agent */
  this.validComponentWatcher = setInterval(() => {
    if (_self.isConnected && !_self.isComponentVerified && _self.config.unique_component_id) {
      _sendPing.call(_self);
    }
    _checkAgentInactiveTime.call(_self);
  }, componentCheck);

  _makeConnection.call(this);
  _tryDifferntPort.call(this);
  this.noOfConnectionAttemptMade = 1;
}

function _setConnCheckInterval() {
  this.connCheckInterval = this.config.connection_check_interval || config.connection_check_interval;
  if (this.differentPortCheckInterval * this.reservedPort.length > this.connCheckInterval) {
    this.connCheckInterval = this.differentPortCheckInterval * this.reservedPort.length + 10;
  }
}

function _attemptConnection() {
  /** check wether the agent is avaliable for every 30 sec if not connected */
  const maxFastAttempt = this.config.no_of_max_fast_connection_attempt || config.no_of_max_fast_connection_attempt;
  const maxIntervel = this.config.connection_check_max_interval || config.connection_check_max_interval;

  this.connectionWatcher = setInterval(() => {
    if (this.isConnected) return;

    this.triedPortNo = [];
    this.port = this.config.agent_port;
    this.noOfConnectionAttemptMade++;
    logger.info(appendMsg, `Making the ${this.noOfConnectionAttemptMade} attempt after ${this.connCheckInterval / 1000} sec`);
    _makeConnection.call(this);
    _tryDifferntPort.call(this);

    // only for first 10 times, the profiler will try to connect for every 30 s. after it is 60s, 120s, 240s, 480s then always it is 960s
    if (this.noOfConnectionAttemptMade >= maxFastAttempt) {
      clearInterval(this.connectionWatcher);
      this.connCheckInterval = this.connCheckInterval * 2;
      if (this.connCheckInterval > maxIntervel) this.connCheckInterval = maxIntervel;
      _attemptConnection.call(this);
    }
  }, this.connCheckInterval);
}

function _checkAgentInactiveTime() {
  if (this.isReadyToTransmit() || this.stopedTracing) return;
  const lastConnectedTimeDiff = moment().diff(this.lastConnectionVerified, 'milliseconds');
  const maxInactiveTime = this.config.agent_max_inactive_time || config.agent_max_inactive_time;

  if (lastConnectedTimeDiff > maxInactiveTime) {
    _info(appendMsg, `For the past ${lastConnectedTimeDiff / (1000 * 60)} minutes, agent is not connected. so we are going to empty the queue and stop the Infra and BTM tracing`);
    this.stopedTracing = true;
    this.event.emit('stopTracing');
  }
}

/**
  * the agent is not connected to one port then the profiler will try with other reserved port
  */
function _tryDifferntPort() {
  const _self = this;
  _clearDifferentPortWatcher.call(this);

  setTimeout(_ => {
    _self.differentPortWatcher = setInterval(_ => {
      if (_self.isConnected) return;
      _self.triedPortNo.push(_self.port);
      const port = _getNextPortNo.call(_self);
      /** if you didn't try any port then try that immediately */
      if (port) {
        _self.port = port;
        _makeConnection.call(_self);
      } else if (_self.triedPortNo.length) {
        _self.triedPortNo = [];
        logger.info(appendMsg, 'Tried all port but not able to connect');
        _clearDifferentPortWatcher.call(this);
      }
    }, this.differentPortCheckInterval);
  });
}

function _clearDifferentPortWatcher() {
  if (this.differentPortWatcher) {
    clearInterval(this.differentPortWatcher);
    this.differentPortWatcher = null;
  }
}

function _stopConnectionWatcher(_self) {
  if (this.connectionWatcher) {
    clearInterval(this.connectionWatcher);
  }
  if (this.validComponentWatcher) {
    clearInterval(this.validComponentWatcher);
  }
  if (this.differentPortWatcher) {
    clearInterval(this.differentPortWatcher);
  }
}

function _destroyClient() {
  if (!this.socket) return;
  try {
    this.socket.end();
    this.socket.destroy();
  } catch (e) {
    logger.error(appendMsg, 'agent socket client destroy error ', e);
  }
}

function _makeConnection() {
  if (this.isPingSend) {
    this.isPingSend = false;
    _info(appendMsg, `No valid response for the ping`);
  }

  const _self = this;
  try {
    _destroyClient.call(this);
    _debug(appendMsg, `making a new connection to ${this.ip}:${this.port}`);

    this.socket = net.createConnection({
      host: this.ip,
      port: this.port,
      allowHalfOpen: true,
      timeout: _getTimeout.call(this),
    }, function () {
      const address = _self.socket.address() || {};
      _info(appendMsg, `connection is established with ${_self.ip}:${_self.port} from => ${address.address}:${address.port}`);

      _sendPing.call(_self);
      _listenData.call(_self);
    });

    this.setTimeout();
    this.socket.setKeepAlive(true);

    this.socket.on('end', () => {
      _onConnectionFailure.call(_self);
      logger.error(appendMsg, 'agent connection ended');
    });

    this.socket.on('error', e => {
      _onConnectionFailure.call(_self);
      logger.error(appendMsg, 'socket error', e.message);
    });

    this.socket.on('close', _ => {
      _onConnectionFailure.call(_self);
    });

    this.socket.on('drain', _ => {
      _self.dataSending = false;
    });

    this.socket.unref();
  } catch (e) {
    _onConnectionFailure.call(_self);
    logger.error(appendMsg, 'socket error', e);
  }
}

function _onConnectionFailure() {
  this.dataSending = false;
  this.isConnected = false;
  this.isComponentVerified = false;
}

function _getNextPortNo() {
  for (let i = 0; i < this.reservedPort.length; i++) {
    const port = this.reservedPort[i];
    if (this.triedPortNo.indexOf(port) === -1) {
      return port;
    }
  }
  return null;
}

function _sendPing() {
  if (!this.socket) return;
  this.isPingSend = true;
  _info(appendMsg, `Verifying whether the connected socket client is EG agent or not`);
  if (this.config.unique_component_id) {
    // sample: OFFLOAD - PING!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c
    _sendData.call(this, constant.PING_MSG + constant.MESSAGE_DELIMITER + this.config.unique_component_id + NEXT_LINE);
  } else {
    _sendData.call(this, constant.PING_MSG + NEXT_LINE);
  }
}

function _requestConfig(isRerequest) {
  if (!this.isReadyToTransmit()) return;
  _info(appendMsg, `Going to send config request`);
  // sample: HeartBeatCheck!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c!;-;!eGTest_Config_Request
  if (!isRerequest) {
    this.testConfigReceived = false;
    this.propsConfigReceived = false;
  }

  _sendData.call(this, constant.HEART_BEAT_MSG + constant.MESSAGE_DELIMITER + this.config.unique_component_id + this.worerInfo + constant.MESSAGE_DELIMITER + constant.CONFIG_REQUEST + NEXT_LINE);
  //console.log("Config request sent for " + process.pid, ' - ', this.worerInfo);
}

function _triggerHeartBeat() {
  if (this.heartBeatRepeater) {
    clearInterval(this.heartBeatRepeater);
    this.heartBeatRepeater = null;
  }

  const _self = this;
  const interval = this.config.heart_beat_interval;
  this.heartBeatRepeater = setInterval(_ => {
    _sendHeartBeat.call(_self);
  }, interval);
}

function _sendHeartBeat() {
  if (!this.isReadyToTransmit()) return;
  _info(appendMsg, `Sending heart beat message`);
  // sample: HeartBeatCheck!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c!
  _sendData.call(this, constant.HEART_BEAT_MSG + constant.MESSAGE_DELIMITER + this.config.unique_component_id + NEXT_LINE);
}

function _listenData() {
  if (!this.socket) return;
  const _self = this;

  this.socket.on('data', data => {

    try {
      _debug(appendMsg, `data from agent is`, data && data.toString() || data);
      if (!data) return;

      const response = data.toString().trim().split(constant.MESSAGE_DELIMITER);
      response.forEach(res => {
        if (!res) return;
        if (res === constant.PONG_MSG) {
          _self.isPingSend = false;
          _self.isConnected = true;
          _info(appendMsg, `connected with EG agent running on => ${_self.ip}:${_self.port}`);
          _self.isComponentVerified = false;
          _sendPriorityData.call(_self);
        } else if (res === constant.COMP_FOUND || constant.COMP_FOUNDS.indexOf(res) > -1) {
          _self.isPingSend = false;
          _onConnection.call(_self);
        } else if (res === constant.COMP_NOT_FOUND) {
          _info(appendMsg, `EG agent is rejected this Component`);
          _self.isPingSend = false;
          _self.isConnected = true;
          _self.isComponentVerified = false;
        } else if (res === constant.HEART_BEAT_ACK) {
          _info(appendMsg, `Heart Beat Acknowledged`);
        } else if (res === constant.DISCOVERY_DETAILS_RECEIVED) {
          _info(appendMsg, `The agent is acknowledged that it is received the discovery details.`);
        } else if (res === constant.DISCOVERY_DETAILS_FINALIZED) {
          _info(appendMsg, `The agent is acknowledged that it is finalized the discovery details.`);
        } else if (res.startsWith(constant.COMPONENT_DISCOVERD)) {
          _onComponentChange.call(this, res);
        } else {
          const agentConfig = _getValidConfig.call(this, res);
          if (agentConfig) {
            _onConfigChange.call(this, agentConfig);
            this.event.emit('startTracing');
          } else {
            _debug(appendMsg, `Some invalid data send from agent...! `, res);
          }
        }
      });
    } catch (e) {
      logger.error(appendMsg, 'data receiver error ', e);
    }
  });
}

function _onComponentChange(data) {
  const uniqueComponentId = data.split(constant.COMPONENT_DISCOVERD)[1];
  if (!uniqueComponentId) return;

  const guidNickPort = uniqueComponentId.split(constant.DICOVERY_DELIMITER);
  logger.info(appendMsg, `Component Details from the agent is ${guidNickPort[0]} and Nick port is ${guidNickPort[1]}`);
  this.event.emit('onNewComponent', {
    guid: guidNickPort[0],
    nickPort: guidNickPort[1],
  });
}

const configKey = ['PropsTable', 'TestConfig', 'ThresholdConfig', 'ModifiedPropsTable', 'modifiedTestConfig', 'modifiedThresholdConfig', 'CommonGuidConfig'];
function _getValidConfig(data) {
  if (!this.isReadyToTransmit()) return;
  const isValidConfig = configKey.find(e => data.indexOf(e) > -1);
  if (!isValidConfig) return;
  let newConfig = null;

  try {
    data = JSON.parse(data);
    data = data || [];

    data.forEach(e => {
      const props = e.PropsTable || e.ModifiedPropsTable;
      const testConfig = e.TestConfig || e.modifiedTestConfig;
      const thresholdConfig = e.ThresholdConfig || e.modifiedThresholdConfig;

      if (e && props) {
        this.propsConfigReceived = true;
        //console.log("propsConfigReceived received for the process id " + process.pid, ' - ', this.worerInfo);
        const modifiedProps = {};
        Object.keys(props).forEach(key => {
          modifiedProps[key.toString().toLowerCase()] = props[key];
        });

        _setConfig(modifiedProps);
        _info(appendMsg, `Sending config ack for PropsTable`);
        // sample: ACK4BTM_!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c!;-;!NODEJS_BTM_PROPS_OK!;-;!
        _sendData.call(this, constant.ACK + constant.MESSAGE_DELIMITER + this.config.unique_component_id + this.worerInfo + constant.MESSAGE_DELIMITER + constant.PROPS_ACK + constant.MESSAGE_DELIMITER + NEXT_LINE);
      }

      if (e && testConfig) {
        _setConfig(testConfig);
        this.testConfigReceived = true;
        //console.log("testConfigReceived received for the process id " + process.pid, ' - ', this.worerInfo);
        _info(appendMsg, `Sending config ack for TestConfig`);
        // sample: ACK4BTM_!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c!;-;!NODEJS_OFFLOAD_CONFIG_OK!;-;!
        _sendData.call(this, constant.ACK + constant.MESSAGE_DELIMITER + this.config.unique_component_id + this.worerInfo + constant.MESSAGE_DELIMITER + constant.CONFIG_ACK + constant.MESSAGE_DELIMITER + NEXT_LINE);
      }

      if (e && thresholdConfig) {
        if (thresholdConfig.Threshold === `NodeTransTest`) {
          _setConfig({ thresholdConfig });
          _info(appendMsg, `Sending config ack for ThresholdConfig - NodeTransTest`);
          // sample: ACK4BTM_!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c!;-;!NODEJS_BTM_THRESH_OK!;-;!
          _sendData.call(this, constant.ACK + constant.MESSAGE_DELIMITER + this.config.unique_component_id + this.worerInfo + constant.MESSAGE_DELIMITER + constant.THRESH_ACK + constant.MESSAGE_DELIMITER + NEXT_LINE);
        } else if (thresholdConfig.Threshold === `KNodeTransTest`) {
          _setConfig({ keyThresholdConfig: thresholdConfig });
          _info(appendMsg, `Sending config ack for ThresholdConfig - KNodeTransTest`);
          // sample: ACK4BTM_!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c!;-;!NODEJS_KBTM_THRESH_OK!;-;!
          _sendData.call(this, constant.ACK + constant.MESSAGE_DELIMITER + this.config.unique_component_id + this.worerInfo + constant.MESSAGE_DELIMITER + constant.KTHRESH_ACK + constant.MESSAGE_DELIMITER + NEXT_LINE);
        }
      }

      if (e && e.CommonGuidConfig) {
        _setConfig(e.CommonGuidConfig);
        _info(appendMsg, `Sending config ack for CommonGuidConfig`);
        // sample: ACK4BTM_!;-;!ecozxmw5q4yh5typqkioh3ww6pafrf62:5d90328c-75f9-4400-a9d7-65f7ae2fd05c!;-;!COMMON_GUID_CONFIG_OK!;-;!
        _sendData.call(this, constant.ACK + constant.MESSAGE_DELIMITER + this.config.unique_component_id + this.worerInfo + constant.MESSAGE_DELIMITER + constant.COM_CONFIG_ACK + constant.MESSAGE_DELIMITER + NEXT_LINE);
      }
    });

    return newConfig;
  } catch (e) {
    logger.error(appendMsg, 'changeConfig json parser error:', e, data);
  }

  function _setConfig(config) {
    if (!newConfig) newConfig = {};
    Object.assign(newConfig, config);
  }
}

function _onConfigChange(data) {
  this.event.emit('configChanged', data);
}

function _onConnection() {
  this.lastConnectionVerified = new Date();
  if (this.isComponentVerified) return;

  _info(appendMsg, `Component is verified with EG agent`);
  this.stopedTracing = false;
  this.isConnected = true;
  this.isComponentVerified = true;
  _setConnCheckInterval.call(this);
  this.noOfConnectionAttemptMade = 1;
  this.event.emit('startTracing');
  _requestConfig.call(this);
}

function _info() {
  logger.info.apply(this, arguments);
}

function _debug() {
  logger.debug.apply(this, arguments);
}

Socket.prototype.setConfig = function (_config) {
  this.config = _config;
}

Socket.prototype.setTimeout = function () {
  const timeout = _getTimeout.call(this);
  this.socket.setTimeout(timeout);
}

function _getTimeout() {
  return this.config.socket_timeout || config.socket_timeout;
}

Socket.prototype.resetConnection = function () {
  this.dataSending = false;
  this.isConnected = false;
  this.isComponentVerified = false;
}

function _sendPriorityData() {
  if (!this.dataToSendAfterConnection) return;
  const { metrics, isStringMsg } = this.dataToSendAfterConnection;
  this.send(metrics, true, isStringMsg);
  this.dataToSendAfterConnection = null;
}

Socket.prototype.send = async function (metrics, isSendAfterConnection, isStringMsg, isFile) {
  if (!this.isConnected) {
    if (isSendAfterConnection) {
      this.dataToSendAfterConnection = { metrics, isStringMsg };
    }
    return;
  }

  this.dataSending = true;
  if (isStringMsg) {
    _sendData.call(this, metrics + NEXT_LINE);
    return;
  }

  let payload = null;
  const _self = this;

  try {
    if (this.config.pay_load_message_type === 'PROTOBUF') {
      payload = _getProtobufEncodedData.call(this, metrics);
    } else {
      payload = _getJsonStringifyData.call(this, metrics);
    }
  } catch (e) {
    logger.error(appendMsg, 'error while encoding data', e);
    return;
  }
  if (!payload) return;

  if (this.config.is_zipped_payload === true) {
    try {
      payload = await zlib.gzipSync(payload);
      payload = payload.toString('base64') + NEXT_LINE;
    } catch (e) {
      logger.error(appendMsg, 'error while zipping data', e);
      return;
    }
  }

  _sendData.call(this, payload, isFile);

  if (!_self.isReadyToTransmit() || (_self.testConfigReceived && _self.propsConfigReceived) || _self.isWaitingForConfig) return;
  _self.isWaitingForConfig = true;

  setTimeout(_ => {
    if (!_self.isReadyToTransmit()) return;
    _self.isWaitingForConfig = false;
    if (!_self.testConfigReceived || !_self.propsConfigReceived) {
      logger.info(appendMsg, `Some config is not received even after 2 min. testConfigReceived: ${_self.testConfigReceived}, propsConfigReceived: ${_self.propsConfigReceived}`);
      //console.log(`Some config is not received even after 2 min. testConfigReceived: ${_self.testConfigReceived}, propsConfigReceived: ${_self.propsConfigReceived} - ${_self.worerInfo}`);
      _requestConfig.call(_self, true);
    }
  }, 2 * 60 * 1000);
}

function _getProtobufEncodedData(payload) {
  const metrics = {
    componentId: this.config.unique_component_id,
    infra: [],
    btm: [],
  };

  payload.forEach(e => {
    if (e.btm) metrics.btm.push(e.btm);
    if (e.infra) metrics.infra.push(e.infra);
  });

  const errMsg = MetricsSchema.verify(metrics);
  if (errMsg) {
    logger.error(appendMsg, 'error while verifing data', e);
    logger.debug(appendMsg, 'data is trying to send', metrics);
    return;
  }

  const message = MetricsSchema.create(metrics);
  const buffer = MetricsSchema.encodeDelimited(message).finish();
  logger.info(appendMsg, `${metrics.infra.length} Infr and ${metrics.btm.length} BTM metric going to send to agent`);
  //return buffer.toString('hex');
  return buffer;
}

function _getJsonStringifyData(msg) {
  msg.forEach(e => {
    if (this.config.unique_component_id) {
      e.componentId = this.config.unique_component_id;
    }

    if (e.infra && e.infra.memoryUsage && e.infra.memoryUsage.heapSpaceDetails) {
      const heapSpaceDetails = {};
      e.infra.memoryUsage.heapSpaceDetails.forEach(he => {
        heapSpaceDetails[he.name] = he.used;
      });

      e.infra.memoryUsage.heapSpaceDetails = heapSpaceDetails;
    }
  });

  let payload = safeStringify(msg);
  payload += NEXT_LINE
  const isInfra = payload.indexOf('infra') > -1;
  logger.debug(appendMsg, (isInfra ? 1 : 0) + ' no of Infr and ' + (isInfra ? msg.length - 1 : msg.length) + ' BTM metric going to send to agent');
  //console.log((isInfra ? 1 : 0) + ' no of Infr and ' + (isInfra ? msg.length - 1 : msg.length) + ' BTM metric going to send to agent for ', this.worerInfo)
  return payload;
}

/**
 * To Send a cpu and memory dump files to the agent 
 *  here we write the data as a stream 
 * we restricted max dump size <25 Mb
 * @param {string} type  memory || cpu dump
 * @param {string} filePath abssoute location of the dump file 
 */
Socket.prototype.sendFile = function (type, payload) {
  if (!type || !payload) return;
  if (this.dataSending) {
    const _self = this;
    setTimeout(_ => {
      _sendFile.call(_self, type, payload);
    });
  }

  _sendFile.call(this, type, payload);
}

function _sendFile(type, payload) {
  //Just dropping the data , if connection is not available
  if (!this.isReadyToTransmit()) {
    logger.warn(appendMsg, `agent is not connected so '${type}' data dump file is dropped`)
    return;
  }

  this.send([payload], false, false, true);
}

function _sendData(data, isFile) {
  try {
    if (!isFile) logger.debug(appendMsg, 'MSG to agent', data);
    else logger.debug(appendMsg, 'File is sent to the agent');
    this.dataSending = true;
    const isFlushed = this.socket.write(data);

    if (isFlushed) {
      this.dataSending = false;
    }
  } catch (e) {
    _onConnectionFailure.call(this);
    logger.error(appendMsg, 'error while sending data', e);
  }
}

module.exports = Socket;