$Global:ErrorActionPreference="SilentlyContinue"
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Add-PSSnapin "Citrix.Common.Commands"
Add-PSSnapin "Citrix.Host.Admin.V2"
Add-PSSnapin "Citrix.Broker.Admin.V2"
Add-PSSnapin "Citrix.ADIdentity.Admin.V2"
Add-PSSnapin "Citrix.Configuration.Admin.V2"
Add-PSSnapin "Citrix.MachineCreation.Admin.V2"
Add-PSSnapin "Citrix.Licensing.Admin.V1"
Add-PSSnapin "Citrix.DelegatedAdmin.Admin.V1"
Add-PSSnapin "Citrix.Monitor.Admin.V1"
Add-PSSnapin "Citrix.ConfigurationLogging.Admin.V1"
Add-PSSnapin "Citrix.AppLibrary.Admin.V1"

<#
Function Get-BrokerApplicationsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($applications.count / $total * 100, 1)
		#Write-Host  "Getting Broker Applications. Retrieved $($applications.count) of $total applications completed in $remainingPerc%"
		$last = $applications[-1].Uid
		$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $applications
}
#>

Function Get-BrokerApplicationsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$LastUid = 0;
	do
	{
		$NewApplications = @(Get-BrokerApplication -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewApplications.Count -gt 0)
		{
			$applications += $NewApplications
			$LastUid = $applications[-1].Uid
		}
	} while ($NewApplications.Count -ge $MaxRecordCount)
	return $applications
}

$ZoneControllersHash = @{}
$ConfiguredZones = Get-ConfigZone -EA 0
foreach($Zone in $ConfiguredZones)
{
	$ZoneControllersHash.Add($Zone.Name,$Zone.ControllerSids)
}
Write-Host "Sites"
Write-Host "-----"
$site = Get-BrokerSite
if($site -ne $null)
{
	if($site.BaseOU -eq $null)
	{
		$ADOU = "-"
	}
	else
	{
		$ADOU = $site.BaseOU
	}

	Write-Host ($site.Name,$ADOU,$site.DnsResolutionEnabled,$site.TrustRequestsSentToTheXmlServicePort,$site.SecureIcaRequired,$site.ConnectionLeasingEnabled) -Separator "#~#"
}

Write-Host "Administrators"
Write-Host "--------------"
$brokerAdmins = Get-AdminAdministrator
if($brokerAdmins -ne $null)
{
	foreach($brokerAdmin in $brokerAdmins)
	{
		$RightsArr = $brokerAdmin.Rights
		$RoleName = $RightsArr[0].RoleName
		$ScopeName = $RightsArr[0].ScopeName
		$RoleObject = Get-AdminRole $RoleName
		$PermissionsArr = $RoleObject.Permissions
		$ScopeObject = Get-AdminScope $ScopeName
		$count = $PermissionsArr.Count
		for($c=0;$c -le $count; $c++)
		{
			if($c -eq 0)
			{
				$permissions = $PermissionsArr[$c]
			}
			else
			{
				$permissions = $permissions + ", " + $PermissionsArr[$c]
			}
		}

		if ($RoleName -eq $null -or $RoleName.length -eq 0)
		{
			$RoleName = "-"
		}

		if ($permissions -eq $null -or $permissions.length -eq 0)
		{
			$permissions = "-"
		}

		if ($ScopeName -eq $null -or $ScopeName.length -eq 0)
		{
			$ScopeName = "-"
		}
		Write-Host ($brokerAdmin.Name,$brokerAdmin.Enabled,$ScopeName,$RoleName,$PermissionsArr.BuiltIn,
		$permissions,$ScopeObject.BuiltIn) -Separator "#~#"
	}
}

Write-Host "Controllers"
Write-Host "------------"
$controllers = Get-BrokerController
if($controllers -ne $null)
{
	foreach($controller in $controllers)
	{
		$servicesArr = $controller.ActiveSiteServices
		$count = $servicesArr.Count
		$ControllerSid = $controller.SID
		for($c=0;$c -le $count; $c++)
		{
			if($c -eq 0)
			{
				$services = $servicesArr[$c]
			}
			else
			{
				$services = $services + "," + $servicesArr[$c]
			}
		}
		$ControllerZoneName = "-"
		if($ZoneControllersHash.Count -gt 0)
		{
			$ZoneKeys = $ZoneControllersHash.KEYS.GetEnumerator()
			foreach($ZoneKey in $ZoneKeys)
			{
				$ZoneValues = $ZoneControllersHash[$ZoneKey]
				if($ZoneValues.Contains($ControllerSid) -eq $true)
				{
					$ControllerZoneName = $ZoneKey
				}
			}
		}
		Write-Host ($controller.DNSName, $controller.ControllerVersion, $controller.DesktopsRegistered, $controller.State, $services,
				$ControllerZoneName) -Separator "#~#"
	}
}

Write-Host "Hosts"
Write-Host "-----"
$hyphosts = get-BrokerHypervisorConnection
if($hyphosts -ne $null)
{
	foreach($hyphost in $hyphosts)
	{
		$a = "XDHyp:\Connections\"
		$b = $a + $hyphost.Name
		$CapabilityList = $hyphost.Capabilities

		if ($CapabilityList -eq $null -or $CapabilityList.length -eq 0)
		{
			$Capabilities = "-"
		}
		else
		{
			$Capabilities = ""
			foreach($capability in $CapabilityList)
			{
				if ($capability -ne $null -or $capability.length -ne 0)
				{
					$Capabilities = $Capabilities + "," +$capability
				}
			}
			if ($Capabilities.startsWith(","))
			{
				$Capabilities = $Capabilities.substring(1)
			}
		}
		$HypObject = Get-Item $b
		if($?  -and $HypObject)
		{
			$ConnectionType = $HypObject.ConnectionType
			$HypervisorAddress = $HypObject.HypervisorAddress
			$HypervisorConnectionName = $HypObject.HypervisorConnectionName
			$MaintenanceMode = $HypObject.MaintenanceMode
			$Metadata = $HypObject.MetadataMap


			if ($Metadata -eq $null -or $Metadata.length -eq 0)
			{
				$metadataList = "-"
			}
			else
			{
				$metadataList = ""
				foreach($key in $Metadata.KEYS)
				{
					$temp = $key + "=" + $Metadata.item($key)
					$metadataList = $metadataList + ", " + $temp
				}
				if ($metadataList.startsWith(","))
				{
					$metadataList = $metadataList.substring(1)
				}
			}
			$Persistent = $HypObject.Persistent
			$SupportsLocalStorageCaching = $HypObject.SupportsLocalStorageCaching
			$SupportsPvsVMs = $HypObject.SupportsPvsVMs
			$UserName = $HypObject.UserName
			$UsesCloudInfrastructure = $HypObject.UsesCloudInfrastructure
		}
		$HypHostName = $hyphost.Name
		$PreferredController = $hyphost.PreferredController
		$State = $hyphost.State

		Write-Host ($HypHostName, $ConnectionType, $HypervisorAddress, $State, $PreferredController,  $Capabilities,
		 $HypervisorConnectionName, $MaintenanceMode, $metadataList, $Persistent, $SupportsLocalStorageCaching,
		$SupportsPvsVMs, $UserName, $UsesCloudInfrastructure) -Separator "#~#"
	}
}

Write-Host
Write-Host "Licensing"
Write-Host "---------"
$LicensingGraceHoursLeft = $site.LicensingGraceHoursLeft
if($LicensingGraceHoursLeft -eq $null -or $LicensingGraceHoursLeft.length -eq 0)
{
	$LicensingGraceHoursLeft = "-"
}
Write-Host ($site.Name,$site.LicenseServerName, $site.LicenseServerPort, $site.LicenseEdition, $site.LicenseModel,
		$site.LicensingBurnIn, $site.LicensingGracePeriodActive ,$site.LicensingOutOfBoxGracePeriodActive, $LicensingGraceHoursLeft) -Separator "#~#"


Write-Host "XenDesktop Licenses"
Write-Host "--------------------"
$LicenseCertHash = $(Get-LicCertificate -AdminAddress $site.LicenseServerName).CertHash
$licenses = get-licinventory -adminaddress $site.LicenseServerName -CertHash $LicenseCertHash
if($licenses)
{
	foreach($license in $licenses)
	{
		$ProductName = $license.LocalizedLicenseProductName
		$Model = $license.LocalizedLicenseModel
		if(($Model -eq $null) -or ($Model.length -le 0))
		{
			$Model = "-"
		}
		$ExpirationDate = $license.LicenseExpirationDate
		if(($ExpirationDate -eq $null) -or ($ExpirationDate.length -le 0))
		{
			$ExpirationDate = "-"
		}

		$SubscriptionAdvantageDate = $license.LicenseSubscriptionAdvantageDate
		$LicenseType = $license.LocalizedLicenseType
		$Total = $license.LicensesAvailable

		Write-Host ($ProductName, $Model, $ExpirationDate, $SubscriptionAdvantageDate, $LicenseType, $Total) -Separator "#~#"
	}
}

Write-Host "Catalogs"
Write-Host "---------"
$catalogs = Get-BrokerCatalog -MaxRecordCount 2147483647
if($catalogs -ne $null)
{
	foreach($catalog in $catalogs)
	{
		$Name = $catalog.Name
		$Description = $catalog.Description
		$SessionSupport = $catalog.SessionSupport
		$IsPhysical = $catalog.MachinesArePhysical
		$IsRemotePC = $catalog.IsRemotePC


		$MachineType = "-"
		if($SessionSupport -eq 'SingleSession')
		{
			$MachineType = "Windows Desktop OS"
		}
		else
		{
			$MachineType = "Windows Server OS"
		}
		if($IsPhysical)
		{
			$MachineType = $MachineType + " (Physical)"
		}
		else
		{
			$MachineType = $MachineType + " (Virtual)"
		}

		$ProvisioningMethod = "-"
		if($catalog.ProvisioningType -eq "MCS")
		{
			$ProvisioningMethod = "Machine creation services"
		}
		else
		{
			$ProvisioningMethod = $catalog.ProvisioningType
		}

		$AllocationType = "-"
		if($catalog.AllocationType -eq "Static")
		{
			$AllocationType = "Permanent"
		}
		else
		{
			$AllocationType = $catalog.AllocationType
		}

		$Scopes = "-"
		if($catalog.Scopes)
		{
			$Scopes = $catalog.Scopes
		}
		else
		{
			$Scopes = "All"
		}

		$UserData = "-"
		if($catalog.PersistUserChanges -eq "OnLocal")
		{
			$UserData = "On local disk"
		}
		elseif($catalog.PersistUserChanges -eq "OnPvd")
		{
			$UserData = "On personal vDisk"
		}
		else
		{
			$UserData = "Discarded"
		}
		$PvsAddress = $catalog.PvsAddress
		$PvsDomain = $catalog.PvsDomain

		if ($Description -eq $null -or $Description.length -eq 0)
		{
			$Description = "-"
		}
		Write-Host ($Name, $Description,$MachineType, $ProvisioningMethod, $AllocationType, $Scopes, $UserData,$IsRemotePC,$PvsAddress,
				$PvsDomain, $catalog.ZoneName) -Separator "#~#"
	}
}

Write-Host "Delivery Groups configuration"
Write-Host "-----------------------------"
$DeliveryGroupUUIDHash = @{}
$desktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 2147483647
if($desktopGroups -ne $null)
{
	foreach($group in $desktopGroups)
	{
		$DeliveryGroupUUID = $group.UUID
		$DeliveryType = $group.DeliveryType
		$DesktopKind = $group.DesktopKind
		$MaintenanceMode = $group.InMaintenanceMode
		$AutomaticPowerOnForAssigned = $group.AutomaticPowerOnForAssigned
		$AutomaticPowerOnForAssignedDuringPeak = $group.AutomaticPowerOnForAssignedDuringPeak
		$ShutdownDesktopsAfterUse = $group.ShutdownDesktopsAfterUse
		$PublishedName = $group.PublishedName
		$Name = $group.Name
		$Enabled = $group.Enabled
		$ColorDepth = $group.ColorDepth
		$TimeZone = $group.TimeZone
		$EnableSecureICA = $group.SecureIcaRequired
		$OffPeakBufferSizePercent = $group.OffPeakBufferSizePercent
		$OffPeakDisconnectAction = $group.OffPeakDisconnectAction
		$OffPeakDisconnectTimeout = $group.OffPeakDisconnectTimeout
		$OffPeakExtendedDisconnectAction = $group.OffPeakExtendedDisconnectAction
		$OffPeakExtendedDisconnectTimeout = $group.OffPeakExtendedDisconnectTimeout
		$OffPeakLogoffAction = $group.OffPeakLogOffAction
		$OffPeakLogoffTimeout = $group.OffPeakLogOffTimeout
		$PeakBufferSizePercent = $group.PeakBufferSizePercent
		$PeakDisconnectAction = $group.PeakDisconnectAction
		$PeakDisconnectTimeout = $group.PeakDisconnectTimeout
		$PeakExtendedDisconnectAction = $group.PeakExtendedDisconnectAction
		$PeakExtendedDisconnectTimeout = $group.PeakExtendedDisconnectTimeout
		$PeakLogoffAction = $group.PeakLogOffAction
		$PealLogoffTimeout = $group.PeakLogOffTimeout

		$DeliveryGroupUUIDHash.Add($DeliveryGroupUUID,$Name)

		$Scopes = "-"
		$DGscopes = $group.Scopes
		if($DGscopes)
		{
			foreach($scope in $DGscopes)
			{
				$Scopes = $Scopes+", "+$scope
			}
			if($Scopes.startsWith(","))
			{
				$Scopes = $Scopes.substring(1)
			}
		}
		else
		{
			$Scopes = "All"
		}

		$SecureIcaRequired = $group.SecureIcaRequired
		$SessionSupport = $group.SessionSupport
		$MachineType = "-"
		if($SessionSupport -eq 'SingleSession')
		{
			$MachineType = "Windows Desktop OS"
		}
		else
		{
			$MachineType = "Windows Server OS"
		}

		$AppDiskUIDs = $group.AppDisks
		if($AppDiskUIDs -eq $null -or $AppDiskUIDs.count -eq 0)
		{
			$AppDiskNames = "-"
		}
		else
		{
			$AppDiskNames = ""
			foreach($AppDiskUID in $AppDiskUIDs)
			{
				if($AppDiskUidMap.ContainsKey($AppDiskUID) -eq $True)
				{
					$AppDiskNames += $AppDiskUidMap[$AppDiskUID]
				}
			}
		}
		$TurnOnAddedMachine = $group.TurnOnAddedMachine

		Write-Host ($Name,$DeliveryType, $DesktopKind, $Enabled, $MaintenanceMode, $PublishedName,
		$ColorDepth, $TimeZone, $EnableSecureICA, $Scopes, $SecureIcaRequired, $MachineType, $AutomaticPowerOnForAssigned,
		$AutomaticPowerOnForAssignedDuringPeak, $ShutdownDesktopsAfterUse, $OffPeakBufferSizePercent, $OffPeakDisconnectAction,
		$OffPeakDisconnectTimeout, $OffPeakExtendedDisconnectAction, $OffPeakExtendedDisconnectTimeout, $OffPeakLogoffAction,
		$OffPeakLogoffTimeout, $PeakBufferSizePercent, $PeakDisconnectAction, $PeakDisconnectTimeout, $PeakExtendedDisconnectAction,
		$PeakExtendedDisconnectTimeout, $PeakLogoffAction, $PealLogoffTimeout, $TurnOnAddedMachine, $AppDiskNames) -Separator "#~#"
	}
}


Write-Host "Application"
Write-Host "------------"
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
$PublishedAppInstances = Get-BrokerApplicationsPaged
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		
		$AdminFolderName = $PublishedAppInstance.AdminFolderName
		$ApplicationName = $PublishedAppInstance.ApplicationName
		$ApplicationType = $PublishedAppInstance.ApplicationType
		$BrowserName = $PublishedAppInstance.BrowserName
		$AssociatedDesktopGroupUUIDs = $PublishedAppInstance.AssociatedDesktopGroupUUIDs
		
		if($AssociatedDesktopGroupUUIDs -eq $null -or $AssociatedDesktopGroupUUIDs.count -eq 0)
		{
			$DeliveryGroups = "-"
		}
		else
		{
			$DeliveryGroups = ""
			foreach($DesktopGroupUUID in $AssociatedDesktopGroupUUIDs)
			{
				$DGName = $DeliveryGroupUUIDHash.Get_Item($DesktopGroupUUID)
				if ($Description -ne $null -and $Description.length -ne 0)
				{
					$DeliveryGroups = $DeliveryGroups + "," + $DGName
				}
			}
			if($DeliveryGroups.startsWith(","))
			{
				$DeliveryGroups = $DeliveryGroups.substring(1)
			}
		}

		$Description = $PublishedAppInstance.Description
		if ($Description -eq $null -or $Description.length -eq 0)
		{
			$Description = "-"
		}
		$DisplayName = "-"
		$WorkingDirectory = $PublishedAppInstance.WorkingDirectory
		if ($WorkingDirectory -eq $null -or $WorkingDirectory.length -eq 0)
		{
			$WorkingDirectory = "-"
		}

		$CommandLineExecutable = $PublishedAppInstance.CommandLineExecutable
		$CommandLineArgs = $PublishedAppInstance.CommandLineArguments
		$AppVCommandLineArgs = "-"
		if ($CommandLineExecutable -ne $null -and ($CommandLineExecutable -ieq "CtxAppVLauncher.exe"))
		{
			$MetadataMap = $PublishedAppInstance.MetadataMap
			if ($MetadataMap -ne $null)
			{
				$AppID = $MetadataMap.get_Item("Citrix_DesktopStudio_AppID")
				$PkgID = $MetadataMap.get_Item("Citrix_DesktopStudio_PackageId")
				if ($AppID -ne $null -and $AppID.length -gt 0 -and $PkgID -ne $null -and $PkgID.length -gt 0)
				{
					$Idx = $PkgID.LastIndexOf("_")
					if ($Idx -gt 0)
					{
						$PkgID = $PkgID.Substring(0,$Idx)
					}
					$AppVCommandLineArgs = "CtxAppVLauncher.exe" + "#" + $AppID + "#" + $PkgID
				}
			}
		}
		if ($CommandLineArgs)
		{
			$CommandLineExecutable = $CommandLineExecutable + " " + $CommandLineArgs
		}

		# $NameAndBrowserName = $BrowserName + " (" +$PublishedAppInstance.Name + ")"
		$ClientFolder = $PublishedAppInstance.ClientFolder
		$CpuPriorityLevel = $PublishedAppInstance.CpuPriorityLevel
		$IconFromClient = $PublishedAppInstance.IconFromClient
		$PublishedName = $PublishedAppInstance.PublishedName
		$SecureCmdLineArgumentsEnabled = $PublishedAppInstance.SecureCmdLineArgumentsEnabled
		$ShortcutAddedToDesktop = $PublishedAppInstance.ShortcutAddedToDesktop
		$ShortcutAddedToStartMenu = $PublishedAppInstance.ShortcutAddedToStartMenu
		$UUID = $PublishedAppInstance.UUID
		$UserFilterEnabled = $PublishedAppInstance.UserFilterEnabled
		$WaitForPrinterCreation = $PublishedAppInstance.WaitForPrinterCreation

		Write-Host ($PublishedAppInstance.Name, $Description, $PublishedName, $AdminFolderName, $ApplicationName, 
		$PublishedAppInstance.ApplicationType,$PublishedAppInstance.CpuPriorityLevel, $ClientFolder,
		$CommandLineExecutable, $AppVCommandLineArgs, $SecureCmdLineArgumentsEnabled, $IconFromClient,
		$ShortcutAddedToDesktop, $ShortcutAddedToStartMenu, $PublishedAppInstance.Enabled, $PublishedAppInstance.Uid,
		$UUID, $UserFilterEnabled, $WaitForPrinterCreation, $PublishedAppInstance.Visible, $WorkingDirectory, $DeliveryGroups) -Separator "#~#"
	}
}

$AppDiskUidMap = @{}
Write-Host "AppDisks"
Write-Host "-----------"
$AppDisks = Get-AppLibAppDisk -MaxRecordCount 2147483647
if($AppDisks -ne $null)
{
	foreach($AppDisk in $AppDisks)
	{
		$AppDiskUidMap[$AppDisk.AppDiskUid] = $AppDisk.AppDiskName
		$PackageString = ""
		$Packages = $AppDisk.Packages
		if($Packages -eq $null -or $Packages.Count -eq 0)
		{
			$PackageString = "-#-#-"
		}
		else
		{
			foreach($Package in $Packages)
			{
				$str1 = $Package.Name
				$str2 = $Package.Manufacturer
				$str3 = $Package.Version
				$PackageString += $str1+"#"+$str2 + "#"+$str3+"$=$"
			}
		}
		if($PackageString.endsWith("$=$"))
		{
			$PackageString = $PackageString.substring(0,$PackageString.length-3)			
		}
		Write-Host($AppDisk.AppDiskName,$AppDisk.State,$AppDisk.HypervisorConnectionName,$AppDisk.VirtualDiskId,$PackageString) -Separator ";"
	}
}

Write-Host "Zones"
Write-Host "------"
foreach($Zone in $ConfiguredZones)
{
	Write-Host($Zone.Name,$Zone.IsPrimary) -Separator ";"
}

Write-Host "#EG_SCRIPT_DONE#"

Remove-PSSnapin "Citrix.Common.Commands"
Remove-PSSnapin "Citrix.Host.Admin.V2"
Remove-PSSnapin "Citrix.Broker.Admin.V2"
Remove-PSSnapin "Citrix.ADIdentity.Admin.V2"
Remove-PSSnapin "Citrix.Configuration.Admin.V2"
Remove-PSSnapin "Citrix.MachineCreation.Admin.V2"
Remove-PSSnapin "Citrix.Licensing.Admin.V1"
Remove-PSSnapin "Citrix.DelegatedAdmin.Admin.V1"
Remove-PSSnapin "Citrix.Monitor.Admin.V1"
Remove-PSSnapin "Citrix.ConfigurationLogging.Admin.V1"
Remove-PSSnapin "Citrix.AppLibrary.Admin.V1"


