'use strict'
const http = require('http');
const semver = require('semver');

const logger = require('../../logger');
const shimmer = require('../shimmer');
const symbols = require('../../symbols');
const httpShared = require('../http-shared');

let isWrapped = false;
const appendMsg = 'Express:'
const layerPatchedSymbol = Symbol('layer-patched');

exports.start = function wrapExpress(express, agent, version, enabled) {
  if (!enabled) return express;

  try {
    if (!semver.satisfies(semver.coerce(version), '>=4.0.0')) {
      logger.error(appendMsg, 'express version not supported, current version is ', version);
      return express;
    }

    isWrapped = true;
    // express 5 moves the router methods onto a prototype
    const routerProto = semver.satisfies(version, '>=5')
      ? (express.Router && express.Router.prototype)
      : express.Router;

    /**
    * monkey patching the express router to handle error in any API request
    */
    if (routerProto && routerProto.handle) {
      logger.debug(appendMsg, 'shimming express.Router.handle function');
      shimmer.wrap(routerProto, 'handle', wrapHandle);
    }

    /**
    * monkey patching the express router to use error in any API request
    * this will be trigggered only if the client application used app.use fn to handle error.
    * if they didn't handle the error then above wrapHandle will handle the error
    */
    if (routerProto && routerProto.use) {
      shimmer.wrap(routerProto, 'use', orig => {
        return function use(path) {
          var route = orig.apply(this, arguments);
          var layer = this.stack[this.stack.length - 1];
          patchLayer(layer, typeof path === 'string' && path);
          return route;
        }
      })
    }

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return express;

  function wrapHandle(original) {
    return function wrappedHandle() {
      const args = [];
      let req, res;

      for (let i = 0; i < arguments.length; i++) {
        const arg = arguments[i];
        if (typeof arg === 'function') {
          args.push(wrapHandleFn(arg, req, res));
        } else {
          if (arg instanceof http.IncomingMessage) req = arg;
          if (arg instanceof http.ServerResponse) res = arg;
          args.push(arg);
        }
      }

      return original.apply(this, args);
    };
  }

  function wrapHandleFn(fn, req, res) {
    return function wrappedHandleFn(err) {
      if (err && err.message && err.stack) {
        logger.debug(appendMsg, "Error captured in handle fn:", err);
        agent.captureError(err, { res });
      }
      return fn.apply(this, arguments);
    };
  }

  function patchLayer(layer, layerPath) {
    if (layer && !layer[layerPatchedSymbol]) {
      layer[layerPatchedSymbol] = true
      logger.debug(appendMsg, 'shimming express.Router.Layer.handle function:', layer.name)

      shimmer.wrap(layer, 'handle', function (orig) {
        let handle

        if (orig.length === 4) {
          handle = function egErrorHandler(err, req, res, next) {
            agent.captureError(err, { res });
            return orig.apply(this, arguments);
          }
        } else {
          handle = function (req, res, next) {
            if (!layer.route && layerPath && typeof next === 'function') {
              safePush(req, symbols.expressMountStack, layerPath)
              arguments[2] = function () {
                if (!(req.route && arguments[0] instanceof Error)) {
                  req[symbols.expressMountStack].pop()
                }
                return next.apply(this, arguments)
              }
            }

            const headersSent = res.headersSent;
            const orgRes = orig.apply(this, arguments);

            if (headersSent !== res.headersSent && res.headersSent === true && res.statusCode === 404) {
              const trans = httpShared.getTransaction(res);
              if (trans) trans.requestIdentifier.isRequestRedirect = true;
            }

            return orgRes;
          }
        }

        for (const prop in orig) {
          if (Object.prototype.hasOwnProperty.call(orig, prop)) {
            handle[prop] = orig[prop]
          }
        }

        return handle
      })
    }
  }

  function safePush(obj, prop, value) {
    if (!obj[prop]) obj[prop] = []
    obj[prop].push(value)
  }
}

exports.stop = function (express, version) {
  if (!isWrapped) return;
  const routerProto = semver.satisfies(version, '^5')
    ? (express.Router && express.Router.prototype)
    : express.Router

  if (routerProto && routerProto.handle) {
    shimmer.unwrap(routerProto, 'handle');
  }

  if (routerProto && routerProto.use) {
    shimmer.unwrap(routerProto, 'use');
  }
  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}