#Requires -Version 3.0
#Requires -Module dnsserver
#This File is in Unicode format.  Do not edit in an ASCII editor.

#region help text

<#
.SYNOPSIS
	Provides DNS server information in the customized format.
.DESCRIPTION
	
	To run the script from a workstation, RSAT is required.
	
	Remote Server Administration Tools for Windows 7 with Service Pack 1 (SP1)
		http://www.microsoft.com/en-us/download/details.aspx?id=7887
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520

.PARAMETER ComputerName
	Specifies a computer to use to run the script against.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	Default is localhost.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  
#>
#endregion

#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None") ]

Param(
	
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost"
	)
#endregion

#region initialize variables
$SaveEAPreference = $ErrorActionPreference
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DNS config data ***"
$scriptEndIdentifier = "*** End - DNS config data ***"
$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region script setup function
Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		If(Test-Connection -ComputerName $CName -quiet)
		{
			Write-Verbose "$(Get-Date): Server $CName is online."
		}
		Else
		{
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $CName is offline.`nScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
	}

	#if computer name is an IP address, get host name from DNS
	
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip).AddressList.IPAddressToString
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
		}
		Else
		{
			Write-Warning "Unable to resolve $CName to a hostname"
		}
	}
	Else
	{
		#computer is online but for some reason $ComputerName cannot be converted to a System.Net.IpAddress
	}

	$Results = Get-DNSServer -ComputerName $CName -EA 0 3>$Null
	If($Null -ne $Results)
	{
		#the computer is a dns server
		$Script:DNSServerData = $Results
		Return $CName
	}
	ElseIf($Null -eq $Results)
	{
		#the computer is not a dns server
		$ErrorActionPreference = $SaveEAPreference
		Write-Error "`n`n`t`tComputer $CName is not a DNS Server.`n`n`t`tRerun the script using -ComputerName with a valid DNS server name.`n`n`t`tScript cannot continue.`n`n"
		Exit
	}
	Return $CName
}

Function ProcessScriptStart
{
	$ComputerName = TestComputerName $ComputerName
}
#endregion

#region Process Trust Points
Function ProcessTrustPoints
{
	$TrustPoints = Get-DNSServerTrustPoint -ComputerName $ComputerName -EA 0
	
	If($? -and $Null -ne $TrustPoints)
	{
		ForEach($Trust in $TrustPoints)
		{
			$Anchors = Get-DnsServerTrustAnchor -name $Trust.TrustPointName -ComputerName $ComputerName -EA 0
			
			If($? -and $Null -ne $Anchors)
			{
				ForEach($Anchor in $Anchors)
				{
					OutputTrustPoint $Trust $Anchor
				}
			}
		}
	}
	ElseIf($? -and $Null -ne $TrustPoints)
	{
		Write-Host "There is no Trust Zones data"
	}
	Else
	{
		Write-Host  "Trust Zones data could not be retrieved"
	}
}

Function OutputTrustPoint
{
	Param([object] $Trust, [object] $Anchor)
	If($Anchor.TrustAnchorData.ZoneKey)
	{
		$ZoneKey = "Yes"
		Switch ($Anchor.TrustAnchorData.KeyProtocol)
		{
			"DnsSec" {$KeyProtocol = "DNSSEC"}
			Default {$KeyProtocol = "Unknown: Zone Key Protocol = $($Anchor.TrustAnchorData.KeyProtocol)"}
		}
	}
	Else
	{
		$ZoneKey = "No"
		$KeyProtocol = "N/A"
	}
	$tmpType = ""
	Switch ($Anchor.TrustAnchorType)
	{
		"A"				{$tmpType = "HOST (A)"; break}
		"AAAA"			{$tmpType = "IPv6 HOST (AAAA)"; break}
		"AFSDB"			{$tmpType = "AFS Database (AFSDB)"; break}
		"ATMA"			{$tmpType = "ATM Address (ATMA)"; break}
		"CNAME"			{$tmpType = "Alias (CNAME)"; break}
		"DHCID"			{$tmpType = "DHCID"; break}
		"DNAME"			{$tmpType = "Domain Alias (DNAME)"; break}
		"DNSKEY"		{$tmpType = "DNS KEY (DNSKEY)"; break}
		"DS"			{$tmpType = "Delegation Signer (DS)"; break}
		"HINFO"			{$tmpType = "Host Information (HINFO)"; break}
		"ISDN"			{$tmpType = "ISDN"; break}
		"KEY"			{$tmpType = "Public Key (KEY)"; break}
		"MB"			{$tmpType = "Mailbox (MB)"; break}
		"MG"			{$tmpType = "Mail Group (MG)"; break}
		"MINFO"			{$tmpType = "Mailbox Information (MINFO)"; break}
		"MR"			{$tmpType = "Renamed Mailbox (MR)"; break}
		"MX"			{$tmpType = "Mail Exchanger (MX)"; break}
		"NAPTR"			{$tmpType = "Naming Authority Pointer (NAPTR)"; break}
		"NS"			{$tmpType = "Name Server (NS)"; break}
		"NSEC"			{$tmpType = "Next Secure (NSEC)"; break}
		"NSEC3"			{$tmpType = "Next Secure 3 (NSEC3)"; break}
		"NSEC3PARAM"	{$tmpType = "Next Secure 3 Parameters (NSEC3PARAM)"; break}
		"NXT"			{$tmpType = "Next Domain (NXT)"; break}
		"PTR"			{$tmpType = "Pointer (PTR)"; break}
		"RP"			{$tmpType = "Responsible Person (RP)"; break}
		"RRSIG"			{$tmpType = "RR Signature (RRSIG)"; break}
		"RT"			{$tmpType = "Route Through (RT)"; break}
		"SIG"			{$tmpType = "Signature (SIG)"; break}
		"SOA"			{$tmpType = "Start of Authority (SOA)"; break}
		"SRV"			{$tmpType = "Service Location (SRV)"; break}
		"TXT"			{$tmpType = "Text (TXT)"; break}
		"WINS"			{$tmpType = "WINS Lookup"; break}
		"WINSR"			{$tmpType = "WINS Reverse Lookup (WINS-R_"; break}
		"WKS"			{$tmpType = "Well Known Services (WKS)"; break}
		"X25"			{$tmpType = "X.25"; break}
		Default 		{$tmpType = "Unable to determine Record Type: $($Anchor.TrustAnchorType)"; break}
	}
	If($Anchor.TrustAnchorType -eq "DS")
	{
		$DigestType = ""
		Switch ($Detail.RecordData.DigestType)
		{
			"Sha1"		{$DigestType = "SHA-1"; break}
			"Sha256"	{$DigestType = "SHA-256"; break}
			"Sha384"	{$DigestType = "SHA-384"; break}
			Default		{$DigestType = "Unknown DigestType: $($Anchor.TrustAnchorType.DigestType)"; break}
		}
	}

	If($Anchor.TrustAnchorData.SecureEntryPoint)
	{
		$SEP = "Selected"
		Switch ($Anchor.TrustAnchorData.CryptoAlgorithm)
		{	
			"RsaSha1"		{$SEPAlgorithm = "RSA/SHA-1"; break}
			"RsaSha1NSec3"	{$SEPAlgorithm = "RSA/SHA-1 (NSEC3)"; break}
			"RsaSha256"		{$SEPAlgorithm = "RSA/SHA-256"; break}
			"RsaSha512"		{$SEPAlgorithm = "RSA/SHA-512"; break}
			Default 		{$SEPAlgorithm = "Unknown: Algorithm = $($Anchor.TrustAnchorData.CryptoAlgorithm)"; break}
		}
	}
	Else
	{
		$SEP = "Not Selected"
		$SEPAlgorithm = "N/A"
	}
	
	If($Trust.TrustPointName -eq ".")
	{
		$TrustPointName =  "$($Trust.TrustPointName)"
	}
	Else
	{
		$TrustPointName = "$($Trust.TrustPointName)"+"TrustAnchors."
	}
	If($Trust.TrustPointState -eq "Active")
	{
		$TrustStatus = "Valid"
	}
	Else
	{
		$TrustStatus = "Invalid"
	}

	Write-Host "server name :  $($env:ComputerName) $($delimiter) Name : (Same as parent folder) $($delimiter) Status : $($TrustStatus) $($delimiter) Trust Point type : $($($tmpType)) $($delimiter) Valid From: $($Trust.LastActiveRefreshTime) $($delimiter) Valid From: $($Anchor.EnteredStateTime) $($delimiter) Fully qualified domain name (FQDN) : $($TrustPointName) $($delimiter) Key Tag : $($Anchor.TrustAnchorData.KeyTag) $($delimiter) Zone Key : $($ZoneKey)  $($delimiter) Protocol : $($KeyProtocol) $($delimiter) Secure Entry Point : $($SEP) $($delimiter) Algorithm : $($SEPAlgorithm) $($delimiter) Public Key : $($Anchor.TrustAnchorData.Base64Data) $($delimiter) DigestType : $($DigestType)  "
}
#endregion

#region script core
#Script begins
Write-Host $scriptStartIdentifier 
ProcessScriptStart
try
{
	ProcessTrustPoints
}
catch 
{
	$global:isErrorOccured = $True
	Write-Verbose "Catch Block is Executed :OutputTrustPoint Function"
}
If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion