# This Script will give all details of XenDesktop
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$Global:ErrorActionPreference="SilentlyContinue"
Add-pssnapin "Citrix.Common.Commands"
Add-pssnapin "Citrix.Host.Admin.V1"
Add-pssnapin "Citrix.Broker.Admin.V1"
Add-pssnapin "Citrix.ADIdentity.Admin.V1"
Add-pssnapin "Citrix.Configuration.Admin.V1"
Add-pssnapin "Citrix.MachineCreation.Admin.V1"
Add-pssnapin "Citrix.MachineIdentity.Admin.V1"

$FQDN = [System.Net.Dns]::GetHostByName((hostname)).HostName
$SAMName = $ENV:USERDOMAIN + "\" + $ENV:COMPUTERNAME

Write-Host "Site"
Write-Host "----"
#DBUnconfigured = The broker does not have a database connection configured.
#DBRejectedConnection = The database rejected the logon from the Broker Service. This may be caused by bad credentials, or the database
#not being installed.
#InvalidDBConfigured = The database schema is missing (possibly just the stored procedures in it).
#DBNewerVersionThanService = The broker is too old to use the database. A newer version is required.
#DBOlderVersionThanService = The database is too old for the Broker Service. Upgrade the database.
#DBVersionChangeInProgress = A database schema upgrade is in progress.
#OK = The broker is connected to a database that is valid, and the service is running.
#PendingFailure = Connectivity between the Broker Service and the database has been lost. This may be a transitory network error,
#but may indicate a loss of connectivity that requires administrator intervention.
#Failed = Connectivity between the broker and the database has been lost for an extended period of time, or has failed du
#e to a configuration problem. The broker service cannot operate while its connection to the database is unavail
#able.
#Unknown = The Service's status cannot be determined.

$Site = Get-BrokerSite
$BrokerStatus = Get-BrokerServiceStatus
$HypStatus = Get-HypServiceStatus
$ADIdentityStatus = Get-AcctServiceStatus
$ConfigStatus = Get-ConfigServiceStatus
$MachineCreationStatus = Get-ProvServiceStatus
$MachineIdentityStatus = Get-PvsVmServiceStatus
if (($Site -ne $null) -and ($BrokerStatus -ne $null))
{
	if ($HypStatus -eq $null)
	{
		$HypStatus = "-"
	}
	if ($ADIdentityStatus -eq $null)
	{
		$ADIdentityStatus = "-"
	}
	if ($ConfigStatus -eq $null)
	{
		$ConfigStatus = "-"
	}
	if ($MachineCreationStatus -eq $null)
	{
		$MachineCreationStatus = "-"
	}
	if ($MachineIdentityStatus -eq $null)
	{
		$MachineIdentityStatus = "-"
	}
	Write-Host ($Site.Name,$Site.LicenseServerName,$Site.LicenseServerPort,$BrokerStatus,$HypStatus,$ADIdentityStatus,$ConfigStatus,$MachineCreationStatus,$MachineIdentityStatus) -Separator ";"
}


Write-Host "Hypervisor Details"
Write-Host "------------------"
# HyperConn.State
# Unavailable - The broker is unable to contact the hypervisor.
# InMaintenanceMode - The hosting server is in maintenance mode.
# On - The broker is in contact with the hypervisor.

$HyperConns = Get-BrokerHypervisorConnection -State On -PreferredController $SAMName
if($HyperConns -ne $null)
{
	foreach($HyperConn in $HyperConns)
	{
		$b = "XDHyp:\Connections\"
		$a = $b + $HyperConn.Name
		$HypervisorIP = "-"
		if ((Get-HypXenServerAddress -Literalpath $a).length -gt 0)
		{
			$HypervisorIP = Get-HypXenServerAddress -Literalpath $a
		}
		Write-Host ($HyperConn.Name,$HyperConn.State,$HypervisorIP) -Separator ";"
	}
}


Write-Host "Controllers"
Write-Host "-----------"
# Controller.State
# Valid values are: Failed, Off, On, and Active

$Controllers = Get-BrokerController
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		Write-Host($Controller.DNSName,$Controller.State,$Controller.DesktopsRegistered,
		$Controller.LastActivityTime,$Controller.LastStartTime,$Controller.ActiveSiteServices) -Separator ";"
	}
}


Write-Host "Catalogs"
Write-Host "--------"
#AllocationType = Permanent or Random
#CatalogKind = Thin cloned and Single image,PowerManaged,Unmanaged and PVS

$Catalogs = Get-BrokerCatalog -MaxRecordCount 2147483647
if($Catalogs -ne $null)
{
	foreach($Catalog in $Catalogs)
	{
		$MachinesArePhysical = "-"
		$PvsAddress = "-"
		$PvsDomain = "-"
		if ($Catalog.MachinesArePhysical.length -gt 0)
		{
			$MachinesArePhysical = $Catalog.MachinesArePhysical
		}
		if ($Catalog.PvsAddress.length -gt 0)
		{
			$PvsAddress = $Catalog.PvsAddress
		}
		if ($Catalog.PvsDomain.length -gt 0)
		{
			$PvsDomain = $Catalog.PvsDomain
		}
		Write-Host ($Catalog.Name,$Catalog.AllocationType,$Catalog.AssignedCount,$Catalog.AvailableAssignedCount,
		$Catalog.AvailableCount,$Catalog.AvailableUnassignedCount,$Catalog.UnassignedCount,$Catalog.UsedCount,$Catalog.CatalogKind,
		$MachinesArePhysical,$PvsAddress,$PvsDomain ) -Separator ";"
	}
}


Write-Host "Broker Machines"
Write-Host "---------------"
Write-Host " "
 

Write-Host "Desktop Groups"
Write-Host "--------------"

$DesktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 2147483647
if($DesktopGroups -ne $null)
{
	foreach($DesktopGroup in $DesktopGroups)
	{
		Write-Host ($DesktopGroup.Name,$DesktopGroup.Enabled,$DesktopGroup.InMaintenanceMode,$DesktopGroup.TotalDesktops,
		$DesktopGroup.DesktopsAvailable,$DesktopGroup.DesktopsDisconnected,$DesktopGroup.DesktopsInUse,
		$DesktopGroup.DesktopsNeverRegistered,$DesktopGroup.DesktopsUnregistered) -Separator ";"
	}
}


Write-Host "Virtual Desktops"
Write-Host "----------------"
# RegistrationState"
# Valid values are Registered, Unregistered, and AgentError.
# "SummaryState"
# Valid values are Off, Unregistered, Available, Disconnected, and InUse

$BrokerDesktops = Get-BrokerDesktop -MaxRecordCount 2147483647 -ControllerDNSName $FQDN
if ($BrokerDesktops -ne $null)
{
	foreach($BrokerDesktop in $BrokerDesktops)
	{	
		$IPAddress = "-"
		$OSType = "-"
		$HostedMachineName = "-"
		$HostingServerName = "-"
		$HypervisorConnectionName = "-"
		$SessionUserName = "-"
		$ControllerDNSName = "-"
		$DesktopConditions = "-"
		$LastConnectionFailure = "-"
		$LastConnectionTime = "-"
		$LastConnectionUser = "-"
		$AssociatedUserName = "-"
		$ImageOutOfDate = "-"
		if ( $BrokerDesktop.IPAddress.length -gt 0)
		{
			$IPAddress = $BrokerDesktop.IPAddress
		}
		if ($BrokerDesktop.OSType.length -gt 0)
		{
			$OSType = $BrokerDesktop.OSType
		}
		if ($BrokerDesktop.HostedMachineName.length -gt 0)
		{
			$HostedMachineName = $BrokerDesktop.HostedMachineName
		}
		if ($BrokerDesktop.HostingServerName.length -gt 0)
		{
			$HostingServerName = $BrokerDesktop.HostingServerName
		}
		if ($BrokerDesktop.HypervisorConnectionName.length -gt 0)
		{
			$HypervisorConnectionName = $BrokerDesktop.HypervisorConnectionName
		}
		if ($BrokerDesktop.SessionUserName.length -gt 0)
		{
			$SessionUserName = $BrokerDesktop.SessionUserName
		}
		if ($BrokerDesktop.ControllerDNSName.length -gt 0)
		{
			$ControllerDNSName = $BrokerDesktop.ControllerDNSName
		}
		if ($BrokerDesktop.DesktopConditions.length -gt 0)
		{
			$DesktopConditions = $BrokerDesktop.DesktopConditions
		}
		if ($BrokerDesktop.LastConnectionFailure -ne $null)
		{
			$LastConnectionFailure = $BrokerDesktop.LastConnectionFailure
		}
		if ($BrokerDesktop.LastConnectionTime -ne $null)
		{
			$LastConnectionTime = $BrokerDesktop.LastConnectionTime
		}
		if ($BrokerDesktop.LastConnectionUser -gt 0)
		{
			$LastConnectionUser = $BrokerDesktop.LastConnectionUser
		}
		if ($BrokerDesktop.AssociatedUserName -ne $null -and $BrokerDesktop.AssociatedUserName.length -gt 0)
		{
			$AssociatedUsers = $BrokerDesktop.AssociatedUserName
			$AssociatedUser = ""
			$count = 0
			foreach ($key in $AssociatedUsers)
			{
				if ($count -eq 0)
				{
					$AssociatedUser = $key
				}
				else
				{
					$AssociatedUser = $AssociatedUser + "," + $key
				}
				$count = $count + 1
			}
			if ($AssociatedUser -eq $null -or $AssociatedUser.length -eq 0)
			{
				$AssociatedUserName = "-"
			}
			else
			{
				$AssociatedUserName = $AssociatedUser
			}
		}
		if ($BrokerDesktop.ImageOutOfDate -ne $null)
		{
			$ImageOutOfDate = $BrokerDesktop.ImageOutOfDate
		}
		Write-Host ($BrokerDesktop.MachineName,$BrokerDesktop.DNSName,$IPAddress,$OSType,$BrokerDesktop.PowerState,
		$BrokerDesktop.RegistrationState,$BrokerDesktop.SummaryState,$BrokerDesktop.DesktopGroupName,$BrokerDesktop.DesktopKind,
		$HostedMachineName,$HostingServerName,$HypervisorConnectionName,$BrokerDesktop.InMaintenanceMode,$SessionUserName,
		$BrokerDesktop.CatalogName,$ControllerDNSName,$LastConnectionFailure,$ImageOutOfDate,$LastConnectionTime,$LastConnectionUser,$AssociatedUserName,$DesktopConditions) -Separator ";"
	}
}

Write-Host "Sessions"
Write-Host "--------"
# SessionState
# Valid values are Other, PreparingSession, Connected, Active, Disconnected, Reconnecting, NonBrokeredSession, and Unknown.

if ($BrokerDesktops -ne $null)
{
	foreach($BrokerDesktop in $BrokerDesktops)
	{
		if ($BrokerDesktop -ne $null)
		{
			$IPAddress = "-"
			$OSType = "-"
			
			if ( $BrokerDesktop.IPAddress.length -gt 0)
			{
				$IPAddress = $BrokerDesktop.IPAddress
			}
			if ($BrokerDesktop.OSType.length -gt 0)
			{
				$OSType = $BrokerDesktop.OSType
			}
			$BrokerSessions = Get-BrokerSession -MaxRecordCount 2147483647 -DesktopUid $BrokerDesktop.Uid
			if ($BrokerSessions -ne $null)
			{
				foreach($BrokerSession in $BrokerSessions)
				{
					$BrokeringUserName = "-"
					$BrokeringTime = "-"
					$ConnectedViaHostName = "-"
					$ClientAddress ="-";
					$ClientName = "-"
					$ClientVersion = "-"
					$Protocol = "-"
					$ControllerDNSName = "-"
					$BrokeringUserName = $BrokerSession.UserName
					if ($BrokeringUserName -eq $null -or $BrokeringUserName.length -eq 0)
					{
						$BrokeringUserName = $BrokerSession.BrokeringUserName
						if ($BrokeringUserName -eq $null -or $BrokeringUserName.length -eq 0)
						{
							$BrokeringUserName = "-"
						}
					}
					if ($BrokerSession.ClientAddress.length -gt 0)
					{
						$ClientAddress = $BrokerSession.ClientAddress
					}
					if ($BrokerSession.ClientName.length -gt 0)
					{
						$ClientName = $BrokerSession.ClientName
					}
					if ($BrokerSession.ClientVersion.length -gt 0)
					{
						$ClientVersion = $BrokerSession.ClientVersion
					}
					if ($BrokerSession.BrokeringTime -ne $null)
					{
						$BrokeringTime = $BrokerSession.BrokeringTime
					}
					if ($BrokerSession.ConnectedViaHostName -ne $null)
					{
						$ConnectedViaHostName = $BrokerSession.ConnectedViaHostName
					}
					if ($BrokerSession.Protocol.length -gt 0)
					{
						$Protocol = $BrokerSession.Protocol
					}
					if ($BrokerDesktop.ControllerDNSName.length -gt 0)
					{
						$ControllerDNSName = $BrokerDesktop.ControllerDNSName
					}
					if ($BrokerDesktop.LicenseId.length -gt 0)
					{
						$LicenseId = $BrokerDesktop.LicenseId
					}
					Write-Host ($BrokerDesktop.MachineName,$BrokerDesktop.DNSName,$IPAddress,$OSType,$BrokeringUserName,
					$ClientAddress,$ClientName,$ClientVersion,$ConnectedViaHostName,
					$Protocol,$BrokerSession.SessionState,$BrokerSession.StartTime,$BrokeringTime,$LicenseId,$ControllerDNSName) -Separator ";"
				}
			}
		}
	}
}


Write-Host "Application Instances"
Write-Host "---------------------"
$AppInstances = Get-BrokerApplicationInstance -MaxRecordCount 2147483647
if ($AppInstances -ne $null)
{
	foreach($AppInstance in $AppInstances)
	{
		$BrokeringTime = "-"
		$BrokerDesktop = Get-BrokerDesktop -SessionUid $AppInstance.SessionUid
		$session = Get-BrokerSession -Uid $AppInstance.SessionUid
		if ($session.BrokeringTime -ne $null)
		{
			$BrokeringTime = $session.BrokeringTime
		}

		$application = Get-BrokerApplication -Uid $AppInstance.ApplicationUid
		$folder = Get-BrokerApplicationFolder -Uid $application.FolderUid
		if (($session -ne $null) -and ($application -ne $null))
		{
			if($folder -ne $null)
			{
				$folderName = $folder.Name
				if ($folderName -ne $null -and $folerName.length -gt 0)
				{
					$name = $folderName + "\" + $application.DisplayName
				}
				else
				{
					$name = $application.DisplayName
				}
			}
			else
			{
				$name = $application.DisplayName
			}
			Write-Host($BrokerDesktop.MachineName,$name,$application.CommandLineExecutable,$session.BrokeringUserName,$session.ClientAddress,$BrokerSession.Protocol,$BrokerSession.SessionState,$BrokerSession.StartTime,$BrokeringTime,$BrokerDesktop.CatalogName,$BrokerDesktop.LicenseId) -Separator ";"
		}
	}
}


Write-Host "Unavailable Virtual Desktops"
Write-Host "----------------------------"
if($DesktopGroups -ne $null)
{
	foreach($DesktopGroup in $DesktopGroups)
	{
		$dirtyVMs = Get-BrokerDesktop -DesktopGroupUid $DesktopGroup.Uid -PowerActionPending $false -PowerState On -SummaryState Available -WillShutdownAfterUse $true -MaxRecordCount 2147483647
		if($dirtyVMs ne $null)
		{
			foreach($dirtyVM in $dirtyVMs)
			{
				$HostedMachineName = "-"
				$IPAddress = "-"
				$OSType = "-"
				$HostingServerName = "-"
				$HypervisorConnectionName = "-"
				$ControllerDNSName = "-"
				if ( $dirtyVM.IPAddress.length -gt 0)
				{
					$IPAddress = $dirtyVM.IPAddress
				}
				if ($dirtyVM.OSType.length -gt 0)
				{
					$OSType = $dirtyVM.OSType
				}
				if ($dirtyVM.HostedMachineName.length -gt 0)
				{
					$HostedMachineName = $dirtyVM.HostedMachineName
				}
				if ($dirtyVM.HostingServerName.length -gt 0)
				{
					$HostingServerName = $dirtyVM.HostingServerName
				}
				if ($dirtyVM.HypervisorConnectionName.length -gt 0)
				{
					$HypervisorConnectionName = $dirtyVM.HypervisorConnectionName
				}
				if ($dirtyVM.ControllerDNSName.length -gt 0)
				{
					$ControllerDNSName = $dirtyVM.ControllerDNSName
				}
				Write-Host($dirtyVM.MachineName,$dirtyVM.DNSName,$IPAddress,$OSType,$dirtyVM.DesktopGroupName,$dirtyVM.DesktopKind,$dirtyVM.CatalogName,$HostedMachineName,$HostingServerName,$HypervisorConnectionName,$ControllerDNSName) -Separator ";"
			}
		}
	}
}

Remove-pssnapin "Citrix.Common.Commands"
Remove-pssnapin "Citrix.Host.Admin.V1"
Remove-pssnapin "Citrix.Broker.Admin.V1"
Remove-pssnapin "Citrix.ADIdentity.Admin.V1"
Remove-pssnapin "Citrix.Configuration.Admin.V1"
Remove-pssnapin "Citrix.MachineCreation.Admin.V1"
Remove-pssnapin "Citrix.MachineIdentity.Admin.V1"

