﻿<#
    .SYNOPSIS
        Test StoreFront or NetScaler Gateway (integrated with StoreFront) and XenApp by launching one or more Published Applications.
    .DESCRIPTION
        This script launches an a Published Applications through StoreFront or NetScaler Gateway (integrated with StoreFront).

        It attempts to closely resemble what an actual user would do by:
        -Opening Internet Explorer.
        -Navigating directly to the Receiver for Web site or NetScaler Gateway portal.
        -Entering Username and Password.
        -Logging in.
        -Clicking on the Application(s).
        -Logging off the StoreFront site.
        
        You can use thie Script to verify your most important Applications load.

        You can also use this Script to verify that each of your servers is repsonding correctly. 
	    To do this you would need to Publish an Application from each server. 
	    For example, you could publish Task Manager (taskmgr.exe), from each server, and name it something like "Server 1 Task Manager", "Server 2 Task Manager", etc, etc.

        Requirements:
        -Citrix Receiver installed in the default location.
        -Must be launched from an Elevated Administrator console of PowerShell x86. The 64 Bit Powershell will not work since the script is using Citrix Reciever, which is 32 bit, to verify a lunch and logoff the session.
        -No other sessions should be connected/running before running the script.
        -SiteURL should be part of the Intranet Zone (or Internet Zone at Medium-Low security) in order to be able to download AND launch the ICA file. This can be done through a GPO.
        -StoreFront 2.0 or higher or NetScaler Gateway, version 9.3 or higher.
        -Changes in web.config under C:\inetpub\wwwroot\Citrix\<storename>Web\: autoLaunchDesktop to false, pluginAssistant to false.
        -Currently works for desktops or already subscribed apps only. 
		-You can auto subscribe users to apps by setting "KEYWORDS:Auto" in the published app's description or make sure the apps have been subscribed to manually.
	    -This script does not support AppController at this time.
	    -In order for the script to interact with the ICO, it requires the following to be set in the registry:
		-x64: HKEY_LOCAL_MACHINE\SOFTWARE\Wow6432Node\Citrix\ICA Client\CCM
		-x86: HKEY_LOCAL_MACHINE\SOFTWARE\Citrix\ICA Client\CCM
		-AllowLiveMonitoring REG_DWORD 1
		-AllowSimulationAPI REG_DWORD 1

    .PARAMETER SiteURL
        The complete URL of the StoreFront Receiver for Web site or NetScaler Gateway portal.
    .PARAMETER UserName
        The name of the user which is used to log on. Acceptable forms are down-level logon name or user principal name.
    .PARAMETER Password
        The password of the user which is used to log on.
    .PARAMETER SleepBeforeLogoff
        The time in seconds to sleep after clicking the resource and before logging off. Default is 5.
    .PARAMETER NumberOfRetries
        The number of retries when retrieving an element. Default is 30.
    .PARAMETER LogFilePath
        Directory path to where the log file will be saved. Default is SystemDrive\Temp.
    .PARAMETER LogFileName
        File name for the log file. By default, the script creates a log file named "SFLauncher.log".
    .PARAMETER TwoFactorAuth
        The token or password used for two-factor authentication. This is used in the NetScaler Gateway portal.
    .PARAMETER ResourceName
        The display name(s) of the Published Application(s) to be launched. Just add the applications at the end of the command line. See the example for more info.
    .PARAMETER TimeToWaitForApp
        The amount of time, in seconds, to wait for the Application to launch
    .PARAMETER TimeToWaitForLogoff
        The amount of time, in seconds, to wait before issuing the Logoff Command        
        
    .EXAMPLE
        ICALogonSimulation.ps1 -SiteURL https://xxxx.eginnovations.com -UserName xxxx\xxxx1 -Password xxxx -ResourceName Paint -SID 1 -LogFilePath f:\work -LogFileName SFLauncher.log -TimeToWaitForApp 90 -TimeToWaitForLogoff 30 -TwoFactorAuth None -Prompt False -UseICA 1 -IsDisclaimer False

        Description
        -----------
        Launches a sPublished Application using the parameters provided. 
    .LINK
        UserName format used in StoreFront.
        http://msdn.microsoft.com/en-us/library/windows/desktop/aa380525(v=vs.85).aspx#down_level_logon_name
    .LINK
        Change to autoLaunchDesktop.
        http://support.citrix.com/proddocs/topic/dws-storefront-20/dws-configure-wr-view.html
    .LINK
        Change to logoffAction.
        http://support.citrix.com/proddocs/topic/dws-storefront-20/dws-configure-wr-workspace.html
    .LINK
        Automating the launch of HDX sessions through StoreFront (and NetScaler Gateway integrated with StoreFront)
        http://blogs.citrix.com/2014/06/12/scripting-update-automating-the-launch-of-hdx-sessions-through-storefront-and-netscaler-gateway-integrated-with-storefront/
    .NOTES
        Created using SFLauncher.ps1 originally written by Citrix Systems, Inc. 
#>

Param (
    [Parameter(Mandatory=$true,Position=0)] [string]$SiteURL,
    [Parameter(Mandatory=$true,Position=1)] [string]$UserName,
    [Parameter(Mandatory=$true,Position=2)] [string]$Password,
    [Parameter(Mandatory=$true,Position=3)] $ResourceName,
    [Parameter(Mandatory=$true,Position=4)] [int]$SID,
    [Parameter(Mandatory=$false)] [int]$UseICA = 1,
    [Parameter(Mandatory=$false)] [int]$SleepBeforeLogoff = 2,
    [Parameter(Mandatory=$false)] [int]$NumberOfRetries = 60,
    [Parameter(Mandatory=$false)] [string]$LogFilePath = "$($env:SystemDrive)\Temp\",
    [Parameter(Mandatory=$false)] [string]$LogFileName = "SFLauncher.log",
    [Parameter(Mandatory=$false)] [string]$TwoFactorAuth,
    [Parameter(Mandatory=$false)] $TimeToWaitForApp = "45",
    [Parameter(Mandatory=$false)] $TimeToWaitForLogoff = "10",
    [Parameter(Mandatory=$false)] $TimeoutForOther = "5",
    [Parameter(Mandatory=$false)] [string]$TabName = $null,
    [Parameter(Mandatory=$false)] [string]$Prompt = "False",
    [Parameter(Mandatory=$false)] [string]$IsDisclaimer = "False",
    [Parameter(Mandatory=$false)] [int]$DisclaimerTimeout = 30,
    [Parameter(Mandatory=$false)] [string]$WindowTitle = "None"
)

#region Variables
#Set-StrictMode -Version 2

# Netscaler
Set-Variable -Name NGLoginButtonId -Value "Log_On" -Option Constant -Scope Script
Set-Variable -Name NGLoginButtonId1 -Value "btnLogin" -Option Constant -Scope Script
Set-Variable -Name NGUserNameTextBoxName -Value "Enter user name" -Option Constant -Scope Script
Set-Variable -Name NGUserNameTextBoxName1 -Value "login" -Option Constant -Scope Script
Set-Variable -Name NGUserNameTextBoxName2 -Value "user" -Option Constant -Scope Script
Set-Variable -Name NGPasswordTextBoxName -Value "passwd" -Option Constant -Scope Script
Set-Variable -Name NGPasswordTextBoxName1 -Value "password" -Option Constant -Scope Script
Set-Variable -Name NGTwoFactorTextBoxName -Value "passwd1" -Option Constant -Scope Script
Set-Variable -Name Eulacheck -Value "eula_check" -Option Constant -Scope Script

# Storefront
Set-Variable -Name SFLoginButtonId -Value "loginBtn" -Option Constant -Scope Script
Set-Variable -Name SFUsernameTextBoxId -Value "username" -Option Constant -Scope Script
Set-Variable -Name SFPasswordTextBoxId -Value "password" -Option Constant -Scope Script

# Log Off ID
Set-Variable -Name SFLogOffLinkId -Value "logoffLink" -Option Constant -Scope Script
Set-Variable -Name SFLogOffLinkId1 -Value "userdetails-logoff" -Option Constant -Scope Script
Set-Variable -Name SFLogOffLinkId2 -Value "wscLink" -Option Constant -Scope Script
Set-Variable -Name SFLogOffLinkId3 -Value "id_LogOff" -Option Constant -Scope Script
Set-Variable -Name SFLogOffLinkId4 -Value "logoutAreaLogoutLink" -Option Constant -Scope Script
Set-Variable -Name SFLogOffLinkId5 -Value "menuLogOffBtn" -Option Constant -Scope Script

# Alerts
Set-Variable -Name AlertBoxId -Value "alertbox" -Option Constant -Scope Script
Set-Variable -Name AlertBoxIdv3 -Value "genericMessageBoxPopup" -Option Constant -Scope Script

# 'Continue' Element
Set-Variable -Name ContinueButton -Value $null -Scope Script
Set-Variable -Name ContinueButtonMsgPaneID -Value "OK" -Option Constant -Scope Script
Set-Variable -Name ContinueButtonMsgPaneID2 -Value "Accept" -Option Constant -Scope Script
Set-Variable -Name ContinueButtonMsgPaneID3 -Value "Log on" -Option Constant -Scope Script
Set-Variable -Name ContinueButtonClicked -Value $false -Scope Script

# ICO DLL locations
Set-Variable -Name x64DLLPath -Value "C:\Program Files (x86)\Citrix\ICA Client" -Option Constant -Scope Script
Set-Variable -Name x86DLLPath -Value "C:\Program Files\Citrix\ICA Client" -Option Constant -Scope Script

Set-Variable -Name ResName -Value "" -Scope Script
Set-Variable -Name ResTitle -Value "" -Scope Script
Set-Variable -Name AnchorTags -Scope Script
New-Variable -Name IsResource -Value $false -Scope Script
New-Variable -Name isNG -Value $true -Scope Script

Set-Variable -Name loginAborted -Value $False -Scope Script
Set-Variable -Name InvalidAccount -Value $False -Scope Script
Set-Variable -Name SwitchUserPassword -Value "Switch to user name and password"  -Option Constant -Scope Script

# Setting up log files
$Script:IETry = 0
[string]$ErrorStyle = "style=""background-color: #000000; color: #FF3300;"""
[string]$Script:LogFile=$($LogFilePath.TrimEnd('\') + "\$LogFileName")

# Reads the current directory path from the location of this file
[string]$Script:currentDir = Split-Path $MyInvocation.MyCommand.Path

#endregion Variables

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region Functions

function CheckforDLL
{
    if ((test-path -Path $x64DLLPath) -eq "TRUE")
    {
        Set-Variable -Name DLLPath -Value "$x64DLLPath\WfIcaLib.dll" -Option Constant -Scope Script 
        "Found DLL in $x64DLLPath" | logme -displaygreen
    }
    elseif ((test-path -Path $x86DLLPath) -eq "TRUE")
    {
        Set-Variable -Name DLLPath -Value "$x86DLLPath\WfIcaLib.dll" -Option Constant -Scope Script 
        "Found DLL in $x86DLLPath\WfIcaLib.dll" | logme -displaygreen
    }
    else
    {
        "Cannot locate DLL." | logme -error
        throw "WfIcaLib.dll cannot be found, is the Citrix Client installed in it's default location?"
    }
}

function Get-ICAClientVersion
{
    # ============================ #
    # Get Client Version Function  #
    # ============================ #
    $ErrorActionPreference = "SilentlyContinue" 
    $ica = New-Object -ComObject 'Citrix.ICAClient' 
    if($ica)
    {
       $version = [String]$ica.ClientVersion
       if ($version -ne $null)
       {
            $version = $version.Substring(0,$version.IndexOf("."))
            [int]$versionNo = 0
            try
            {
                 [bool]$result = [int]::TryParse($version, [ref]$versionNo)
            } catch { }
            return $versionNo
       }
       else { return 0 }
    } 
    else { return 0 } 
} 

function CheckforAlertBox
{
    $alertbox = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $AlertboxIdv3)
    if($alertbox -eq $null -or $alertbox.GetType() -eq [DBNull])
    {
        $alertbox = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $AlertboxId)
        if($alertbox -eq $null -or $alertbox.GetType() -eq [DBNull])
        {
            $Script:alertboxdetected = $false
        }
        else
        {
            #$alertboxlink=[System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $internetExplorer.Document, "a") | where-object {$_.className -eq "button defaultDialogButton" }   
            $alertboxlink=[System.__ComObject].InvokeMember("getElementsByClassName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $internetExplorer.Document, "button defaultDialogButton")
        }
    }
    else
    {
        #$alertboxlink=[System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $internetExplorer.Document, "a") | where-object {$_.className -eq "dialog button default" }
        $alertboxlink=[System.__ComObject].InvokeMember("getElementsByClassName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $internetExplorer.Document, "dialog button default")
    }
    if ($alertboxlink -ne $null)
    {
        $alertext = $alertbox.innerText.Trim()
        $alertext=$alertext.Trim([char]0x0020,[char]0x004F,[char]0x004B)
        $alertext=$alertext.trimend()                       
        $Script:alertboxdetected = $true
        
	    $alertext | LogMe -error
        
    }
    else
    {
        $Script:alertboxdetected = $false
    }            
            
    
}


function GetTimeDiff([datetime]$endtime,[datetime]$starttime)
{
    # ====================== #
    # Time Elapsed Function  #
    # ====================== #
    $runtime = $endtime - $starttime
    $retStr = [string]::format("{0} sec(s)", $runtime.TotalSeconds)
    $retStr
}

function GetElapsedTime([datetime]$starttime,$isDisclaimerWindowFound,$DisclaimerTimeout)
{
    # ====================== #
    # Time Elapsed Function  #
    # ====================== #
    $runtime = $(get-date) - $starttime
    if($isDisclaimerWindowFound -eq $True)
    {
        if($runtime.TotalSeconds -ge $DisclaimerTimeout)
        {
            $TotalRunTime = ($runtime.TotalSeconds - $DisclaimerTimeout)
            # We spent extra 5 seconds sleep
            if($TotalRunTime -gt 5)
            {
                $TotalRunTime = $TotalRunTime - 5
            }
            $retStr = [string]::format("{0} sec(s)",$TotalRunTime)
        }
    }
    else
    {      
        $retStr = [string]::format("{0} sec(s)",($runtime.TotalSeconds))
    }

    $retStr
}

function CreateRegEntry ($RegPath,$RegName,$PropType,$Val)
{
    # ===================== #
    # Create Registry Item  #
    # ===================== #
    try {New-ItemProperty -Path $RegPath -Name $RegName -PropertyType $PropType -Value $Val -ErrorAction Stop}
    catch {
    $Script:RegError += $_.Exception.Message
    $_.Exception.Message  | LogMe -error
    }
}

function ModifyRegEntry ($RegPath,$RegName,$Val)
{
    # ===================== #
    # Modify Registry Item  #
    # ===================== #
    try {Set-ItemProperty -Path $RegPath -Name $RegName -Value $Val -ErrorAction Stop}
    catch {
    $Script:RegError += $_.Exception.Message
    $_.Exception.Message  | LogMe -error
    }
}

function CreateRegKey ($RegPath)
{
    # ===================== #
    # Create Registry Item  #
    # ===================== #
    try {New-Item -Path $RegPath -ErrorAction Stop}
    catch {
    $Script:RegError += $_.Exception.Message
    $_.Exception.Message  | LogMe -error
    }
}

function CheckRegistry
{
    # Making sure Registry Entries exist
    # If not, try to create or modify
    # If they cannot be created or modified, logging an error 
    # and skipping ICA test
    "Checking Registry for Required Settings" | logme -displaynormal
    $Path = "HKLM:SOFTWARE\Citrix\ICA Client\CCM"
    $TestRegPath = Get-ItemProperty -Path $Path -ErrorAction SilentlyContinue
    
    if (!$TestRegPath) {CreateRegKey $Path}
    
    $AllowLiveMonitoringName = "AllowLiveMonitoring"
    $AllowSimulationAPIName = "AllowSimulationAPI"
    $PropertyType = "DWORD"
    $Value =  "1"
    $AllowLiveMonitoring = Get-ItemProperty -Path $Path -Name $AllowLiveMonitoringName -ErrorAction SilentlyContinue
    $AllowSimulationAPI = Get-ItemProperty -Path $Path -Name $AllowSimulationAPIName  -ErrorAction SilentlyContinue
    if (!$AllowLiveMonitoring) {
        "AllowLiveMonitoring does not exist, creating it" | LogMe -warning
        CreateRegEntry $Path $AllowLiveMonitoringName $PropertyType $Value
    } elseif ($AllowLiveMonitoring.AllowLiveMonitoring -ne "1") {
        "AllowLiveMonitoring Value does not equal 1, setting to 1" | LogMe -warning
        ModifyRegEntry $Path $AllowLiveMonitoringName $Value
    }
    if (!$AllowSimulationAPI) { 
        "AllowSimulationAPI does not exist, creating it" | LogMe -warning
        CreateRegEntry $Path $AllowSimulationAPIName $PropertyType $Value
    } elseif ($AllowSimulationAPI.AllowSimulationAPI -ne "1") {
        "AllowSimulationAPI value exists but does not equal 1, setting to 1" | LogMe -warning
        ModifyRegEntry $Path $AllowSimulationAPIName $Value
    }
    if ($Script:RegError -ne $Null) {
        "Errors were encountered when trying to add or modify registry entries required for this script to work" | LogMe -error
        "You will either need to run this script once as an Adminitrator or create the following registry entries manually:" | LogMe -error
        "HKEY_LOCAL_MACHINE\SOFTWARE\Wow6432Node\Citrix\ICA Client\CCM, AllowLiveMonitoring, DWORD, 1" | LogMe -error
        "HKEY_LOCAL_MACHINE\SOFTWARE\Wow6432Node\Citrix\ICA Client\CCM, AllowSimulationAPI, DWORD, 1" | LogMe -error
    }
}

function KillProcess
{
	# Killing left over processes from Receiver so the script will work the next time it is executed
	# This is really only needed when running as a scheduled task
	# If these processes are allowed to remain the Connection fails after the second try
    $ctxprocess = @()
    $ctxproclist = @("wfcrun32","concentr","redirector","wfica32","receiver","cdviewer","cpviewer","ctxcfrui","ssonsvr","webhelper","selfserviceplugin","cleanup","ConfigurationWizard")
    foreach ($proc in $ctxproclist)
    {
        Get-Process $proc -ErrorAction silentlycontinue | select-object id | % { $ctxprocess += $_.id }
    }
    if ($ctxprocess -ge 1)
    {
        "Citrix Receiver processes were detected. Killing the processes so the script will work, if they exist they will prevent the ICO from working."  | logme -displaynormal
        foreach ($id in $ctxprocess) 
        {
            stop-process -id $id -force -ErrorAction silentlycontinue -ErrorVariable +err
        }
    }
    if ($err.count -gt 0)
    {
        "Could not kill Citrix Reciever processes" | logme -error
        throw "Could not kill Citrix Reciever processes"
    }
}

Function LogMe()
{
    #===================================================================== #
    # Sends the results into a logfile as well as in the powershell window #
    #===================================================================== #
    Param( [parameter(Mandatory = $true, ValueFromPipeline = $true)] $logEntry,
	   [switch]$displaygreen,
	   [switch]$error,
	   [switch]$warning,
	   [switch]$displaynormal
     )
    
    $Status = $logEntry
    if($error)
    { 
        $script:ErrorMessage += "<p $ErrorStyle>$logEntry<p>"
        Write-Host "$logEntry" -Foregroundcolor Red; $logEntry = (Get-Date -Format "[MM'/'dd'/'yyy HH':'mm':'ss'.'fff]:").ToString() + "[ERROR] $logEntry" 
    }
	elseif($warning) { Write-Host "$logEntry" -Foregroundcolor Yellow; $logEntry = (Get-Date -Format "[MM'/'dd'/'yyy HH':'mm':'ss'.'fff]:").ToString() + "[WARNING] $logEntry"}
	elseif ($displaynormal) { Write-Host "$logEntry" -Foregroundcolor White; $logEntry = (Get-Date -Format "[MM'/'dd'/'yyy HH':'mm':'ss'.'fff]:").ToString() + "[INFO] $logEntry" }
	elseif($displaygreen) { Write-Host "$logEntry" -Foregroundcolor Green; $logEntry = (Get-Date -Format "[MM'/'dd'/'yyy HH':'mm':'ss'.'fff]:").ToString() + "[SUCCESS] $logEntry" }
    else { Write-Host "$logEntry"; $logEntry = "$logEntry" }
    if ($logEntry -ne $null) {$logEntry | Out-File -FilePath $Script:LogFile -Append}
}

function Get-UACEnabled
{
  $uacRegPath = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Policies\System"
  $uacRegValue = "EnableLUA"
  $uacEnabled = $false
  # http://msdn.microsoft.com/en-us/library/windows/desktop/ms724832(v=vs.85).aspx
  $osVersion = [Environment]::OSVersion.Version
  if ($osVersion -ge [Version]'6.0')
  {
    $uacRegSetting = Get-ItemProperty -Path $uacRegPath
    try
    {
      $uacValue = $uacRegSetting.EnableLUA
      if ($uacValue -eq 1)
      {
        $uacEnabled = $true
      }
    }
    catch
    {
      #regkey doesn't exist, so proceed with false
    }
  }
return $uacEnabled
}

function ConnectIExplorer() {
    param($HWND)

    $objShellApp = New-Object -ComObject Shell.Application 
    try {
      $EA = $ErrorActionPreference; $ErrorActionPreference = 'Stop'
      $objNewIE = $objShellApp.Windows() | ?{$_.HWND -eq $HWND}
      $objNewIE.Visible = $true
    } catch {
      #it may happen, that the Shell.Application does not find the window in a timely-manner, therefore quick-sleep and try again
      "Waiting for page to be loaded ..." | LogMe -displaynormal
      Start-Sleep -Milliseconds 500
      try {
        $objNewIE = $objShellApp.Windows() | ?{$_.HWND -eq $HWND}
        $objNewIE.Visible = $true
      } catch {
        "Could not retreive the -com Object InternetExplorer. Aborting." |  LogMe -error
      }     
    } finally { 
      $ErrorActionPreference = $EA
    }
    return $objNewIE
  } 

function Wait-ForPageReady
{
    
    $Script:IEReady = $true
	"Waiting for Internet Explorer to return Page Ready" | LogMe -displaynormal
    $try = 1
    do
	{
		"Try #$try`: Waiting for Internet Explorer to return Page Ready" | LogMe -displaynormal
		Start-Sleep 1
		$Script:IETry = $try
		$try++
    }
    until ($internetExplorer.ReadyState -eq 4 -or $try -gt $TimeToWaitForApp)
    if ($internetExplorer.ReadyState -ne 4)
	{
        "Internet Explorer did not repsond Page Ready within time allotted" | LogMe -error
        $Script:IEReady = $false
    }
}

function Open-InternetExplorer
{
    Param ([Parameter(Mandatory=$true)] [string]$SiteURL)
    "Creating Internet Explorer Component Object Model (COM)" | LogMe -displaynormal
    New-Variable -Name internetExplorer -Value (New-Object -ComObject "InternetExplorer.Application") -Scope Script 
    if ($Script:UACStatus -eq $true)
    {
        $Script:HWND = $internetExplorer.HWND
        "Navigating to '$SiteURL'" | Logme -displaynormal
        $internetExplorer.Navigate2($SiteURL)
        $internetExplorer = ConnectIExplorer -HWND $Script:HWND
    }
    else
    {
        "Navigating to '$SiteURL'" | Logme -displaynormal
        $internetExplorer.Navigate2($SiteURL)
    }

    if($internetExplorer -ne $null)
    {
        "Setting Internet Explorer visible" | LogMe -displaynormal
        $internetExplorer.visible = $true
        "Waiting until the page is ready" | Logme -displaynormal
        Wait-ForPageReady
        if($Script:IEReady -eq $true)
        {
            "Accessing Document Object Model (DOM)" | Logme -displaynormal
            New-Variable -Name document -Value $internetExplorer.Document -Scope Script
            try
            {
                $skipLogonLinks = @([System.__ComObject].InvokeMember("getElementsByClassName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "_ctxsattr_title_SkipToLogonTip"))
		        foreach($skipLogonLink in $skipLogonLinks)
		        {
			        if(($skipLogonLink -ne $NULL -and $skipLogonLink.GetType() -ne [DBNull]) -and $skipLogonLink.innerText.toLower().contains("log on"))
			        {
				        $skipLogonLink.click()
			        }
		        }
            }
            catch{}
        }
    }
}

function Test-LoginForm
{
    
	try 
    {
        <#$SwitchUserName = [System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "a") | where { $_.outerText -ne $null -and $_.outerText -eq $SwitchUserPassword }
        if($SwitchUserName -ne $null -and $SwitchUserName.GetType() -ne [DBNull])
        {
            $SwitchUserName.Click()
            Start-sleep -seconds 3
        }#>

        $SwitchLinks = @([System.__ComObject].InvokeMember("getElementsByClassName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "auth-choice-link authentication-link"))
        foreach($SwitchLink in $SwitchLinks)
        {
            if(($SwitchLink -ne $NULL -and $SwitchLink.GetType() -ne [DBNull]) -and $SwitchLink.outerText.toLower() -eq "switch to user name and password")
            {
                $SwitchLink.click()
                Start-sleep -seconds 3
                break
            }    
        }

	} catch {}

    "Detecting NetScaler Gateway or StoreFront login form" | Logme -displaynormal
    Set-Variable -Name loginButton -Value $Null -Scope Script
    $try = 1
    
    do
	{
        do
        {
            #"Trying to find the logon button with id $NGLoginButtonId" | LogMe -displaynormal
            $NGloginButton = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGLoginButtonId)
            if ( $NGloginButton -eq $null -or $NGloginButton.GetType() -eq [DBNull] )
		    {
                #"Trying to find the logon button with id $NGLoginButtonId1" | LogMe -displaynormal
                $NGloginButton = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGLoginButtonId1)    
                if ( $NGloginButton -eq $null -or $NGloginButton.GetType() -eq [DBNull] )
                {
                    #"Trying to find the logon button with id $SFLoginButtonId" | LogMe -displaynormal
                    $SFloginButton = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLoginButtonId)
                    if($SFloginButton -ne $null -and  $SFloginButton.GetType() -ne [DBNull])
                    {
                        break
                    }
                }
                elseif($NGloginButton -ne $null -and  $NGloginButton.GetType() -ne [DBNull])
                {
                    break
                }
            }
            elseif($NGloginButton -ne $null -and  $NGloginButton.GetType() -ne [DBNull])
            {
                break
            }
            "Try #$try`: Still detecting NetScaler Gateway or StoreFront login form..." | LogMe -displaynormal
            Start-Sleep -Seconds 1
            $try++
            
        }
        until($try -gt $TimeToWaitForApp)

        if(($NGloginButton -eq $null -or $NGloginButton.GetType() -eq [DBNull]) -and ($SFloginButton -eq $null -or  $SFloginButton.GetType() -eq [DBNull]))
        {
            break
        }
        $try = 1
        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId)
		if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
		{
            $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId1)
            if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
			{
                $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId2)
				if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
				{
                    $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId3)
                    if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
					{
                        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId4)
                        if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
						{
							$logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId5)
						}
                    }    
                }    
            }
        }

	    if ($Prompt -eq $true)
	    {
		    $ContinueButton = $null
		    try {
			    $ContinueButton = [System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "a") | where { $_.outerText -ne $null -and $_.outerText -eq $ContinueButtonMsgPaneID }		
		    } catch {}

		    if ( $ContinueButton -eq $null -or $ContinueButton.GetType() -eq [DBNull] )
		    {
			    try {
				    $ContinueButton = [System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "a") | where { $_.outerText -ne $null -and $_.outerText -eq $ContinueButtonMsgPaneID2 -or $_.outerText -eq $ContinueButtonMsgPaneID3 }		
			    } catch {}
		    }
		    $ContinueButton1 = $null
		    try {
			    $ContinueButton1 = [System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "input") | where { $_.name -ne $null -and $_.name -eq "yes" }            
		    } catch {}
	    }

        if ($NGloginButton -ne $null -and $NGloginButton.GetType() -ne [DBNull])
	    {
            "NETSCALER GATEWAY / WEBINTERFACE DETECTED"  | Logme -displaygreen
            $Script:isNG = $true
            Set-Variable -Name loginButton -Value $NGloginButton -Scope Script
            break
        }
        elseif ($SFloginButton -ne $null -and $SFloginButton.GetType() -ne [DBNull])
	    {
            "STOREFRONT DETECTED"  | Logme -displaygreen
            $Script:isNG = $false
            Set-Variable -Name loginButton -Value $SFloginButton -Scope Script
            break
        }
        elseif ($logoffLink -ne $null -and $logoffLink.GetType() -ne [DBNull])
	    {
            "Detected Resource Page; Skipping Login Page " | Logme -displaygreen
		    Set-Variable -Name IsResource -Value $true -Scope Script
		    break
	    }
        elseif ($ContinueButton -ne $null -and $ContinueButton.GetType() -ne [DBNull])
	    {
		    try
		    {
			    $ContinueButton.Click()
			    $ContinueButtonClicked = $true
			    "Click 'Continue' button to Get to Login page" | logme -displaygreen
		    }
		    catch
		    {
			    "Click 'Continue' button failed" | logme -error
		    }
		    
	    }
	    elseif ($ContinueButton1 -ne $null -and $ContinueButton1.GetType() -ne [DBNull])
	    {
		    try
		    {
			    $ContinueButton1.Click()
			    $ContinueButtonClicked = $true
			    "Click 'Continue' button to Get to Login page" | logme -displaygreen
		    }
		    catch
		    {
			    "Click 'Continue' button failed" | logme -error
		    }
	    }
	    Start-Sleep -Seconds 1
		$try++	
    } until ($try -gt $TimeToWaitForApp)
    
    
    if ( $IsResource -eq $false -and ($loginButton -eq $null -or $loginButton.GetType() -eq [DBNull]) ) 
	{
        "Log on button not found" | Logme -error
        $Script:loginAborted = $true
        #Logoff-Storefront
        #throw "Log on button not found"
    }    
}

function Submit-UserCredentials
{
    "Submitting User Credentials" | logme -displaynormal
    if ($Script:isNG -eq $True)
	{
        "Getting Login button from NG page" | logme -displaynormal
		$loginButton = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGLoginButtonId)
		if ( $loginButton -eq $null -or $loginButton.GetType() -eq [DBNull] )
		{
			$loginButton = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGLoginButtonId1)
		}
        "Getting UserName textbox" | Logme -displaynormal
        <# Checking for latest NetScaler 11.x version as name can't be used #>
        $userNameTextBox = @([System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGUserNameTextBoxName)) | where { $_.Id -ne $null -and $_.Id -eq $NGUserNameTextBoxName }
        if ( $userNameTextBox -eq $null -or $userNameTextBox.GetType() -eq [DBNull] )
		{
            <# Checking for previous version of NetScaler #>
			$userNameTextBox = @([System.__ComObject].InvokeMember("getElementsByName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGUserNameTextBoxName1)) | where { $_.name -ne $null -and $_.name -eq $NGUserNameTextBoxName1 }
            if ( $userNameTextBox -eq $null -or $userNameTextBox.GetType() -eq [DBNull] )
            {
                <# Checking for webinterface 5.x #>
                $userNameTextBox = @([System.__ComObject].InvokeMember("getElementsByName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGUserNameTextBoxName2)) | where { $_.name -ne $null -and $_.name -eq $NGUserNameTextBoxName2 }
            }
		}
        "Getting Password textbox" | Logme -displaynormal
        $passwordTextBox = @([System.__ComObject].InvokeMember(“getElementsByName”,[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGPasswordTextBoxName)) | where { $_.name -ne $null -and $_.name -eq $NGPasswordTextBoxName }
		if ( $passwordTextBox -eq $null -or $passwordTextBox.GetType() -eq [DBNull] )
		{
			$passwordTextBox = @([System.__ComObject].InvokeMember(“getElementsByName”,[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGPasswordTextBoxName1)) | where { $_.name  -ne $null -and $_.name -eq $NGPasswordTextBoxName1 }
            if ( $passwordTextBox -eq $null -or $passwordTextBox.GetType() -eq [DBNull] )
            {
                $passwordTextBox = @([System.__ComObject].InvokeMember(“getElementById”,[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGPasswordTextBoxName)) | where { $_.Id -ne $null -and $_.Id -eq $NGPasswordTextBoxName }
            }
		}
        if ($2FA -and $2FA -ne "none")
		{
            "Getting Two Factor Authentication textbox" | Logme -displaynormal
            $twoFactorTextBox = @([System.__ComObject].InvokeMember("getElementsByName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGTwoFactorTextBoxName)) | where { $_.name -ne $null -and $_.name -eq $NGTwoFactorTextBoxName }
            if ($twoFactorTextBox -eq $null -or $twoFactorTextBox.GetType() -eq [DBNull])
            {
                $twoFactorTextBox = @([System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $NGTwoFactorTextBoxName)) | where { $_.Id -ne $null -and $_.Id -eq $NGTwoFactorTextBoxName }
            }
            if ($twoFactorTextBox -ne $null -and $twoFactorTextBox.GetType() -ne [DBNull])
			{
				"Setting Two Factor Authentication" | Logme -displaynormal
                $twoFactorTextBox.value = $2FA
            }
			else
			{
				"Two-factor authentication textbox not found" | Logme -error
            }
        }

        #Terms & Conditions Checkbox might be come in login page
        try
        {
            $termsAndCondCheckBox = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $Eulacheck)
		}
        catch { }

    }
	else
	{

        "Getting Login button from SF page" | logme -displaynormal
        $loginButton = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLoginButtonId)
        "Getting UserName textbox" | Logme -displaynormal
        $userNameTextBox = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFUsernameTextBoxId)
        "Getting Password textbox" | Logme -displaynormal
        $passwordTextBox = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFPasswordTextBoxId)
        #Terms & Conditions Checkbox might be come in login page
        try
        {
            $termsAndCondCheckBox = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $Eulacheck)
		}
        catch { }
    }
	if ($userNameTextBox -ne $null -and $userNameTextBox.GetType() -ne [DBNull])
	{
        "Setting UserName`: $UserName" | Logme -displaygreen
        $userNameTextBox.Value = $UserName
    }
	else
	{
		"UserName textbox not found" | Logme -error
        "Make sure your Netscaler Gateway Virtual Server Portal Theme is set to either Default, X1, Greenbubble or your Custom Theme is created using one of these." | Logme -error
        $Script:loginAborted = $true
        #throw "UserName textbox not found"
	    return
    }
	if ($passwordTextBox -ne $null -and $passwordTextBox.GetType() -ne [DBNull])
	{
        "Setting Password`: *********" | Logme -displaygreen
        $passwordTextBox.Value = $pwd
    }
	else
	{
		"Password textbox not found" | Logme -error
        "Make sure your Netscaler Gateway Virtual Server Portal Theme is set to either Default, X1, Greenbubble or your Custom Theme is created using one of these." | Logme -error
        $Script:loginAborted = $true
        #throw "Password textbox not found"
	    return
    }
	
    #Click Terms & Conditions Checkbox
    try
    {
        if ($termsAndCondCheckBox -ne $null -and $termsAndCondCheckBox.GetType() -ne [DBNull])
        {
            "Setting Terms & Conditions" | Logme -displaygreen
            $termsAndCondCheckBox.Click()
        }
    }
    catch { }

    if ($loginButton -ne $null -and $loginButton.GetType() -ne [DBNull])
	{
        "Clicking Log On button" | Logme -displaynormal
        $loginButton.Click()
		Start-Sleep -Seconds 1
	}
	else
	{
        "Login button not found" | Logme -error
        $Script:loginAborted = $true
        #Logoff-Storefront
        #throw "Login button not found"
    }
}

function Check-Resource
{
	"Getting Storefront resources page" | Logme -displaynormal
    $try = 1
	do
	{
        
        try
        {
		    $skipLogonLinks = @([System.__ComObject].InvokeMember("getElementsByClassName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "_ctxsattr_title_SkipToLogonTip"))
		    foreach($skipLogonLink in $skipLogonLinks)
		    {
			    if(($skipLogonLink -ne $NULL -and $skipLogonLink.GetType() -ne [DBNull]) -and $skipLogonLink.innerText.toLower().contains("log on"))
			    {
				    $skipLogonLink.click()
			    }
		    }
        }
        catch{}
	
		$logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId)
		if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
		{
			$logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId1)
			if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
			{
				$logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId2)
				if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
				{
					$logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId3)
					if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
					{
						$logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId4)
						if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
						{
							$logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId5)
						}	
					}	
				}	
			}	
		}

		if ($Prompt -eq $true)
		{
			$ContinueButton = $null
			try
			{
				$ContinueButton = [System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "a") | where { $_.outerText -eq $ContinueButtonMsgPaneID }		
			}
			catch {}

			if ( $ContinueButton -eq $null -or $ContinueButton.GetType() -eq [DBNull] )
			{
				try
				{
					$ContinueButton = [System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "a") | where { $_.outerText -eq $ContinueButtonMsgPaneID2 -or $_.outerText -eq $ContinueButtonMsgPaneID3 }		
				} 
				catch {}
			}
		}

		if ($logoffLink -ne $null -and $logoffLink.GetType() -ne [DBNull])
		{
			"Successfully found Storefront resources page" | LogMe -displaygreen
            $Script:LoginTime2 = get-date
			break
		}
		elseif ($ContinueButtonClicked -eq $false -and $ContinueButton -ne $null -and $ContinueButton.GetType() -ne [DBNull])
		{
			try
			{
                $ContinueButton.Click()
				#"Click 'Continue' button to Get to Resource page" | logme -displaynormal
			}
			catch
			{
				
			}
		}
		else
		{
			"Try #$try`: Still looking for Storefront resources page..." | LogMe -displaynormal
		}
        Start-Sleep -Seconds 1
		$try++

	}
	until ($try -gt $TimeToWaitForApp)
	
	if ($logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull])
	{
		"StoreFront recources page not found, possible page loading issue or logon issue" | LogMe -error
		$Script:StoreFrontRP = $False
        $Script:InvalidAccount = $True
	}
}

function LookforResource
{
    #----------------------------------------------------------------------
	# If Nested Tab/Folder, Navigate to It 
	# Look for any nested IFrames First indicated by a '#' after each nested IFrame name
	#----------------------------------------------------------------------
	$windoc = $document;
    while ($TabName -ne $null -and $TabName.contains('#'))
	{
	    $frameName, $TabName = $TabName.split('#', 2);
		"Getting Frame '$frameName'..."	| logme -displaynormal		
		$elements = @([System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $windoc, $frameName))
		Start-Sleep -Seconds 3	
		$win = $elements.contentWindow
		$windoc = $win.document
	}		
	
	$folder = $null
	$folderLC = $null
	while ( $TabName -ne "" -and $TabName -ne $null )
	{
		$folder, $TabName = $TabName.split('/', 2);
		"Getting Tab/Folder '$folder'..." | logme -displaynormal			
			
		try
		{
		    $folderLC = $folder.toLower().Trim()
		} catch {}
			
		$try = 0
        $isTabFound = $False
        do
		{					
		    if ( $try -eq 0 )
			{
                if($folderLC -eq "apps")
                {
                    $AppElementId = "allAppsBtn"
                    "Start fetching for APPS tab element" | logme -displaygreen
                    try
                    {
                        $TabElement = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $windoc, $AppElementId)
                        if($TabElement -ne $NULL -and $TabElement.GetType() -ne [DBNull])
                        {
                            $tab = $TabElement
                            $isTabFound = $True    
                        }
                    }
                    catch {}
                    
                }
                elseif($folderLC -eq "desktops")
                {
                    $DesktopsElementId = "desktopsBtn"
                    "Start fetching for DESKTOPS tab element" | logme -displaygreen
                    try
                    {
                        $TabElement = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $windoc, $DesktopsElementId)
                        if($TabElement -ne $NULL -and $TabElement.GetType() -ne [DBNull])
                        {
                            $tab = $TabElement
                            $isTabFound = $True    
                        }
                    }
                    catch {}
                }
                elseif($folderLC -eq "favorites")
                {
                    $FavoritesElementId = "myAppsBtn"
                    "Start fetching for FAVORITES tab element" | logme -displaygreen
                    try
                    {
                        $TabElement = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $windoc, $FavoritesElementId)
                        if($TabElement -ne $NULL -and $TabElement.GetType() -ne [DBNull])
                        {
                            $tab = $TabElement
                            $isTabFound = $True    
                        }
                    }
                    catch {}
                }
           
			}
			    				
            if($isTabFound -eq $False)
            {
                "Start ArTag Get..." | logme -displaynormal
				$AnchorTags = @([System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $windoc, "a"))
                    
		        $tab = $null
			    $ot = $null
			    $title = $null
			    $alt = $null

			    # Reverse order of Objects in Array
			    try { [array]::Reverse($AnchorTags) } catch {}
				
			    $ResList = @()
			    foreach ( $sobj in $AnchorTags )
			    {
				    if ( $try -eq 0 )
				    {
					    try { $ot = $sobj.outerText.tolower().Trim() } catch {}
					    try { $title = $sobj.title.tolower().Trim() } catch {}
						
					    $ResList += $ot
				    }
				    else
				    {
					    try { $alt = $sobj.alt.tolower().Trim() } catch {}
						
					    $ResList += $alt
				    }
				    $ResList += ";"
					if ( ($ResList.count % 14) -eq 0 )
				    {
					    #"Searching for Tab/Folder: $ResList" | logme -displaynormal
					    $ResList = @()
				    }
                    if ( $ot -eq $folderLC -or $title -eq $folderLC -or $alt -eq $folderLC )
				    {
					    $tab = $sobj
					    break
				    }	
					
			    }
			    if ( $ResList -gt 0 )
			    {
				    #"Searching for Tab/Folder: $ResList" | logme -displaynormal
			    }			
			
            }
			# Look for and Click Tab Name if Found
			if ($tab -ne $null) 
			{
                "Try #$try`: Waiting..." | logme -displaynormal
				Start-Sleep -Seconds 1
					
				"Click Tab/Folder '$folder' to Get Resources" | logme -displaynormal
				$tab.Click()
					
				Start-Sleep -Seconds 1
					
				$try = 2
			}				
			$try++				
		} 
		until ($try -gt 1 ) #$NumberOfRetries)
	}
    $folder = $null
	$try = 0
	do
	{				
		if ( $try -eq 1 )			
		{
			"Start ArTag Get..." | logme -displaynormal
			$AnchorTags = @([System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $windoc, "a"))
		}
		else
		{
			"Start IgTag Get..." | logme -displaynormal
			$AnchorTags = @([System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $windoc, "img"))
		}
					
        $Script:resource = $null
		$tab = $null	
		$ot = $null
		$title = $null
		$alt = $null

		# Reverse order of Objects in Array
		try { [array]::Reverse($AnchorTags) } catch {}
			
		$ResList = @()
		foreach ( $sobj in $AnchorTags )
		{
				
			if ( $try -eq 1 )
			{
				$ot = $sobj.outerText
				$title = $sobj.title
					
				$ResList += $ot
			}
			else
			{
				$alt = $sobj.alt
					
				$ResList += $alt
			}
			$ResList += ";"
				
			if ( ($ResList.count % 14) -eq 0 )
			{
				#"Searching for Resource : $ResList" | logme -displaynormal
				$ResList = @()
			}

			if ( $ot -eq $App -or $title -eq $App -or $alt -eq $App )
			{
				$Script:resource = $sobj
				break
			}
		}
		if ( $ResList -gt 0 )
		{
			#"Searching for Resource : $ResList" | logme -displaynormal
		}			
		
		if ($resource -ne $null)
		{
            "Try #$try`: RESOURCE SUCCESS" | logme -displaynormal
			break
		} 
		else
		{
			$try++
		}
	} 
	until ($try -gt 1 ) #$NumberOfRetries)
}


function Set-WindowForeground
{
    Param(
	   [Parameter(Mandatory=$true,Position=0)] [int]$processid
	)
Add-Type @"
  using System;
  using System.Runtime.InteropServices;
  public class SFW {
     [DllImport("user32.dll")]
     [return: MarshalAs(UnmanagedType.Bool)]
     public static extern bool SetForegroundWindow(IntPtr hWnd);
     [DllImport("user32.dll")]  
     public static extern bool ShowWindow(IntPtr hWnd, int nCmdShow);
  }
"@
    $h=  (get-process -id $processid).MainWindowHandle
    #"The mainwindowhandle is $h" | logme -displaynormal
    [SFW]::ShowWindow($h,2) #Minimize Window
    start-sleep -seconds 1
    [SFW]::ShowWindow($h,3) #Maximize Window
    start-sleep -seconds 1
    $status = [SFW]::SetForegroundWindow($h)
    start-sleep -seconds 1
    #"status of foreground window is $status" | logme -displaynormal
}



function Launch-App
{
    foreach ($App in $ResourceName)
    {
        "========================================================" | logme -displaygreen
        "Testing `"$App`"" | logme -displaygreen
        "========================================================" | logme -displaygreen
        LookforResource

        if ($Script:resource -eq $null)
        {
            "The ResourceName specified, `"$App`", was not found either `"$App`" was mistyped on the command line or it is not visible on the Storefront main page" | LogMe -error
            continue
        }
        
        if($UseICA -eq 1)
        {
            $ICO = $null
            $EnumHandle = $null
            $NumSessions = $Null
            $sessionid = $Null
            $APPSERVER = $Null
            "Creating the ICA Client Object" | logme -displaynormal
            try {Add-Type -Path $DLLPath} 
            catch {throw "Error loading WfIcaLib.dll"}
            $ICO = New-Object WFICALib.ICAClientClass
           
            "Clicking resource `"$App`"" | Logme -displaynormal
	        $Script:resource.Click()
            $LaunchTime = Get-Date
	        "Connect Started at $LaunchTime"  | Logme -displaygreen
        
	        $index = $null
	        $try = 1
	        "Looking for a Session, time out is $TimeToWaitForApp seconds." | Logme -displaynormal
            
            
	        Do
	        {
                $ICO.OutputMode = [WFICALib.OutputMode]::OutputModeNormal
                $EnumHandle = $ICO.EnumerateCCMSessions()
                "Try #$try`: Waiting to find a Session" | logme -displaynormal
	            CheckforAlertBox
                if($Script:alertboxdetected -eq $true)
                {
                    break
                }
	            $NumSessions = $ICO.GetEnumNameCount($EnumHandle)
                if($NumSessions -gt 0) 
                {
                    break
                } 
	            Start-Sleep -Seconds 1
	            $try++
	        }
	        Until ($try -gt $TimeToWaitForApp)
            
            
	        if($Script:alertboxdetected -eq $true)
	        {
	            continue
	        }
	        else
            {
                if($NumSessions -eq 0)
	            {
	                "There is no Server connection and there are no sessions. Unable to confirm that session launched for `"$App`". If this occurs again, check the Storefront logs and/or manually try launching `"$App`"." | LogMe -error
	                continue
	            }
            }
	        
            $isICAconnected = $false
            $conntry = 1
            if($NumSessions -ge 1 )
	        {
                "Session found $NumSessions" | Logme -displaygreen
	            "Waiting for an Active Connection, time out is $TimeToWaitForApp seconds."   | Logme -displaynormal
	            $sessionid = $ICO.GetEnumNameByIndex($EnumHandle, $index)
	            $ICO.StartMonitoringCCMSession($sessionid,$true)
                Do
	            {
	                "Try #$conntry`: Waiting for a connection" | LogMe -displaynormal
                    $isICAconnected = $ICO.IsConnected()
	                if ($isICAconnected -eq $true)
                    {
                        
                        "Session is connected" | logme -displaygreen
                        break
                    }
	                $conntry++
	                Start-Sleep -Seconds 1
	            }
	            Until ($conntry -gt $TimeToWaitForApp)
                $AppName = $App.toLower()
                $isDisclaimerWindowFound = $false
	            if($isICAconnected -eq $true)
	            {
                                  
	                "Looking for an active XenApp Server for `"$App`"" | logme -displaynormal
                    
                    
                    
                    if($isDisclaimer -eq $True )
                    {
                        $apptry = 1
                        do
                        {
                            $wficaPPIDs = @()
                            $wficaPPIDs = @(Get-Process wfica32* -ErrorAction SilentlyContinue | where-object { $_.SessionId -eq $SID } | select-object -ExpandProperty id)
                            $wficaPPIDs += @(Get-Process cdviewer* -ErrorAction SilentlyContinue | where-object { $_.SessionId -eq $SID } | select-object -ExpandProperty id)
                            $pidCount = $wficaPPIDs.Count
                            if ($pidCount -gt 0)
                            {
                                foreach($wfica in $wficaPPIDs)
                                {
                                    $Process = Get-Process -Id $wfica -ErrorAction SilentlyContinue 
                                    $wficaTitle = $Process.MainWindowTitle
                                    if( -not [string]::IsNullOrEmpty($wficaTitle) -and ($wficaTitle.toLower().Contains("citrix receiver - security warning") -eq $false))
                                    {
                                        if(($wficaTitle.toLower().Contains("windows sign-in") -eq $True) -or ($wficaTitle.ToLower().Contains($AppName) -eq $True))
                                        {
                                            #"waiting for disclaimer to click" | logme -displaynormal
                                            start-sleep -seconds $DisclaimerTimeout
                                            $isDisclaimerWindowFound = $true
                                            break
                                        }
                                    }    
                                        
                                }

                                if($isDisclaimerWindowFound -eq $True)
                                {
                                    
                                    Set-WindowForeground -processid $wfica
                                    # send keys
					                $wshell = New-Object -ComObject wscript.shell
                                    $status = $wshell.AppActivate($wficaTitle)
                                    start-sleep -Seconds 2
                                    "The AppActivate status is $status" | logme -displaynormal
                                    $wshell.SendKeys('~')
                                    break
                                }
                            }
                            $apptry++
	                        Start-Sleep -Seconds 1
                        }
                        Until($apptry -gt $TimeToWaitForApp)
                    }
                    $apptry = 1
                    $numTry = ($TimeToWaitForApp / 6)
                    if($numTry -le 15)
                    {
                        $numTry = 15
                    }
                        
	                Do
	                {
	                    "Try #$apptry`: Waiting to find an active XenApp Server for `"$App`"" | LogMe -displaynormal
	                    $APPSERVER = $ICO.GetSessionString(0)
	                    if (-not [string]::IsNullOrEmpty($APPSERVER))
	                    {
	                        "The AppServer is $APPSERVER " | logme -displaygreen
	                        "Number of sessions found is $NumSessions" | logme -displaygreen
	                        "Found an active XenApp Server for `"$App`"" | LogMe -displaygreen
                            break
	                    }
                        $apptry++
	                    Start-Sleep -Seconds 6
	                }
	                Until ($apptry -gt $numTry)
	                                    
                    $WTitle = $WindowTitle.toLower()
                    if(-not [string]::IsNullOrEmpty($APPSERVER))
                    {
                          
                        $apptry = 1
                        
                        do
                        {
                            $wficaPPIDs = @()
                            $wficaPPIDs = @(Get-Process wfica32* -ErrorAction SilentlyContinue | where-object { $_.SessionId -eq $SID } | select-object -ExpandProperty id)
                            $wficaPPIDs += @(Get-Process cdviewer* -ErrorAction SilentlyContinue | where-object { $_.SessionId -eq $SID } | select-object -ExpandProperty id)
                            $pidCount = $wficaPPIDs.Count
                            if ($pidCount -gt 0)
                            {
                                foreach ($wfica in $wficaPPIDs)
                                {
                                    try
                                    {
                                        $Process = Get-Process -Id $wfica -ErrorAction SilentlyContinue 
                                        $wficaTitle = $Process.MainWindowTitle
                                        "Found process title is: $wficaTitle and given window title is: $WTitle and given resource name is: $AppName"   | Logme -displaynormal
                                        if( -not [string]::IsNullOrEmpty($wficaTitle) -and (($WTitle -ne "none") -and ($wficaTitle.toLower().Contains($WTitle))) -or ($wficaTitle.toLower().Contains($AppName)))
                                        {
                                            $logonelapsed = GetElapsedTime $LaunchTime $isDisclaimerWindowFound $DisclaimerTimeout
                                            $TitleMatched = $True
                                            break
                                        }
                                        else
                                        {
                                            $TopLevelWindows = $ICO.Session.TopLevelWindows
                                            [string] $TLW = $TopLevelWindows.Count
                                            [string] $pname = $Process.Name
                                            "Number of top level windows found $TLW for process $pname" | logme -displaynormal
                                            if ($TopLevelWindows.Count -gt 0)
                                            {   
                                                $logonelapsed = GetElapsedTime $LaunchTime $isDisclaimerWindowFound $DisclaimerTimeout
                                                $TitleMatched = $True
                                                break    
                                            }
                                                 
                                        }

                                    }
                                    catch {}
                                }

                                if($TitleMatched -eq $True)
                                {
                                    break
                                }
                            }
                            else 
                            {
                                "Neither wfica32 or cdviewer processes found for session: $SID at #$apptry"  | Logme -displaynormal
                            }
                            $apptry++
	                        Start-Sleep -Seconds 3
                        }
                        Until ($apptry -gt $numTry)
                        if($TitleMatched -eq $True)
                        {
                            "Time taken to launch App is: $logonelapsed"   | Logme -displaygreen
					        "Session ID: " + $sessionid  | Logme -displaygreen
					        "Active XenApp Server is: " + $APPSERVER   | Logme -displaygreen
                        }
                    }
                    else
                    {
                        "Unable to confirm that session launched for `"$App`". If this occurs again, check the Storefront (or Server logs) and/or manually try launching `"$App`"." | LogMe -error
                    }
                    Start-Sleep -Seconds $TimeToWaitForApp
                    for($index = 0; $index -lt $NumSessions;$index++)
                    {
                        $sessionid = $ICO.GetEnumNameByIndex($EnumHandle, $index)
                        $ICO.StartMonitoringCCMSession($sessionid,$true)
                        $ICO.Logoff()
                        $ICO.StopMonitoringCCMSession($sessionid)            
                    }
                    $ICO.CloseEnumHandle($EnumHandle)
                    "Waiting for session to Logoff, timeout is $TimeToWaitForLogoff seconds" | Logme -displaynormal
                    $logofftry = 1
		            $index = 0

					Do 
					{
						"Waiting for session to Logoff" | LogMe -displaynormal
						start-sleep -Seconds 3
                        try
                        {
                            $EnumHandle = $ICO.EnumerateCCMSessions()
						    $sessionid = $ICO.GetEnumNameByIndex($EnumHandle, $index)
                            $ICO.StartMonitoringCCMSession($sessionid,$true)
						    $NumSessions = $ICO.GetEnumNameCount($EnumHandle)
                        }
                        catch
                        {
                            #$e1 = $ICO.GetErrorMessage($ICO.GetLastError()) 
                            #"Error occured after logoff sent. The error is $e1" | Logme -error
                        }
                        "Number of sessions found after logoff sent $NumSessions" | logMe -displaynormal
                        if($NumSessions -eq 0)
                        {
                            break
                        }
						$logofftry++
					}
					until($logofftry -gt $TimeToWaitForLogoff)
                        
                    if ($NumSessions -ge 1 -or $logofftry -ge $TimeToWaitForLogoff) 
					{
						"Session is still connected, sending the Disconnect command" | logme -warning
						$ICO.Disconnect()
					}
                    
	            }
            
	        }
	        
	    }
        else
        {
            $sessionPId = 0
            $sessionMainWindowTitle = ""
            $isSessionConnected = $false
            $IsConnected = $false
            $AppName = $App.toLower()
            $WTitle = $WindowTitle.toLower()
            $wficaPPIDs = @()
		    "Clicking resource `"$App`"" | Logme -displaynormal
            $Script:resource.Click()
            $LaunchTime = Get-Date
            "Connect Started at $LaunchTime"  | Logme -displaygreen
            "Looking for a Session, time out is $TimeToWaitForApp seconds." | Logme -displaynormal
            $try = 1
            do
            {
                CheckforAlertBox
                if($Script:alertboxdetected -eq $true)
	            {
	                continue
	            }
                if($isSessionConnected -eq $false)
                {
                    $wficaPPIDs = @(Get-Process wfica32* -ErrorAction SilentlyContinue | where { $_.SessionId -eq $SID } | select -ExpandProperty id)
                    $wficaPPIDs += @(Get-Process cdviewer* -ErrorAction SilentlyContinue | where { $_.SessionId -eq $SID } | select -ExpandProperty id)
                    $pidCount = $wficaPPIDs.Count
                    if ($pidCount -gt 0)
                    {
                        foreach ($wfica in $wficaPPIDs)
                        {
                            $Process = Get-Process -Id $wfica -ErrorAction SilentlyContinue 
                            $wficaTitle = $Process.MainWindowTitle
                            if( -not [string]::IsNullOrEmpty($wficaTitle) -and ($wficaTitle.toLower().Contains("citrix receiver - security warning") -eq $false))
                            {                                 
                                if($wficaTitle.toLower() -eq "windows sign-in - \\remote")
                                {
                                    if($isDisclaimer -eq $True)
                                    {
                                        Start-Sleep -Seconds $DisclaimerTimeout
                                        Set-WindowForeground -processid $wfica
                                        # send keys
					                    $wshell = New-Object -ComObject wscript.shell
                                        $status = $wshell.AppActivate($wficaTitle)
                                        "The AppActivate status is $status" | logme -displaynormal
										start-sleep -Seconds 2
                                        $wshell.SendKeys('~')
                                        $isDisclaimer = $False
                                    }
              
                                }
                                "Found process title is: $wficaTitle and given window title is: $WTitle and given resource name is: $AppName"   | Logme -displaygreen
				                $c1 = ($wficaTitle.toLower().indexOf($AppName) -ge 0)
				                $c2 = ($WTitle -ne "none" -and $wficaTitle.tolower().indexOf($WTitle) -ge 0)
				                if ($c1 -or $c2)
					            {
                                    "Session is connected" | logme -displaygreen	
                                    $isSessionConnected = $True
                                    $sessionPId = $wfica
                                    $sessionMainWindowTitle = $wficaTitle
                                    break
                                }
                            }
                        }

                    }

                }
                if($isSessionConnected -eq $true)
                {
                    try
                    {
                        $Process = Get-Process -Id $sessionPId -ErrorAction SilentlyContinue 
                        $Process.WaitForInputIdle(1000) | out-null
                    }    
                    catch
                    {}
                    if($Process.Responding -eq $true)
                    {
                        $LaunchDetected = $true
                        break
                    }
                }
                "Try #$try`: Waiting to find a Session" | logme -displaynormal
                Start-Sleep -Seconds 1
                $try++
                
            }
            until ($try -gt $TimeToWaitForApp)
            
            if($LaunchDetected -eq $true)
            {
                if($isDisclaimer -eq $True)
                {
                    Start-Sleep -Seconds $DisclaimerTimeout
                    Set-WindowForeground -processid $sessionPId
                    # send keys
					$wshell = New-Object -ComObject wscript.shell
					$status = $wshell.AppActivate($sessionMainWindowTitle)
                    "The AppActivate status is $status" | logme -displaynormal
					Start-Sleep -Seconds 2
					$wshell.SendKeys('~')
                }
                "wficaTitle is $sessionMainWindowTitle" | logme -displaynormal
                $logonelapsed = GetElapsedTime $LaunchTime $isDisclaimerWindowFound $DisclaimerTimeout
                "Time taken to launch App is: $logonelapsed"   | Logme -displaygreen
                Start-Sleep -Seconds $TimeToWaitForApp
            }
            else
            {

                "Unable to confirm that session launched for `"$App`"." | logMe -warning
            }
            
            
        }
        "Sending Logoff Command"  | Logme -displaynormal
    }
}

function Logoff-Storefront
{
    "Logoff StoreFront" | logme -displaynormal
    if ($Script:StoreFrontRP -ne $False -and $Script:IEReady -eq $true -and $Script:loginAborted -eq $False)
	{
        "Sleeping $SleepBeforeLogoff seconds before logging off Storefront" | Logme -displaynormal
        Start-Sleep -Seconds $SleepBeforeLogoff
        $NumberOfLogoffRetries = $NumberOfRetries/2
        "Getting log off link" | LogMe -displaynormal
        $try = 1
        if($Script:InvalidAccount -eq $False)
        {
		    do
		    {    
		        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId)
                if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
		        {
			        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId1)
			        if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
			        {
				        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId2)
				        if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
				        {
					        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId3)
					        if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
					        {
						        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId4)
						        if ( $logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull] )
						        {
							        $logoffLink = [System.__ComObject].InvokeMember("getElementById",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, $SFLogOffLinkId5)
						        }	
					        }	
				        }	
			        }	
		        }
            
                if ($logoffLink -ne $null -and $logoffLink.GetType() -ne [DBNull])
                {
                    "log off link found" | logme -displaygreen
                    break
                }
                else
                {
                    "Try #$try`: Still looking for log off link..." | LogMe -displaynormal
				    Start-Sleep -Seconds 1
				    $try++
			    }
            } 
		    until ($try -gt $NumberOfLogoffRetries)
        
            
            if ($logoffLink -eq $null -or $logoffLink.GetType() -eq [DBNull])
		    {
		     "Log off link not found" | Logme -error
		    }
		    else
		    {
			    "Clicking log off link" | Logme -displaynormal
			    $logoffLink.Click()
            }
	    }
    }
}



#endregion Functions

function checkAuthentication
{
 	$script:authErrorStatus = $false
    try
	{
        <#$tags =  [System.__ComObject].InvokeMember("getElementsByTagName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "div")
        foreach($tag in $tags)
        {
            if($tag.innerText -ne $null -and $tag.innerText -eq "Incorrect user name or password.")
            {
                $script:authErrorStatus = $true
                break
            }
        }#>

        $error = @([System.__ComObject].InvokeMember("getElementsByClassName",[System.Reflection.BindingFlags]::InvokeMethod, $null, $document, "error"))
        if($error -ne $null)
        {
            $script:authErrorStatus = $true    
        }
           
	}
	catch
	{
        
	}
	return $script:authErrorStatus
}

Function Get-Screenshot
{
    param(
        [parameter(Position  = 0, Mandatory = 0)]
        [string]$OutPath
       )

     begin
     {
        $ErrorActionPreference = 'SilentContinue'
        Add-Type -AssemblyName System.Windows.Forms
 
        if (-not (Test-Path $OutPath))
        {
            New-Item $OutPath -ItemType Directory -Force
        }
     }

     process
     {
        try
        {
            #screenshot_[yyyyMMdd_HHmmss_ffff].png
            $FileNamePattern = 'Lsscreenshot_{0}.png'
            $fileName = $FileNamePattern -f (Get-Date).ToString('yyyyMMdd_HHmmss_ffff')
            $path = Join-Path $OutPath $fileName
            $b = New-Object System.Drawing.Bitmap([System.Windows.Forms.Screen]::PrimaryScreen.Bounds.Width, [System.Windows.Forms.Screen]::PrimaryScreen.Bounds.Height)
            $g = [System.Drawing.Graphics]::FromImage($b)
            $g.CopyFromScreen((New-Object System.Drawing.Point(0,0)), (New-Object System.Drawing.Point(0,0)), $b.Size)
            $g.Dispose()
            $b.Save($path)
        }
        catch
        {}            
     }
}


#region Try, Catch and Finally
try
{
    clear-host
	$param1 = $password
    $agentLoc = $LogfilePath.LastIndexOf("agent")
	$DirPath = $LogfilePath.substring(0,$agentLoc)
	$DirPath = $DirPath + "lib\OE.exe"
	& $DirPath $param1 | Tee-Object -Variable pwd | Out-Null
    if(-not [string]::IsNullOrEmpty($TwoFactorAuth))
    {
        $param2 = $TwoFactorAuth
        & $DirPath $param2 | Tee-Object -Variable 2FA | Out-Null
    }
    else
    {
        $2FA = "none"
    }
    if($pwd -eq $null -or $pwd.length -eq 0)
	{
        "Issue in decrypting the password. So Exit the shell"  | Logme -displaynormal
		throw "Issue in decrypting the password. So Exit the shell"
	}
    if (test-path $($LogFilePath.TrimEnd('\') + "\$LogFileName")) {remove-item -path $($LogFilePath.TrimEnd('\') + "\$LogFileName") -force }
    $startScriptTime = get-date
    $Script:UACStatus = Get-UACEnabled
    ""
    "**********  LAUNCHER SCRIPT START  **********" | Logme -displaynormal
    ""
    $SimUser = [System.Security.Principal.WindowsIdentity]::GetCurrent().Name
    "The Simulation is working under $SimUser Account on this endpoint" | logMe -displaynormal
    "UAC Status $Script:UACStatus " | LogMe -displaynormal
    "Script started at $startScriptTime" | LogMe -displaynormal
    "Checking for any Citrix Receiver Processes" | logme -displaynormal
    KillProcess
    "Checking for Citrix Client DLL" | logMe -displaynormal
    CheckforDLL
    "Checking Citrix Client Verion"  | Logme -displaynormal
    $ClientVersion = Get-ICAClientVersion 
    if ($ClientVersion -lt 12)
	{
        "Citrix Client $ClientVersion is untested, please use Citrix Receiver 12.x or higher"  | Logme -displaynormal
		throw "Citrix Client $ClientVersion is untested, please use Citrix Receiver 12.x or higher"
    }
	else
	{
        "Citrix Client Version: " + $ClientVersion  | Logme -displaygreen
    }
    
    CheckRegistry
    if ($Script:RegError -eq $Null)
	{
        $Script:LoginTime1= Get-Date
        Open-InternetExplorer -SiteURL $SiteURL
        if ($Script:IEReady -eq $true)
        {
            $Script:IETime = Get-Date
            $IEDiffTime = GetTimeDiff $Script:IETime $Script:LoginTime1
            "Time taken to launch IE process is: $IEDiffTime"   | Logme -displaygreen
            $Script:Time1 = Get-Date
		    Test-LoginForm
            if($Script:loginAborted -eq $False)
            {
                Submit-UserCredentials
                if($Script:loginAborted -eq $False)
                {
                    if ($Script:UACStatus -eq $true)
                    {
                        $internetExplorer = ConnectIExplorer -HWND $Script:HWND
                    }
                    if($internetExplorer -ne $null)
                    {
                        Wait-ForPageReady
						$script:authErrorStatus = checkAuthentication
                        if ($Script:IEReady -eq $true -and $script:authErrorStatus -eq $false)
                        {
                            $Script:UserTime = Get-Date
                            $UserDiffTime = GetTimeDiff $Script:UserTime $Script:Time1
                            "Time taken for user authentications is: $UserDiffTime"   | Logme -displaygreen
                            Check-Resource
                            if ($IsResource -eq $false -and $Script:loginAborted -eq $False -and $Script:InvalidAccount -eq $False)
		                    {
                                $LogonTimeElapsedTime = GetTimeDiff $Script:LoginTime2 $Script:LoginTime1
                                "Time taken to logon process is: $LogonTimeElapsedTime"   | Logme -displaygreen
		                        if ($Script:StoreFrontRP -ne $False)
		                        {
		                            
			                        Launch-App
		                        }
                            }
		                    else
                            {
		                        "Logon process failed"   | Logme -error
		                    }
                        }
                        else
                        {
                             "Logon process failed"   | Logme -error
                        }
                    }
                    else
                    {
                        "Logon process failed"   | Logme -error
                    }
                }
                else
                {
                    "Logon process failed"   | Logme -error
                }
            }
            else
            {
                "Logon process failed"   | Logme -error
            }
        }
    }
}
catch
{
	"Exception caught by script" | Logme -error
    $Error1 = $_.ToString()
    $Error1  | Logme -error
    $Error2 = $_.InvocationInfo.PositionMessage
    $Error2  | Logme -error
    throw $_
}
finally
{   
    if ($internetExplorer -is [System.__ComObject])
	{
        if ($internetExplorer | Get-Member 'Quit')
		{
            if($Script:loginAborted -eq $False -and $Script:authErrorStatus -eq $false)
            {
                Logoff-Storefront
            }
            if ($Script:IEReady -eq $true )
            {
                "Quitting Internet Explorer" | Logme -displaynormal
                $IELogoff = ([int32] $TimeToWaitForLogoff/2)
                if ($IELogoff -le 5)
                {
                    $IELogoff = 5
                }
	            Start-Sleep -Seconds $IELogoff
                try
                {
                    $internetExplorer.Quit()
                }
                catch{}
            }
        }
        "Releasing Internet Explorer Component Object Model (COM)" | Logme -displaynormal
        [System.Runtime.Interopservices.Marshal]::ReleaseComObject($internetExplorer) | Out-Null
        Remove-Variable -Name internetExplorer
   }
   KillProcess
   $scriptEndDate = get-date
   "Script end date $scriptEndDate" | LogMe -displaynormal
   ""
   "**********  LAUNCHER SCRIPT END  **********" | Logme -displaynormal
   ""
} 
#endregion Try, Catch and Finally



# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUhr5FN2AtlL2zbviIP57ClO0Q
# MO2gghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFLNdTGB4VTCJSeL+
# a4nVNbEIWdqaMA0GCSqGSIb3DQEBAQUABIICAKldkzFUoHTPeM2jhazwcKe/y/y0
# khQA/eiDZZf3OKnUQj7g+jEsnNzXkJ2/YEw/0XhNzKDAlXwWBHonpS1WbZo1QAha
# 7zQ+qnCQVAt+Cww03ni3BidaVimij9kiMOM6Ckk7Mq9qLLTnbaW50QEFSlzFZkfH
# sFgyD7LDRnkWWaccAN1TRYpkCW7cLJQ2lxxMo29wHRRFra/nCGNc+ztoxHgluoT5
# HEi+uilE1yzdIgGTDWzKdzZT+31F2wdjmifGRbIPZI/PEnnPJ2AoDpMKFu/42YIc
# K5ehzYecu89awDD+UiuA3cKya+WYcabgC67HZ2mlfEShrILYhBuFIJOmugind4k/
# rqpHfKPuYthdyb4k+vNpNG7b/rBE3hvG2QXT3U9wG44N4N8JZRCb8KLH6voXUOxo
# EkZA2V7qSQcai50ZAiO+PtAHufpvc3nU6BRXkaoV8DyAnWBjztaAzWkJL9PJYy5V
# A2nBTOcjsg2NFKWTU5e2OIdsjeZEpXZkN3ci56Jw4J+ad64wba5oM6bMMKZTv7EK
# Jso82HDqrL2BGwSe61/B2Qk3QbUDXeck2jwP+EolSDx+dNu9dujm4f+u8zNtKJtN
# 6WzPgo3GLof88Dmw21cBnF/zoY32loweIN6bHgTSeeihicoWYSj3NEI8jRBEjlpU
# lRrJBsceWMVhaCVqoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MTYwNjUxMTZaMC8GCSqGSIb3DQEJBDEiBCCFR5vKsRJJh1+bSK+GBxxeUWBA
# NHDrurTDH3XXY0llmDANBgkqhkiG9w0BAQEFAASCAQCxPMGRRUoMHteukjX9yrZl
# mC9XP5h8c4CoNqfwzadK/brmMwKVT2cg/M2eoaDbb5GVJXDr01f4wwFBBW4UtWiP
# t88gLNN22lRB5BwE0fy/h6r8WsLqsmBTVE4+qWlSBM0L52ixlnZU1HdylhsyWRaq
# qjM5fLMAs7emkW7z+6PH4R/6MCS28yIQ6g1fvRKVa45Yjac+mkpHDziT4TvuC5s5
# umJAs6rEzFj1rfqdwsedU/4CjKO67hq1i8mLQg09zEozUj94fUe472Jt3UeGpAI6
# a/AdZSzcoELeb6SgtfeLxacGak3zwq5eD2nBNxTckrAeWDrcQlcG0rj1G+3Sq3tN
# SIG # End signature block
