﻿#requires -Version 3.0
#requires -Module DHCPServer
#This File is in Unicode format.  Do not edit in an ASCII editor. Notepad++ UTF-8-BOM

#region help text
<#
.SYNOPSIS
	Creates a complete inventory of a Microsoft 2012+ DHCP server.
.DESCRIPTION
	Creates a complete inventory of a Microsoft 2012+ DHCP server using Microsoft 
	PowerShell, Word, plain text or HTML.
	
	Script requires at least PowerShell version 3 but runs best in version 5.
	
	Requires the DHCPServer module.
	Can be run on a DHCP server or on a Windows 8.x or Windows 10 computer with RSAT installed.
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520
	
	For Windows Server 2003, 2008 and 2008 R2, use the following to export and import the 
	DHCP data:
		Export from the 2003, 2008 or 2008 R2 server:
			netsh dhcp server export C:\DHCPExport.txt all
			
			Copy the C:\DHCPExport.txt file to the 2012+ server.
			
		Import on the 2012+ server:
			netsh dhcp server import c:\DHCPExport.txt all
			
		The script can now be run on the 2012+ DHCP server to document the older DHCP 
		information.

	For Windows Server 2008 and Server 2008 R2, the 2012+ DHCP Server PowerShell cmdlets 
	can be used for the export and import.
		From the 2012+ DHCP server:
			Export-DhcpServer -ComputerName 2008R2Server.domain.tld -Leases -File 
			C:\DHCPExport.xml 
			
			Import-DhcpServer -ComputerName 2012Server.domain.tld -Leases -File 
			C:\DHCPExport.xml -BackupPath C:\dhcp\backup\ 
			
			Note: The c:\dhcp\backup path must exist before the Import-DhcpServer 
			cmdlet is run.
	
	Using netsh is much faster than using the PowerShell export and import cmdlets.
	
	Processing of IPv4 Multicast Scopes is only available with Server 2012 R2 DHCP.
	
	Word and PDF Documents include a Cover Page, Table of Contents and Footer.
	
.PARAMETER ComputerName
	DHCP server to run the script against.
	The computername is used for the report title.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
.PARAMETER AllDHCPServers
	The script will process all Authorized DHCP servers that are online.
	"All DHCP Servers" is used for the report title.
	This parameter is disabled by default.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
	This parameter has an alias of ALL.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  This script creates a Word, PDF, HTML or 
	formatted text document.
#>
#endregion

#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None", DefaultParameterSetName = "Word") ]

Param(
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost",
	
	[parameter(Mandatory=$False)] 
	[Alias("ALL")]
	[Switch]$AllDHCPServers=$False
	)
#endregion

#region initialize variables
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DHCP config data ***"
$scriptEndIdentifier = "*** End - DHCP config data ***"
$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		
		If(Test-Connection -ComputerName $CName -quiet)
		{
			##Write-Verbose "$(Get-Date): Server $($CName) is online."
		}
		Else
		{
			##Write-Verbose "$(Get-Date): Computer $($CName) is offline"
			$ErrorActionPreference = $SaveEAPreference
			#Write-Error "`n`n`t`tComputer $($CName) is offline.`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			
			$ErrorActionPreference = $SaveEAPreference
			#Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip)
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
			
			$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
			If($? -and $Null -ne $results)
			{
				#the computer is a dhcp server
				
				Return $CName
			}
			ElseIf(!$? -or $Null -eq $results)
			{
				#the computer is not a dhcp server
				
				$ErrorActionPreference = $SaveEAPreference
				Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
				Exit
			}
		}
		Else
		{
			Write-Warning "Unable to resolve $($CName) to a hostname"
		}
	}
	Else
	{
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}
	Return $CName
}

Function TestComputerName2
{
	Param([string]$Cname)
	
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		##Write-Verbose "$(Get-Date): Testing to see if $($CName) is online and reachable"
		If(Test-Connection -ComputerName $CName -quiet)
		{
			##Write-Verbose "$(Get-Date): Server $($CName) is online."
		}
		Else
		{
			#Write-Verbose "$(Get-Date): Computer $($CName) is offline"
			Write-Output "$(Get-Date): Computer $($CName) is offline" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		##Write-Verbose "$(Get-Date): Computer name has been renamed from localhost to $($CName)"
		#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip)
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
			##Write-Verbose "$(Get-Date): Computer name has been renamed from $($ip) to $($CName)"
			#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
			$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
			If($? -and $Null -ne $results)
			{
				#the computer is a dhcp server
				#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
				Return $CName
			}
			ElseIf(!$? -or $Null -eq $results)
			{
				#the computer is not a dhcp server
				#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
				Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
				Return "BAD"
			}
		}
		Else
		{
			#Write-Verbose "$(Get-Date): Unable to resolve $($CName) to a hostname"
			Write-Output "$(Get-Date): Unable to resolve $($CName) to a hostname" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}
	Else
	{
		#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}

	#Write-Verbose "$(Get-Date): "
	Return $CName
}
#endregion

#region DHCP script  functions

Function ProcessIPv4Bindings
{   
 	$IPv4Bindings = Get-DHCPServerV4Binding -ComputerName $Script:DHCPServerName -EA 0 | Sort-Object IPAddress
	If($? -and $Null -ne $IPv4Bindings)
	{
	    ForEach($IPv4Binding in $IPv4Bindings)
	    {
			$interfaceAlias = $IPv4Binding.InterfaceAlias
			If($IPv4Binding.BindingState)
			{
				$bindingState = "Enabled"
			   
			}
			Else
			{
				$bindingState = "Disabled"  
				
			}
			Write-Host "Server Name : $($Script:DHCPServerName) $($delimiter)  ProtocolVersion : IPV4 $($delimiter) BindingState : $($bindingState) $($delimiter) IPAddress : $($IPv4Binding.IPAddress) $($delimiter) InterfaceAlias : $($interfaceAlias )"
			$interfaceAlias = ""
	    }
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host	 "Error retrieving IPv4 server bindings"
	}
	
	$IPv4Bindings = $Null
	[gc]::collect() 
}

Function ProcessIPv6Bindings
{
	$IPv6Bindings = Get-DHCPServerV6Binding -ComputerName $Script:DHCPServerName -EA 0 | Sort-Object IPAddress
	If($? -and $Null -ne $IPv6Bindings)
	{
		ForEach($IPv6Binding in $IPv6Bindings)
		{
		     $interfaceAlias = $IPv6Binding.InterfaceAlias
			If($IPv6Binding.BindingState)
			{
			  $BindingState = "Enabled"
			}
			Else
			{
			  $BindingState = "Disabled"
			}
			Write-Host "Server Name : $($Script:DHCPServerName) $($delimiter) ProtocolVersion : IPV6 $($delimiter) BindingState : $($BindingState) $($delimiter) IPAddress : $($IPv6Binding.IPAddress) $($delimiter) InterfaceAlias : $($interfaceAlias)"
            $interfaceAlias = ""
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host "Error retrieving IPv6 server bindings"
	}
	$IPv6Bindings = $Null
	[gc]::collect() 
}
#endregion

#region script setup function

Function ProcessScriptSetup
{
	$script:startTime = Get-Date
	
	#pre 1.40
	#$ComputerName = TestComputerName $ComputerName
	#$Script:DHCPServerName = $ComputerName
	
	#change for 1.40 and -AllDHCPServers
	$Script:DHCPServerNames = @()
	If($AllDHCPServers -eq $False)
	{
		##Write-Verbose "$(Get-Date): Resolving computer name"
		$ComputerName = TestComputerName $ComputerName
		$Script:DHCPServerNames += $ComputerName
	}
	Else
	{
		#Write-Verbose "$(Get-Date): Retrieving all DHCP servers in domain"
		$ComputerName = "All DHCP Servers"
		
		$ALLServers = Get-DHCPServerInDc -EA 0
		
		If($Null -eq $AllServers)
		{
			#oops no DHCP servers
			Write-Error "Unable to retrieve any DHCP servers.  Script cannot continue"
			Exit
		}
		Else
		{
			[int]$cnt = 0
			If($AllServers -is [array])
			{
				$cnt = $AllServers.Count
				#Write-Verbose "$(Get-Date): $($cnt) DHCP servers were found"
			}
			Else
			{
				$cnt = 1
				#Write-Verbose "$(Get-Date): $($cnt) DHCP server was found"
			}
			
			$Script:BadDHCPErrorFile = "$($pwd.Path)\BadDHCPServers_$(Get-Date -f yyyy-MM-dd_HHmm).txt"

			ForEach($Server in $AllServers)
			{
				$Result = TestComputerName2 $Server.DnsName
				
				If($Result -ne "BAD")
				{
					$Script:DHCPServerNames += $Result
				}
			}
			#Write-Verbose "$(Get-Date): $($Script:DHCPServerNames.Count) DHCP servers will be processed"
			#Write-Verbose "$(Get-Date): "
		}
	}
}
#endregion

#region script core
#Script begins

ProcessScriptSetup
Write-Host $scriptStartIdentifier

ForEach($DHCPServer in $Script:DHCPServerNames)
{
	$Script:DHCPServerName = $DHCPServer
	ProcessIPv4Bindings
	ProcessIPv6Bindings
}
If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion