﻿function New-CryptographyKey()
{
[CmdletBinding()]
[OutputType([System.Security.SecureString])]
[OutputType([String], ParameterSetName='PlainText')]
Param(
    [Parameter(Mandatory=$false, Position=1)]
    [ValidateSet('AES','DES','RC2','Rijndael','TripleDES')]
    [String]$Algorithm='AES',
    [Parameter(Mandatory=$false, Position=2)]
    [Int]$KeySize,
    [Parameter(ParameterSetName='PlainText')]
    [Switch]$AsPlainText
)
    Process
    {
        try
        {
            $Crypto = [System.Security.Cryptography.SymmetricAlgorithm]::Create($Algorithm)
            if($PSBoundParameters.ContainsKey('KeySize')){
                $Crypto.KeySize = $KeySize
            }
            $Crypto.GenerateKey()
            if($AsPlainText)
            {
                return [System.Convert]::ToBase64String($Crypto.Key)
            }
            else
            {
                return [System.Convert]::ToBase64String($Crypto.Key) | ConvertTo-SecureString -AsPlainText -Force
            }
        }
        catch
        {
            Write-Error $_
        }
        
    }
}

Function Protect-File
{
[CmdletBinding(DefaultParameterSetName='SecureString')]
[OutputType([System.IO.FileInfo[]])]
Param(
    [Parameter(Mandatory=$true, Position=1, ValueFromPipeline=$true, ValueFromPipelineByPropertyName=$true)]
    [Alias('PSPath','LiteralPath')]
    [string[]]$FileName,
    [Parameter(Mandatory=$false, Position=2)]
    [ValidateSet('AES','DES','RC2','Rijndael','TripleDES')]
    [String]$Algorithm = 'AES',
    [Parameter(Mandatory=$false, Position=3, ParameterSetName='SecureString')]
    [System.Security.SecureString]$Key = (New-CryptographyKey -Algorithm $Algorithm),
    [Parameter(Mandatory=$true, Position=3, ParameterSetName='PlainText')]
    [String]$KeyAsPlainText,
    [Parameter(Mandatory=$false, Position=4)]
    [System.Security.Cryptography.CipherMode]$CipherMode,
    [Parameter(Mandatory=$false, Position=5)]
    [System.Security.Cryptography.PaddingMode]$PaddingMode,
    [Parameter(Mandatory=$false, Position=6)]
    [String]$Suffix = ".dat",
    [Parameter()]
    [Switch]$RemoveSource
)
    Begin
    {
        #Configure cryptography
        try
        {
            if($PSCmdlet.ParameterSetName -eq 'PlainText')
            {
                $Key = $KeyAsPlainText | ConvertTo-SecureString -AsPlainText -Force
            }

            #Decrypt cryptography Key from SecureString
            $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($Key)
            $EncryptionKey = [System.Convert]::FromBase64String([System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR))

            $Crypto = [System.Security.Cryptography.SymmetricAlgorithm]::Create($Algorithm)
            if($PSBoundParameters.ContainsKey('CipherMode')){
                $Crypto.Mode = $CipherMode
            }
            if($PSBoundParameters.ContainsKey('PaddingMode')){
                $Crypto.Padding = $PaddingMode
            }
            $Crypto.KeySize = $EncryptionKey.Length*8
            $Crypto.Key = $EncryptionKey
        }
        Catch
        {
            Write-Error $_ -ErrorAction Stop
        }
    }
    Process
    {	
    	if([System.IO.File]::Exists($FileName)){
		$Files = Get-Item -LiteralPath $FileName
	}			
    
        ForEach($File in $Files)
        {
            #$DestinationFile = $File.FullName + $Suffix
	    $dir=Split-Path $File
            $DestinationFile = $dir+'\' + $File.BaseName + $Suffix

            Try
            {
                $FileStreamReader = New-Object System.IO.FileStream($File.FullName, [System.IO.FileMode]::Open)
                $FileStreamWriter = New-Object System.IO.FileStream($DestinationFile, [System.IO.FileMode]::Create)

                #Write IV (initialization-vector) length & IV to encrypted file
                $Crypto.GenerateIV()
                $FileStreamWriter.Write([System.BitConverter]::GetBytes($Crypto.IV.Length), 0, 4)
                $FileStreamWriter.Write($Crypto.IV, 0, $Crypto.IV.Length)

                #Perform encryption
                $Transform = $Crypto.CreateEncryptor()
                $CryptoStream = New-Object System.Security.Cryptography.CryptoStream($FileStreamWriter, $Transform, [System.Security.Cryptography.CryptoStreamMode]::Write)
                $FileStreamReader.CopyTo($CryptoStream)
    
                #Close open files
                $CryptoStream.FlushFinalBlock()
                $CryptoStream.Close()
                $FileStreamReader.Close()
                $FileStreamWriter.Close()

                #Delete unencrypted file
                if($RemoveSource){Remove-Item -LiteralPath $File.FullName}

                #Output ecrypted file
                $result = Get-Item $DestinationFile
                $result | Add-Member –MemberType NoteProperty –Name SourceFile –Value $File.FullName
                $result | Add-Member –MemberType NoteProperty –Name Algorithm –Value $Algorithm
                $result | Add-Member –MemberType NoteProperty –Name Key –Value $Key
                $result | Add-Member –MemberType NoteProperty –Name CipherMode –Value $Crypto.Mode
                $result | Add-Member –MemberType NoteProperty –Name PaddingMode –Value $Crypto.Padding
                $result
            }
            Catch
            {
                Write-Error $_
                If($FileStreamWriter)
                {
                    #Remove failed file
                    $FileStreamWriter.Close()
                    Remove-Item -LiteralPath $DestinationFile -Force
                }
                Continue
            }
            Finally
            {
                if($CryptoStream){$CryptoStream.Close()}
                if($FileStreamReader){$FileStreamReader.Close()}
                if($FileStreamWriter){$FileStreamWriter.Close()}
            }
        }
    }
}

Function Unprotect-File
{
[CmdletBinding(DefaultParameterSetName='SecureString')]
[OutputType([System.IO.FileInfo[]])]
Param(
    [Parameter(Mandatory=$true, Position=1, ValueFromPipeline=$true, ValueFromPipelineByPropertyName=$true)]
    [Alias('PSPath','LiteralPath')]
    [string[]]$FileName,
    [Parameter(Mandatory=$false, Position=2, ValueFromPipelineByPropertyName=$true)]
    [ValidateSet('AES','DES','RC2','Rijndael','TripleDES')]
    [String]$Algorithm = 'AES',
    [Parameter(Mandatory=$true, Position=3, ValueFromPipelineByPropertyName=$true, ParameterSetName='SecureString')]
    [System.Security.SecureString]$Key,
    [Parameter(Mandatory=$true, Position=3, ParameterSetName='PlainText')]
    [String]$KeyAsPlainText,
    [Parameter(Mandatory=$false, Position=4, ValueFromPipelineByPropertyName=$true)]
    [System.Security.Cryptography.CipherMode]$CipherMode = 'CBC',
    [Parameter(Mandatory=$false, Position=5, ValueFromPipelineByPropertyName=$true)]
    [System.Security.Cryptography.PaddingMode]$PaddingMode = 'PKCS7',
    [Parameter(Mandatory=$false, Position=6)]
    [String]$Suffix, #Assigning default value in code due to it not processing ".$Algorithm" properly when Algorithm is ValueFromPipelineByPropertyName
    [Parameter(Mandatory=$false, Position=7)]
    [String]$DstSuffix,
    [Parameter()]
    [Switch]$RemoveSource
)
    Process
    {
        #Configure cryptography
        try
        {
            if($PSCmdlet.ParameterSetName -eq 'PlainText')
            {
                $Key = $KeyAsPlainText | ConvertTo-SecureString -AsPlainText -Force
            }

            #Decrypt cryptography Key from SecureString
            $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($Key)
            $EncryptionKey = [System.Convert]::FromBase64String([System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR))

            $Crypto = [System.Security.Cryptography.SymmetricAlgorithm]::Create($Algorithm)
            $Crypto.Mode = $CipherMode
            $Crypto.Padding = $PaddingMode
            $Crypto.KeySize = $EncryptionKey.Length*8
            $Crypto.Key = $EncryptionKey
        }
        Catch
        {
            Write-Error $_ -ErrorAction Stop
        }

        if(-not $PSBoundParameters.ContainsKey('Suffix'))
        {
            $Suffix = ".dat"
        }
	if(-not $PSBoundParameters.ContainsKey('DstSuffix'))
        {
            $DstSuffix = ".csv"
        }

        #Used to store successfully decrypted file names.
        $Files = Get-Item -LiteralPath $FileName

        ForEach($File in $Files)
        {
            #Verify file ends with supplied suffix
            If(-not $File.Name.EndsWith($Suffix))
            {
                Write-Error "$($File.FullName) does not have an extension of '$Suffix'."
                Continue
            }

            #$DestinationFile = $File.FullName -replace "$Suffix$"
	        $DestFile = $File.FullName -replace "$Suffix$"
            $DestinationFile=$DestFile+$DstSuffix

            Try
            {
                $FileStreamReader = New-Object System.IO.FileStream($File.FullName, [System.IO.FileMode]::Open)
                $FileStreamWriter = New-Object System.IO.FileStream($DestinationFile, [System.IO.FileMode]::Create)

                #Get IV from file
                [Byte[]]$LenIV = New-Object Byte[] 4
                $FileStreamReader.Seek(0, [System.IO.SeekOrigin]::Begin) | Out-Null
                $FileStreamReader.Read($LenIV,  0, 3) | Out-Null
                [Int]$LIV = [System.BitConverter]::ToInt32($LenIV,  0)
                [Byte[]]$IV = New-Object Byte[] $LIV
                $FileStreamReader.Seek(4, [System.IO.SeekOrigin]::Begin) | Out-Null
                $FileStreamReader.Read($IV, 0, $LIV) | Out-Null
                $Crypto.IV = $IV

                #Peform Decryption
                $Transform = $Crypto.CreateDecryptor()
                $CryptoStream = New-Object System.Security.Cryptography.CryptoStream($FileStreamWriter, $Transform, [System.Security.Cryptography.CryptoStreamMode]::Write)
                $FileStreamReader.CopyTo($CryptoStream)

                #Close open files
                $CryptoStream.FlushFinalBlock()
                $CryptoStream.Close()
                $FileStreamReader.Close()
                $FileStreamWriter.Close()

                #Delete encrypted file
                if($RemoveSource){Remove-Item $File.FullName}

                #Output decrypted file
                Get-Item $DestinationFile | Add-Member –MemberType NoteProperty –Name SourceFile –Value $File.FullName -PassThru
            }
            Catch
            {
                Write-Error $_
                If($FileStreamWriter)
                {
                    #Remove failed file
                    $FileStreamWriter.Close()
                    Remove-Item -LiteralPath $DestinationFile -Force
                }
                Continue
            }
            Finally
            {
                if($CryptoStream){$CryptoStream.Close()}
                if($FileStreamReader){$FileStreamReader.Close()}
                if($FileStreamWriter){$FileStreamWriter.Close()}
            }
        }
    }
}

function Get-CryptoKey(){
[CmdletBinding(DefaultParameterSetName='SecureString')]
    Param(
        [Parameter(Mandatory=$true, Position=1)]
        [String]$FileDir,
        [Parameter(Mandatory=$true, Position=2)]
        [String]$FileName
    )
    Begin
    {
        Add-Type -AssemblyName System.Security
        $null=[Reflection.Assembly]::LoadWithPartialName("System.Security")
    }
    Process
    {
        $rijndael = new-Object System.Security.Cryptography.RijndaelManaged
        $rijndael.GenerateKey()
        $key=[Convert]::ToBase64String($rijndael.Key)
        $rijndael.Dispose()

        $destFile=$FileDir+$FileName

        $stream = [System.IO.StreamWriter] $destFile 
        $stream.WriteLine($key)
        $stream.close()
        return $key
    } 
}

Function Protect-Value {
    Param(
        [Parameter(Mandatory=$true, Position=1)]
        [String]$PlainValue,
        [Parameter(Mandatory=$true, Position=2)]
        [String]$Key
    )
    Process
    {	
        $Crypto       = $null
        $Encryptor    = $null
        $StreamWriter = $null
        $CryptoStream = $null
        $MemoryStream = $null

    	Try
            {
                $secureString = $Key | ConvertTo-SecureString -AsPlainText -Force

                #Decrypt cryptography Key from SecureString
                $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($secureString)
                $EncryptionKey = [System.Convert]::FromBase64String([System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR))

                $Crypto = [System.Security.Cryptography.SymmetricAlgorithm]::Create('AES')
                $Crypto.Mode = 'CBC'
                $Crypto.Padding = 'PKCS7'
                $Crypto.KeySize = $EncryptionKey.Length*8
                $Crypto.Key = $EncryptionKey
                $Crypto.GenerateIV()

                # Create the streams used for encryption.
                $MemoryStream = [System.IO.MemoryStream]::new()
                
                #Perform encryption
                $Encryptor  = $Crypto.CreateEncryptor()
                $CryptoStream = New-Object System.Security.Cryptography.CryptoStream($MemoryStream, $Encryptor , [System.Security.Cryptography.CryptoStreamMode]::Write)
                $StreamWriter = [System.IO.StreamWriter]::new($CryptoStream)

                # Write all data to the stream.
                $StreamWriter.Write($PlainValue)
                $StreamWriter.Close()
                $CryptoStream.Close()

                $EncryptedBytes = $MemoryStream.ToArray()
                $MemoryStream.Close()

                # Append the initialization vector to the encrypted bytes.
                $EncryptedValue = New-Object -TypeName Byte[] -ArgumentList ($Crypto.IV.Length + $EncryptedBytes.Length)
                [Array]::Copy($Crypto.IV, 0, $EncryptedValue, 0, $Crypto.IV.Length)
                [Array]::Copy($EncryptedBytes, 0, $EncryptedValue, $Crypto.IV.Length, $EncryptedBytes.Length)

                # Return the encrypted bytes with initialization vector.
                Write-Output -InputObject $EncryptedValue
            }
            Catch
            {
                $ErrorMessage=([string]$_.Exception)+($_.ScriptStackTrace)
                Write-Host ('Exception occurred while protect value - '+$ErrorMessage) -ForegroundColor Red
            }
            Finally
            {
                if($null -ne $Crypto){$Crypto.Dispose()}
                if($null -ne $Encryptor){$Encryptor.Dispose()}
                if($null -ne $CryptoStream){$CryptoStream.Dispose()}
                if($null -ne $MemoryStream){$MemoryStream.Dispose()}
                if($null -ne $StreamWriter){$StreamWriter.Dispose()}
            }
	}			
}

Function Unprotect-Value {
    Param(
        [Parameter(Mandatory=$true, Position=1)]
        [String]$ProtectdValue,
        [Parameter(Mandatory=$true, Position=2)]
        [String]$Key
    )
    Process
    {	
        $Crypto       = $null
        $Decryptor    = $null
        $StreamReader = $null
        $CryptoStream = $null
        $MemoryStream = $null

    	Try
            {
                $EncryptedBytesTemp = [Convert]::FromBase64String($ProtectdValue)

                $secureString = $Key | ConvertTo-SecureString -AsPlainText -Force

                #Decrypt cryptography Key from SecureString
                $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($secureString)
                $DecryptionKey = [System.Convert]::FromBase64String([System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR))

                $Crypto = [System.Security.Cryptography.SymmetricAlgorithm]::Create('AES')
                $Crypto.Mode = 'CBC'
                $Crypto.Padding = 'PKCS7'
                $Crypto.KeySize = $DecryptionKey.Length*8
                $Crypto.Key = $DecryptionKey

                # Extract the initialization vector and encrypted bytes.
                $BitsInByte = 8
                $InitializationVector = New-Object -TypeName Byte[] -ArgumentList ($Crypto.BlockSize / $BitsInByte)
                $EncryptedBytes       = New-Object -TypeName Byte[] -ArgumentList ($EncryptedBytesTemp.Length - $InitializationVector.Length)

                [Array]::Copy($EncryptedBytesTemp, $InitializationVector, $InitializationVector.Length)
                [Array]::Copy($EncryptedBytesTemp, $InitializationVector.Length, $EncryptedBytes, 0, $EncryptedBytes.Length)

                $Crypto.IV = $InitializationVector

                # Create the streams used for encryption.
                $MemoryStream = [System.IO.MemoryStream]::new($EncryptedBytes)
                
                #Perform decryption
                $Decryptor  = $Crypto.CreateDecryptor()
                $CryptoStream = New-Object System.Security.Cryptography.CryptoStream($MemoryStream, $Decryptor , [System.Security.Cryptography.CryptoStreamMode]::Read)
                $StreamReader = [System.IO.StreamReader]::new($CryptoStream)

                # Read the decrypted bytes from the decrypting stream and place them in a string.
                $PlainValue = $StreamReader.ReadToEnd()

                $StreamReader.Close()
                $CryptoStream.Close()
                $MemoryStream.Close()

                Write-Output -InputObject $PlainValue
            }
            Catch
            {
                $ErrorMessage=([string]$_.Exception)+($_.ScriptStackTrace)
                Write-Host ('Exception occurred while unprotect value - '+$ErrorMessage) -ForegroundColor Red
            }
            Finally
            {
                if($null -ne $Crypto){$Crypto.Dispose()}
                if($null -ne $Decryptor){$Decryptor.Dispose()}
                if($null -ne $CryptoStream){$CryptoStream.Dispose()}
                if($null -ne $MemoryStream){$MemoryStream.Dispose()}
                if($null -ne $StreamReader){$StreamReader.Dispose()}
            }
	}			
}

Export-ModuleMember -Function New-CryptographyKey
Export-ModuleMember -Function Protect-File
Export-ModuleMember -Function Unprotect-File
Export-ModuleMember -Function Get-CryptoKey
Export-ModuleMember -Function Protect-Value
Export-ModuleMember -Function Unprotect-Value