$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Add-PSSnapin Microsoft.Exchange.Management.PowerShell.E2010
$SRVSettings = Get-ADServerSettings
if ($SRVSettings.ViewEntireForest -eq "False")
{
	Set-ADServerSettings -ViewEntireForest $true
}

Write-Host "MailBox Server Rollup Details"
Write-Host "---------------------------------------"
$computer = $env:COMPUTERNAME
$Exch = Get-ExchangeServer -identity $computer  | where {$_.ServerRole -ne "Edge"} 
if($Exch)
{
	# Check for exchange server admin version
	if($Exch.AdminDisplayVersion.major -eq "8")
	{
		$E2K7 = $true
	}
	if($Exch.AdminDisplayVersion.major -eq "14")
	{
		$E2K = $true
	}
	Write-Host "Server Name: " $computer
	Write-Host "Server Role: " $Exch.ServerRole
	Write-Host "Exchange Version: " $Exch.AdminDisplayVersion
	
	#Get Exchange server version
	$MsxVersion = $Exch.ExchangeVersion		
		
	$Srv = $Exch.Name
		
	if($E2K -eq $true)
	{
		$key = "SOFTWARE\Microsoft\Windows\CurrentVersion\Installer\UserData\S-1-5-18\Products\AE1D439464EB1B8488741FFA028E291C\Patches\"
	}
		
	if($E2K7 -eq $true)
	{
		$key = "SOFTWARE\Microsoft\Windows\CurrentVersion\Installer\UserData\S-1-5-18\Products\461C2B4266EDEF444B864AD6D9E5B613\Patches\"
	}
		
	$type = [Microsoft.Win32.RegistryHive]::LocalMachine
	$regKey = [Microsoft.Win32.RegistryKey]::OpenRemoteBaseKey($type, $Srv)
	$regKey = $regKey.OpenSubKey($key)
		
	if ($regKey.SubKeyCount -eq 0)
	{
		$DetailRollup+= "No Patches installed"
	}
	else
	{
		#Loop each of the subkeys (Patches) and gather the Installed date and Displayname of the Exchange patch
		$ErrorActionPreference = "SilentlyContinue"
		ForEach($sub in $regKey.GetSubKeyNames())
		{
			$SUBkey = $key + $Sub
			$SUBregKey = [Microsoft.Win32.RegistryKey]::OpenRemoteBaseKey($type, $Srv)
			$SUBregKey = $SUBregKey.OpenSubKey($SUBkey)
			ForEach($SubX in $SUBRegkey.GetValueNames())
			{
				# Display Installed date and Displayname of the Exchange patch
				IF ($Subx -eq "Installed") 
				{
					$d = $SUBRegkey.GetValue($SubX)
					$d = $d.substring(4,2) + "/" + $d.substring(6,2) + "/" + $d.substring(0,4)
				}
				IF ($Subx -eq "DisplayName") 
				{
					$cd = $SUBRegkey.GetValue($SubX)
					$DetailRollup+=  "Rollup Version:  $($d) - $($cd)"
				}
			}
		}	
		$DetailRollup+=  ""
	}
	Write-Host $DetailRollup	
}


Write-Host "Mailbox Server - Summary Information (Out of DAG servers)"
Write-Host "------------------------------------------------------------"

$MBXInfo = Get-MailboxServer -identity $env:COMPUTERNAME -Status | where-object{$_.DatabaseAvailabilityGroup -eq $null}
if($MBXInfo)
{
	Write-Host "Server Name: " $MBXInfo.Name
	Write-Host "AutoDatabaseMountDial: " $MBXInfo.AutoDatabaseMountDial
	Write-Host "Exchange Version: " $MBXInfo.ExchangeVersion
	Write-Host "WorkCycle: " $MBXInfo.CalendarRepairWorkCycle
	Write-Host "WorkCycleCheckpoint: " $MBXInfo.CalendarRepairWorkCycleCheckpoint
	Write-Host "LogEnabled: " $MBXInfo.CalendarRepairLogEnabled
	Write-Host "LogSubjectLoggingEnabled: " $MBXInfo.CalendarRepairLogSubjectLoggingEnabled
	Write-Host "LogFileAgeLimit: " $MBXInfo.CalendarRepairLogFileAgeLimit
	Write-Host "LogDirectorySizeLimit: " $MBXInfo.CalendarRepairLogDirectorySizeLimit
	Write-Host "LogPath: " $MBXInfo.CalendarRepairLogPath	
}

Write-Host "Mailbox Server - Databases Backup Status"
Write-Host "-----------------------------------------------------"

$MBXBKList = Get-MailboxDatabase -Server $env:COMPUTERNAME -Status | where {$_.Recovery -eq $False -AND $_.ReplicationType -ne "Remote"} | sort Server
foreach ($MBXBK in $MBXBKList)
{
	Write-Host "Server Name: " $MBXBK.Server
	if($MBXBK.LastFullBackup -eq $null)
	{
		Write-Host "LastFullBackup: Never Backedup"
	}
	else
	{
		Write-Host "LastFullBackup: " $MBXBK.LastFullBackup
	}	
	Write-Host "Database Name: " $MBXBK.Identity
	Write-Host "LastIncrementalBackup: " $MBXBK.LastIncrementalBackup
	Write-Host "LastDifferentialBackup: " $MBXBK.LastDifferentialBackup
	Write-Host "LastCopyBackup: " $MBXBK.LastCopyBackup
	Write-Host "SnapshotLastFullBackup: " $MBXBK.SnapshotLastFullBackup
	Write-Host "SnapshotLastIncrementalBackup: " $MBXBK.SnapshotLastIncrementalBackup
	Write-Host "SnapshotLastDifferentialBackup: " $MBXBK.SnapshotLastDifferentialBackup
	Write-Host "SnapshotLastCopyBackup: " $MBXBK.SnapshotLastCopyBackup
	Write-Host "RPC Client Access Server: " $MBXBK.RpcClientAccessServer
	
}

Write-Host "Mailbox Permissions"
Write-Host "---------------------------"
$mailboxes=get-mailbox -server $env:computername -resultsize unlimited | Sort-Object name
foreach ($mailbox in $mailboxes)
{
	$sendbehalfs=Get-Mailbox $mailbox | select-object -expand grantsendonbehalfto | select-object -expand rdn | Sort-Object Unescapedname
	$senders=Get-ADPermission $mailbox.identity | ?{($_.extendedrights -like "*send-as*")} | Sort-Object name
	$fullsenders=Get-Mailbox $mailbox | Get-MailboxPermission | ?{($_.AccessRights -like "*fullaccess*") -and ($_.User -notlike "*nt authority\system*") -and ($_.User -notlike "*Exchange Trusted Subsystem*") -and ($_.User -notlike "*Exchange Servers*")}
	
	Write-Host "Mailbox Name: " $mailbox.Name
	Write-Host "PrimarySMTPAddress: " $mailbox.primarysmtpaddress

	if(!$senders)
	{
		Write-Host "Send As Permission Owner: "("No Users","-","-") -Separator "#~#"
	}
	else
	{
		foreach($sender in $senders)
		{
			Write-Host "Send As Permission Owner: "($sender.user,$sender.deny,$sender.isinherited) -Separator "#~#"
		}
	}
	
	if(!$fullsenders)
	{
		Write-Host "Full Access Permission Owner: "("No Users","-","-") -Separator "#~#"
	}
	else
	{
		foreach($fullsender in $fullsenders)
		{
			Write-Host "Full Access Permission Owner: "($fullsender.user,$fullsender.deny,$fullsender.isinherited) -Separator "#~#"
		}
	}
	
	if(!$sendbehalfs)
	{
		Write-Host "Send On Behalf Permission Owner: "("No Users","-","-") -Separator "#~#"
	}
	else
	{
		foreach($sendbehalf in $sendbehalfs)
		{
			Write-Host "Send On Behalf Permission Owner: " ($sendbehalf.user,$sendbehalf.deny,$sendbehalf.isinherited) -Separator "#~#"
		}
	}	
}

Write-Host "Mailbox server - Offline Address Book"
Write-Host "----------------------------------------"

$OABS = Get-OfflineAddressBook -server $env:computername

foreach ($OAB in $OABS)
{
	Write-Host "Server Name: " $OAB.Server
	Write-Host "Name: " $OAB.Name
	Write-Host "Address Lists: " $OAB.AddressLists
	Write-Host "Versions: " $OAB.Versions
	Write-Host "IsDefault: " $OAB.IsDefault
	Write-Host "PublicFolderDatabase: " $OAB.PublicFolderDatabase
	Write-Host "PublicFolderDistributionEnabled: " $OAB.PublicFolderDistributionEnabled
	Write-Host "VirtualDirectories: " $OAB.VirtualDirectories
}

Write-Host "Public Folders"
Write-Host "---------------"

$pf = Get-publicfolder -recurse -server $env:computername 

foreach ($p in $pf)
{
	Write-Host "Name: " $p.Name
	Write-Host "MaxItemSize: " $p.MaxItemSize
	Write-Host "ProhibitPostQuota: " $p.ProhibitPostQuota
	Write-Host "AgeLimit: " $p.AgeLimit
	Write-Host "HasSubFolders: " $p.HasSubFolders
	Write-Host "HiddenFromAddressListsEnabled: " $p.HiddenFromAddressListsEnabled
	Write-Host "IssueWarningQuota: " $p.IssueWarningQuota
	Write-Host "LocalReplicaAgeLimit: " $p.LocalReplicaAgeLimit
	Write-Host "MailEnabled: " $p.MailEnabled
	Write-Host "ParentPath: " $p.ParentPath
	Write-Host "PerUserReadStateEnabled: " $p.PerUserReadStateEnabled
	Write-Host "UseDatabaseAgeDefaults: " $p.UseDatabaseAgeDefaults
	Write-Host "UseDatabaseQuotaDefaults: " $p.UseDatabaseQuotaDefaults
	Write-Host "UseDatabaseRetentionDefaults: " $p.UseDatabaseRetentionDefaults
	Write-Host "FolderType: " $p.FolderType
	Write-Host "HasRules: " $p.HasRules
	Write-host "HasModerator: " $p.HasModerator
	Write-Host "Identity: " $p.Identity
	Write-Host "MapiIdentity: " $p.MapiIdentity
}

Write-Host "Public Folder Databases"
Write-Host "--------------------------"

$PFDall = Get-ExchangeServer -identity $env:computername |where-object{$_.serverrole -eq "none" -OR $_.serverrole -like "*mailbox*"} | Get-PublicFolderDatabase -Status | where-Object{$_.PublicFolderHierarchy -eq "Public Folders"}

foreach ($PFD in $PFDall)
{
	Write-Host "Server Name: " $PFD.Server
	Write-Host "Name: " $PFD.Name
	Write-Host "MaxItemSize: " $PFD.MaxItemSize
	Write-Host "ProhibitPostQuota: " $PFD.ProhibitPostQuota
	Write-Host "ReplicationMessageSize: " $PFD.ReplicationMessageSize
	Write-Host "UseCustomReferralServerList: " $PFD.UseCustomReferralServerList
	Write-Host "CustomReferralServerList: " $PFD.CustomReferralServerList
	Write-Host "AdministrativeGroup: " $PFD.AdministrativeGroup
	Write-Host "ActivationPreference: " $PFD.ActivationPreference
	Write-Host "DatabaseSize: " $PFD.DatabaseSize
	Write-Host "AvailableNewMailboxSpace: " $PFD.AvailableNewMailboxSpace
	Write-Host "LastFullBackup: " $PFD.LastFullBackup
	Write-Host "DeletedItemRetention: " $PFD.DeletedItemRetention

}


Write-Host "Public Folders Client Permissions"
Write-Host "--------------------------------------"

$pf = Get-publicfolder -recurse -server $env:computername | get-publicfolderclientpermission

foreach ($p in $pf)
{
	Write-Host "Identity: " $p.Identity
	Write-Host "User: " $p.User
	Write-Host "AccessRights: " $p.AccessRights
}


Remove-PSSnapin Microsoft.Exchange.Management.PowerShell.E2010