# This Script will give all details of Citrix XenDesktop Site Apps in Citrix Excalibur
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$Global:ErrorActionPreference="SilentlyContinue"
Add-PSSnapin "Citrix.Common.Commands"
Add-PSSnapin "Citrix.Host.Admin.V2"
Add-PSSnapin "Citrix.Broker.Admin.V2"

<#
Function Get-BrokerMachinesPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "MachineName")
	$desktops = @()
	$desktops += Get-BrokerMachine -Filter {RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'} -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($desktops.count / $total * 100, 1)
		#Write-Host  "Getting Broker Machines. Retrieved $($desktops.count) of $total machines completed in $remainingPerc%"
		$last = $desktops[-1].MachineName
		$desktops += Get-BrokerMachine -Filter {RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'} -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { MachineName -gt $last } @argumentList
	}
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($applications.count / $total * 100, 1)
		#Write-Host  "Getting Broker Applications. Retrieved $($applications.count) of $total applications completed in $remainingPerc%"
		$last = $applications[-1].Uid
		$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $applications
}
#>

Function Get-BrokerMachinesPaged
{
	param (
	$DesktopGroupUid,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$desktops = @()
	$LastUid = 0;
	do
	{
		$NewDesktops = @(Get-BrokerMachine -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid -and RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'} @argumentList)
		if ($NewDesktops.Count -gt 0)
		{
			$desktops += $NewDesktops
			$LastUid = $desktops[-1].Uid
		}
	} while ($NewDesktops.Count -ge $MaxRecordCount)
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$LastUid = 0;
	do
	{
		$NewApplications = @(Get-BrokerApplication -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewApplications.Count -gt 0)
		{
			$applications += $NewApplications
			$LastUid = $applications[-1].Uid
		}
	} while ($NewApplications.Count -ge $MaxRecordCount)
	return $applications
}

#BrowserName and Name Map Hashtbale
$AppNameHash = @{}
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
$PublishedAppInstances = Get-BrokerApplicationsPaged
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		$BrowserName = $PublishedAppInstance.BrowserName
		if ($BrowserName -ne $null -and $BrowserName.length -gt 0)
		{
			$Name = $PublishedAppInstance.Name
			if ($Name -eq $null -or $Name.length -eq 0)
			{
				$Name = $BrowserName
			}
			$Name = $Name + "(" + $BrowserName + ")"
			$AppNameHash.Add($BrowserName,$Name)
		}
	}
}

Write-Host "Broker Machines"
Write-Host "---------------"
#$BrokerMachines = Get-BrokerMachine -Filter {RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'}
$BrokerMachines = Get-BrokerMachinesPaged
if($BrokerMachines -ne $null)
{
	foreach($BrokerMachine in $BrokerMachines)
	{
		$PublishedApplications = $BrokerMachine.PublishedApplications
		if ($PublishedApplications -eq $null -or $PublishedApplications.length -eq 0)
		{
			$PublishedApplications = "-"
		}
		else
		{
			$PublishApps = ""
			$count = 0
			foreach ($key in $PublishedApplications)
			{
				if ($count -eq 0)
				{
					$PublishApps = $AppNameHash.Get_Item($key)
				}
				else
				{
					$PublishApps = $PublishApps + "#" + $AppNameHash.Get_Item($key)
				}
				$count = $count + 1
			}
			if ($PublishApps -eq $null -or $PublishApps.length -eq 0)
			{
				$PublishedApplications = "-"
			}
			else
			{
				$PublishedApplications = $PublishApps
			}

			$IPAddress = $BrokerMachine.IPAddress
			if ($IPAddress -eq $null -or $IPAddress.length -eq 0)
			{
				$IPAddress = "-"
			}

			Write-Host($BrokerMachine.CatalogName,$BrokerMachine.DesktopGroupName,$BrokerMachine.MachineName,$BrokerMachine.DNSName,
			$IPAddress,$PublishedApplications) -Separator ";"
		}
	}
}

Write-Host "Published Applications"
Write-Host "----------------------"
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		$CommandLineExecutable = $PublishedAppInstance.CommandLineExecutable
		$CommandLineArgs = $PublishedAppInstance.CommandLineArguments
		$AppVCommandLineArgs = "-"
		if ($CommandLineExecutable -ne $null -and ($CommandLineExecutable -ieq "CtxAppVLauncher.exe"))
		{
			$MetadataMap = $PublishedAppInstance.MetadataMap
			if ($MetadataMap -ne $null)
			{
				$AppID = $MetadataMap.get_Item("Citrix_DesktopStudio_AppID")
				$PkgID = $MetadataMap.get_Item("Citrix_DesktopStudio_PackageId")
				if ($AppID -ne $null -and $AppID.length -gt 0 -and $PkgID -ne $null -and $PkgID.length -gt 0)
				{
					$Idx = $PkgID.LastIndexOf("_")
					if ($Idx -gt 0)
					{
						$PkgID = $PkgID.Substring(0,$Idx)
					}
					$AppVCommandLineArgs = "CtxAppVLauncher.exe" + "#" + $AppID + "#" + $PkgID
				}
			}
		}
		if ($CommandLineArgs)
		{
			$CommandLineExecutable = $CommandLineExecutable + " " + $CommandLineArgs
		}

		$NameAndBrowserName = $PublishedAppInstance.Name + "(" + $PublishedAppInstance.BrowserName + ")"
		Write-Host($NameAndBrowserName,$CommandLineExecutable,$AppVCommandLineArgs) -Separator ";"
	}
}

Remove-PSSnapin "Citrix.Common.Commands"
Remove-PSSnapin "Citrix.Host.Admin.V2"
Remove-PSSnapin "Citrix.Broker.Admin.V2"



