$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#REQUIRES -Version 2.0

<#
.DESCRIPTION
    This script is used to get FailoverCluster information
.NOTES  
    File Name      : FailoverCluster.ps1  
    Author         : T.R.Ranjithkumar
    Created        : 22 July 2013
    Prerequisite   : PowerShell V2 over Vista and upper
#>

Import-Module FailoverClusters -EA SilentlyContinue
$isPassive = "no"
if($args.Count -gt 0)
{
    $isPassive = $args[0]
}   
#Write-Host "isPassive" ($isPassive)

# Check the Clustere Service installed or not installed
$hostName = $env:computername
$isCluster = "False"
$clusterService = Get-Service -ComputerName $hostName -Name 'ClusSvc' -EA SilentlyContinue | Select-Object -expand Status 
if ($clusterService -ne $null)
{
    $isCluster = "True"
    Write-Host "Cluster Status"
    Write-Host "-----------------"
    if ($clusterService -eq "Running")
    {
        Write-Host ($isCluster, $clusterService) -Separator "~!~"
    }
    else
    {
        Write-Host ($isCluster, $clusterService) -Separator "~!~"
        return
    }
    Write-Host ""
}
else
{
    Write-Host "Cluster Service not installed"
    return
}
# Suppose the cluster service is not installed, further we are not going to do anything, so return
# Ends here

$No_of_nodes = 0

$Cluster = Get-Cluster
if($null -ne $Cluster)
{
    $ClusterName = $Cluster.Name
    $ClusterDomainName = $Cluster.Domain
}
else 
{
    return
}

#$keyClusetName = (Get-Cluster).Name

# Check the Active node
$isOwner = "False"
try
{
    $ClusterIP = [System.Net.Dns]::GetHostAddresses($ClusterName)
}
catch [Exception]
{
    #write-host $_.Exception.GetType().FullName;
    #write-host $_.Exception.Message;
}
if($null -eq $ClusterIP)
{
    return
}
$CurrentIPs = @();
get-wmiobject win32_networkadapterconfiguration | ? { $_.IPAddress -ne $null } | Sort-Object IPAddress -Unique | % { $CurrentIPs+=$_.IPAddress }
foreach ($tempIP in $CurrentIPs -split(" "))
{
    if ($ClusterIP -like $tempIP)
    {
        $isOwner = "True"
        break
    }
}

Write-Host "ActiveNode" ($ClusterName, $clusterIP, $isOwner, $hostName) -Separator "~!~"
if ($isOwner -like "false")
{
    if ($isPassive -like "no")
    {
        return
    }
}

#Write-Host "Nodes:" ($No_of_nodes)
# Suppose this is not an active node, return the code
# Ends here

# Cluster service is running and this is an active node
# further we are getting cluster information
Write-Host "Cluster Resource"
Write-Host "-----------------"
$ClusterResources = Get-ClusterResource
if($null -ne $ClusterResources)
{
    foreach($ClusterResource in $ClusterResources)
    {
        $ClusterResourceName = $ClusterName+":"+$ClusterResource.Name
        $ClusterResourceState = $ClusterResource.State
        $ClusterResourceCluster = $ClusterResource.Cluster
        $ClusterResourceOwnernode = $ClusterResource.OwnerNode
        $ClusterResourceResourcetype = $ClusterResource.ResourceType
        $ClusterResourceOwnergroup = $ClusterResource.OwnerGroup
        $ClusterResourceIsCoreResource = $ClusterResource.IsCoreResource
        Write-Host ($ClusterResourceName, $ClusterResourceState, $ClusterResourceCluster, $ClusterResourceOwnernode, $ClusterResourceResourcetype, $ClusterResourceOwnergroup,$ClusterResourceIsCoreResource) -Separator "~!~"
    }
}
Write-Host ""

$No_of_groups = 0
Write-Host "Cluster Groups"
Write-Host "---------------"
$ResourceGroups = Get-ClusterGroup
if ($null -ne $ResourceGroups)
{
    foreach($ResourceGroup in $ResourceGroups)
    {
        $ResourceGroupName = $ResourceGroup.Name
        if ($ResourceGroupName -like "Cluster Group" -or $ResourceGroupName -like "Available Storage")
        {
            continue // These two lists are default groups so skip this group
        }
        $ResourceGroupState = $ResourceGroup.State
        $ResourceGroupCluster = $ResourceGroup.Cluster
        $ResourceGroupOwnernode = $ResourceGroup.OwnerNode
        Write-Host ($ResourceGroupName, $ResourceGroupState, $ResourceGroupCluster, $ResourceGroupOwnernode) -Separator "~!~"
        $No_of_groups++
    }
}
Write-Host ""

#Write-Host "Groups:" ($No_of_groups)

Write-Host "Cluster Shared Volume"
Write-Host "----------------------"
$csvs = Get-ClusterSharedVolume
if($null -ne $csvs)
{
    foreach ($csv in $csvs)
    {
        $Name        = $ClusterName+":"+$csv.Name
        $state       = $csv.State
        $ownernode   = $csv.OwnerNode
        #Write-Host "State#####" ($state)
        if ($state -eq "Offline" -or $state -eq "Failed")
        {
            Write-Host ($Name, $state, $ownernode, "-", "0.00", "0.00", "0.00", "0.00") -Separator "~!~"
        }
        else
        {
            $csvinfos = $csv | select -Property Name -ExpandProperty SharedVolumeInfo
            foreach ($csvinfo in $csvinfos)
            {
                $Name        = $ClusterName+":"+$csv.Name
                $state       = $csv.State
                $ownernode   = $csv.OwnerNode
                $Path        = $csvinfo.FriendlyVolumeName
                [float]$Size        = ($csvinfo.Partition.Size / 1GB)
                [float]$FreeSpace   = ($csvinfo.Partition.FreeSpace / 1GB)
                [float]$UsedSpace   = ($csvinfo.Partition.UsedSpace / 1GB)
                if($Size -eq 0)
                {
                    [float]$PercentUsed = 0
                }
                else {
                    [float]$PercentUsed = ($UsedSpace /  $Size * 100)
                }    
                Write-Host ($Name, $state, $ownernode, $Path, $Size, $FreeSpace, $UsedSpace, $PercentUsed) -Separator "~!~"
            }
        }    
    }
}
Write-Host ""

Write-Host "Cluster Network Interface"
Write-Host "--------------------------"
$ClusterInterfaces = Get-ClusterNetworkInterface
if ($null -ne $ClusterInterfaces)
{
    foreach ($ClusterInterface in $ClusterInterfaces)
    {
        [String]$name = $ClusterInterface.Name
        $state = $ClusterInterface.State
        $cluster = $ClusterInterface.Cluster
        $network = $ClusterInterface.Network
        $node = $ClusterName+":"+$ClusterInterface.Node
        $adapter = $ClusterInterface.Adapter
        $adapterId = $ClusterInterface.AdapterId
        #$address = $Item.Address

        $address = ""
        $ipv4Address = ""
        $ipv6Address = ""

        $ipv4Addresses = $ClusterInterface.Ipv4Addresses
        if ($ipv4Addresses -ne $null -and $ipv4Addresses.count -gt 0)
        {
            $count = 0
            foreach ($ipv4 in $ipv4Addresses)
            {
                if ($count -eq 0)
                {
                    $ipv4Address = $ipv4
                }
                else
                {
                    $ipv4Address = $ipv4Address + "," + $ipv4
                }
                $count = $count + 1
            }
        }

        $ipv6Addresses = $ClusterInterface.Ipv6Addresses
        if ($ipv6Addresses -ne $null -and $ipv6Addresses.count -gt 0)
        {
            $count = 0
            foreach ($ipv6 in $ipv6Addresses)
            {
                if ($count -eq 0)
                {
                    $ipv6Address = $ipv6
                }
                else
                {
                    $ipv6Address = $ipv6Address + "," + $ipv6
                }
                $count = $count + 1
            }
        }

        if ($ipv4Address.length -gt 0 -and $ipv6Address.length -gt 0)
        {
            $address = $ipv4Address +","+$ipv6Address
        }
        elseif ($ipv4Address.length -gt 0)
        {
            $address = $ipv4Address
        }
        elseif ($ipv6Address.length -gt 0)
        {
            $address = $ipv6Address
        }
        Write-Host ($node.toLower(), $name, $state, $network, $adapter, $adapterId, $address) -Separator "~!~" 
    }
}
Write-Host ""

Write-Host "Cluster Node"
Write-Host "-------------"
$ClusterNodes = Get-ClusterNode
if ($null -ne $ClusterNodes)
{
    foreach ($ClusterNode in $ClusterNodes)
    {
        $name = $ClusterName+":"+$ClusterNode.Name
        $state = $ClusterNode.State
        $cluster = $ClusterNode.Cluster
        $nodename = $ClusterNode.NodeName
        $csdversion = $ClusterNode.CSDVersion
        Write-Host ($name, $state, $cluster) -Separator "~!~"
        $No_of_nodes++
    }
}
Write-Host ""

Write-Host "Network Cluster"
Write-Host "----------------"
$ClusterNetworks = Get-ClusterNetwork
if ($null -ne $ClusterNetworks)
{
    foreach ($ClusterNetwork in $ClusterNetworks)
    {
        $name = $ClusterName+":"+$ClusterNetwork.Name
        $state = $ClusterNetwork.State
        $cluster = $ClusterNetwork.Cluster
        $address = $ClusterNetwork.Address
        $addressmask = $ClusterNetwork.AddressMask
        Write-Host ($name, $state) -Separator "~!~" 
    }
}
Write-Host ""

$ClusterFQDN = $ClusterName+"."+$ClusterDomainName
Write-Host "Cluster Summary"
Write-Host "----------------"
Write-Host ($ClusterName, $No_of_groups, $No_of_nodes, $ClusterFQDN, $hostName) -Separator "~!~"
Write-Host ""

Write-Host "Available Disk"
Write-Host "----------------"
$AvailableStorageDisks = Get-ClusterGroup "Available Storage" | Get-ClusterResource
if ($null -ne $AvailableStorageDisks)
{
    foreach ($AvailableStorageDisk in $AvailableStorageDisks)
    {
        $name = $ClusterName+":"+$AvailableStorageDisk.Name
        $state = $AvailableStorageDisk.State
        $group = $AvailableStorageDisk.OwnerGroup
        $type = $AvailableStorageDisk.ResourceType
        Write-Host ($name, $state, $group, $type) -Separator "~!~" 
    }
}
Write-Host ""

Write-Host "Groups Resources"
Write-Host "---------------"
$ResourceGroups = Get-ClusterGroup
if ($null -ne $ResourceGroups)
{
    foreach($ResourceGroup in $ResourceGroups)
    {
        $ResourceGroupName = $ResourceGroup.Name
        if ($ResourceGroupName -like "Cluster Group" -or $ResourceGroupName -like "Available Storage")
        {
            continue // These two lists are default groups so skip this group
        } 
        $ResourceGroupState = $ResourceGroup.State
        $ResourceGroupOwnernode = $ResourceGroup.OwnerNode
        $Resources = Get-ClusterGroup $ResourceGroupName | Get-ClusterResource
        $ResourceGroupName = $ClusterName+":"+$ResourceGroup.Name
        foreach ($Resource in $Resources)
        {
            $ResourceName = $ResourceGroupName+":"+$Resource.Name
            $ResourceState = $Resource.State
            $ResourceCluster = $Resource.Cluster
            $ResourceOwnerNode = $Resource.OwnerNode
            $ResourceType = $Resource.ResourceType
            $ResourceOwnerGroup = $Resource.OwnerGroup 
            Write-Host ($ResourceGroupName, $ResourceGroupState, $ResourceGroupOwnernode, $ResourceName, $ResourceState, $ResourceCluster, $ResourceOwnerNode, $ResourceType, $ResourceOwnerGroup) -Separator "~!~"
        }
    }
}
Write-Host ""

Write-Host "Cluster Disks"
Write-Host "--------------"
$resources = Get-WmiObject -namespace root\MSCluster MSCluster_Resource -filter "Type='Physical Disk'"
if ($resources -ne $null)
{
    foreach($res in $resources)
    {
        $Name = $res.Name
        $Status = $res.State
        if($Status -eq 3 -or $Status -eq 4 -or $Status -eq 130)
        {
        
            Write-Host "Name           : " $ClusterName":"$Name
            Write-Host "Status         : " $Status
            Write-Host
        }
        else
        {
            $disks = $res.GetRelated("MSCluster_Disk")
            $disks | foreach {
            $_.GetRelated("MSCluster_DiskPartition") |
            select @{N="Name"; E={$($ClusterName)+":"+$res.Name}}, @{N="Status"; E={$res.State}}, @{N="IsSharedVolume"; E={$res.IsClusterSharedVolume}},Path, VolumeLabel, TotalSize, FreeSpace 
            }
        }
    }
}


