﻿# This Script giving details of Application Groups
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$Global:ErrorActionPreference="SilentlyContinue"
Add-PSSnapin "Citrix.Host.Admin.V2"
Add-PSSnapin "Citrix.Broker.Admin.V2"

<#
Function Get-BrokerMachinesPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "MachineName")
	$desktops = @()
	$desktops += Get-BrokerMachine -Filter {RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'} -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($desktops.count / $total * 100, 1)
		#Write-Host  "Getting Broker Machines. Retrieved $($desktops.count) of $total machines completed in $remainingPerc%"
		$last = $desktops[-1].MachineName
		$desktops += Get-BrokerMachine -Filter {RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'} -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { MachineName -gt $last } @argumentList
	}
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($applications.count / $total * 100, 1)
		#Write-Host  "Getting Broker Applications. Retrieved $($applications.count) of $total applications completed in $remainingPerc%"
		$last = $applications[-1].Uid
		$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $applications
}
#>

Function Get-BrokerMachinesPaged
{
	param (
	$DesktopGroupUid,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$desktops = @()
	$LastUid = 0;
	do
	{
		$NewDesktops = @(Get-BrokerMachine -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid -and RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'} @argumentList)
		if ($NewDesktops.Count -gt 0)
		{
			$desktops += $NewDesktops
			$LastUid = $desktops[-1].Uid
		}
	} while ($NewDesktops.Count -ge $MaxRecordCount)
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$LastUid = 0;
	do
	{
		$NewApplications = @(Get-BrokerApplication -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewApplications.Count -gt 0)
		{
			$applications += $NewApplications
			$LastUid = $applications[-1].Uid
		}
	} while ($NewApplications.Count -ge $MaxRecordCount)
	return $applications
}

Write-Host "Application Groups"
Write-Host "------------------"

$BrokerApplicationGroups = Get-BrokerApplicationGroup -MaxRecordCount 2147483647
if ($BrokerApplicationGroups -ne $null)
{
	foreach ($BrokerApplicationGroup in $BrokerApplicationGroups)
	{
		$AssociatedDesktopGroupUids = $BrokerApplicationGroup.AssociatedDesktopGroupUids
		$DesktopGroupUids = "-"
		if ($AssociatedDesktopGroupUids -ne $null -and $AssociatedDesktopGroupUids.length -gt 0)
		{
			$UidCount = 0
			foreach ($AssociatedDesktopGroupUid in $AssociatedDesktopGroupUids)
			{
				if ($UidCount -eq 0)
				{
					$DesktopGroupUids = "" + $AssociatedDesktopGroupUid
				}
				else
				{
					$DesktopGroupUids = $DesktopGroupUids + "#" + $AssociatedDesktopGroupUid
				}
				$UidCount = $UidCount + 1
		}
	}

	$AssociatedUserNames = $BrokerApplicationGroup.AssociatedUserNames
	$UserNames = "-"
	if ($AssociatedUserNames -ne $null -and $AssociatedUserNames.length -gt 0)
	{
		$UserCount = 0
		foreach ($AssociatedUserName in $AssociatedUserNames)
		{
			if ($UserCount -eq 0)
			{
				$UserNames = $AssociatedUserName
			}
			else
			{
				$UserNames = $UserNames + "#" + $AssociatedUserName
			}
			$UserCount = $UserCount + 1
		}
	}

	$Tags = $BrokerApplicationGroup.Tags
	$BrokerApplicationTags = "-"
	if ($Tags -ne $null -and $Tags.length -gt 0)
	{
		$TagCount = 0
		foreach ($Tag in $Tags)
		{
			if ($TagCount -eq 0)
			{
				$BrokerApplicationTags = $Tag
			}
			else
			{
				$BrokerApplicationTags = $BrokerApplicationTags + "#" + $Tag
			}
			$TagCount = $TagCount + 1
		}
	}

	$RestrictToTag = $BrokerApplicationGroup.RestrictToTag
	if ($RestrictToTag -eq $null -or $RestrictToTag.length -eq 0)
	{
		$RestrictToTag = "-"
	}

	Write-Host ($BrokerApplicationGroup.Uid,$BrokerApplicationGroup.Name,$BrokerApplicationGroup.Enabled,$BrokerApplicationGroup.SessionSharingEnabled,$BrokerApplicationGroup.SingleAppPerSession,$BrokerApplicationGroup.UserFilterEnabled,
		$BrokerApplicationGroup.TotalApplications,$BrokerApplicationGroup.TotalMachines,$BrokerApplicationGroup.TotalMachinesWithTagRestriction,$RestrictToTag,
		$DesktopGroupUids,$BrokerApplicationTags,$UserNames) -Separator ";"
	}
}

Write-Host "Broker Machines"
Write-Host "---------------"
#$BrokerMachines = Get-BrokerMachine -Filter {RegistrationState -eq 'Registered' -and DeliveryType -ne 'DesktopsOnly'} -MaxRecordCount 2147483647
$BrokerMachines = Get-BrokerMachinesPaged
if ($BrokerMachines -ne $null)
{
	foreach ($BrokerMachine in $BrokerMachines)
	{
		$PublishedApps = "-"
		$PublishedApplications = $BrokerMachine.PublishedApplications
		if ($PublishedApplications -ne $null -and $PublishedApplications.length -gt 0)
		{
			$PublishedAppsCount = 0
			foreach ($PublishedApplication in $PublishedApplications)
			{
				if ($PublishedAppsCount -eq 0)
				{
					$PublishedApps = $PublishedApplication
				}
				else
				{
					$PublishedApps = $PublishedApps + "#" + $PublishedApplication
				}
				$PublishedAppsCount = $PublishedAppsCount + 1
			}
		}
		$UserNames = "-"
		$AssociatedUserNames = $BrokerMachine.AssociatedUserNames
		if ($AssociatedUserNames -ne $null -and $AssociatedUserNames.length -gt 0)
		{
			$UserCount = 0
			foreach ($AssociatedUserName in $AssociatedUserNames)
			{
				if ($UserCount -eq 0)
				{
					$UserNames = $AssociatedUserName
				}
				else
				{
					$UserNames = $UserNames + "#" + $AssociatedUserName
				}
				$UserCount = $UserCount + 1
			}
		}

		$Tags = $BrokerMachine.Tags
		$BrokerMachineTags = "-"
		if ($Tags -ne $null -and $Tags.length -gt 0)
		{
			$TagCount = 0
			foreach ($Tag in $Tags)
			{
				if ($TagCount -eq 0)
				{
					$BrokerMachineTags = $Tag
				}
				else
				{
					$BrokerMachineTags = $BrokerMachineTags + "#" + $Tag
				}
				$TagCount = $TagCount + 1
			}
		}
		Write-Host($BrokerMachine.MachineName,$BrokerMachine.DesktopGroupUid,$BrokerMachine.DesktopGroupName,$PublishedApps,$UserNames,$BrokerMachineTags) -Separator ";"
	}
}

Write-Host "Published Applications"
Write-Host "----------------------"
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
$PublishedAppInstances = Get-BrokerApplicationsPaged
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		$AssociatedApplicationGroupUids = $PublishedAppInstance.AssociatedApplicationGroupUids
		$ApplicationGroupUids = "-"
		if ($AssociatedApplicationGroupUids -ne $null -and $AssociatedApplicationGroupUids.length -gt 0)
		{
			$AGroupUidCount = 0
			foreach ($AssociatedApplicationGroupUid in $AssociatedApplicationGroupUids)
			{
				if ($AGroupUidCount -eq 0)
				{
					$ApplicationGroupUids = "" + $AssociatedApplicationGroupUid
				}
				else
				{
					$ApplicationGroupUids = $ApplicationGroupUids + "#" + $AssociatedApplicationGroupUid
				}
				$AGroupUidCount = $AGroupUidCount + 1
			}
		}

		$AssociatedDesktopGroupUids = $PublishedAppInstance.AssociatedDesktopGroupUids
		$DesktopGroupUids = "-"
		if ($AssociatedDesktopGroupUids -ne $null -and $AssociatedDesktopGroupUids.length -gt 0)
		{
			$DGroupUidCount = 0
			foreach ($AssociatedDesktopGroupUid in $AssociatedDesktopGroupUids)
			{
				if ($DGroupUidCount -eq 0)
				{
					$DesktopGroupUids = "" + $AssociatedDesktopGroupUid
				}
				else
				{
					$DesktopGroupUids = $DesktopGroupUids + "#" + $AssociatedDesktopGroupUid
				}
				$DGroupUidCount = $DGroupUidCount + 1
			}
		}

		$Tags = $PublishedAppInstance.Tags
		$PublishedAppInstanceTags = "-"
		if ($Tags -ne $null -and $Tags.length -gt 0)
		{
			$TagCount = 0
			foreach ($Tag in $Tags)
			{
				if ($TagCount -eq 0)
				{
					$PublishedAppInstanceTags = $Tag
				}
				else
				{
					$PublishedAppInstanceTags = $PublishedAppInstanceTags + "#" + $Tag
				}
				$TagCount = $TagCount + 1
			}
		}
		Write-Host($PublishedAppInstance.BrowserName,$PublishedAppInstance.Name,$ApplicationGroupUids,$DesktopGroupUids,$PublishedAppInstanceTags) -Separator ";"
	}
}

Remove-PSSnapin "Citrix.Host.Admin.V2"
Remove-PSSnapin "Citrix.Broker.Admin.V2"


