# This Script will give all details of Citrix Cloud Discovery

$Global:ErrorActionPreference="SilentlyContinue"
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

if($args.Count -eq 0)
{
	"Syntax: CitrixCloudDiscovery.ps1 <CustomerId> <Path_of_secureclientfile> [Optional]<eG_Install_Dir> [Optional]<ProxySettings>"
	return
}

function GetBearerToken
{
	param (
	[Parameter(Mandatory=$true)]
	[string] $clientId,
	[Parameter(Mandatory=$true)]
	[string] $clientSecret
	)
  
	$postHeaders = @{"Content-Type"="application/json"}
	$body = @{
	"ClientId"=$clientId;
	"ClientSecret"=$clientSecret
	}
	$trustUrl = "https://trust.citrixworkspacesapi.net/root/tokens/clients"

	$response = Invoke-RestMethod -Uri $trustUrl -Method POST -Body (ConvertTo-Json $body) -Headers $postHeaders
	$bearerToken = $response.token

	return $bearerToken;
}

Add-PSSnapin Citrix*
$ProxyHost = 'none'
$ProxyUser = 'none'
$ProxyPwd = 'none'

if($args.Length -eq 2)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
}
elseif($args.Length -eq 4)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
	$eGInstallDir = $args[2]
	# The format is https://ip:port#user#password
	$ProxyArr = $args[3].ToString().Split('#') 
	if($ProxyArr.Length -eq 3)
	{
		$ProxyHost = $ProxyArr[0]
		$ProxyUser = $ProxyArr[1]
		$ProxyPwd = $ProxyArr[2]
		$OEPath = $eGInstallDir + "\lib\OE.exe"
		$ProxyPwd = & $OEPath $ProxyPwd
	}
}

if($ProxyHost -ne 'none')
{
	[system.net.webrequest]::defaultwebproxy = new-object system.net.webproxy($ProxyHost)
	if($ProxyUser -ne 'none' -and $ProxyPwd -ne 'none')
	{
		$proxyCred = New-Object -TypeName System.Management.Automation.PSCredential -argumentlist $ProxyUser, $(convertto-securestring $ProxyPwd -asplaintext -force)
		[system.net.webrequest]::defaultwebproxy.credentials = $proxyCred   #[System.Net.CredentialCache]::DefaultNetworkCredentials
		[system.net.webrequest]::defaultwebproxy.BypassProxyOnLocal = $true
	}
}

Set-XDCredentials -CustomerId $CustomerId -SecureClientFile $SecureClientFile -ProfileType CloudApi -StoreAs "default"
Get-XDCredentials -ProfileName "default" | out-null

<#
Function Get-BrokerMachinesPaged
{
	param (
	$MachineName,
	$AgentVersion,
	$AllocationType,
	$ApplicationInUse,
	$AssignedClientName,
	$AssignedIPAddress,
	$AssociatedUserFullName,
	$AssociatedUserName,
	$AssociatedUserSID,
	$AssociatedUserUPN,
	$BrowserName,
	$CatalogName,
	$CatalogUid,
	$CatalogUUID,
	$ColorDepth,
	$ControllerDNSName,
	$DeliveryType,
	$Description,
	$DesktopCondition,
	$DesktopGroupName,
	$DesktopGroupUid,
	$DesktopGroupUUID,
	$DesktopKind,
	$DesktopUid,
	$DNSName,
	$FaultState,
	$FunctionalLevel,
	$HostedMachineId,
	$HostedMachineName,
	$HostingServerName,
	$HypervisorConnectionName,
	$HypervisorConnectionUid,
	$HypHypervisorConnectionUid,
	$IconUid,
	$ImageOutOfDate,
	$InMaintenanceMode,
	$IPAddress,
	$IsAssigned,
	$IsPhysical,
	$LastConnectionFailure,
	$LastConnectionTime,
	$LastConnectionUser,
	$LastDeregistrationReason,
	$LastDeregistrationTime,
	$LastErrorReason,
	$LastErrorTime,
	$LastHostingUpdateTime,
	$LastPvdErrorReason,
	$LastPvdErrorTime,
	$LoadIndex,
	$MachineInternalState,
	$Metadata,
	$OSType,
	$OSVersion,
	$PersistUserChanges,
	$PowerActionPending,
	$PowerState,
	$ProvisioningType,
	$PublishedApplication,
	$PublishedName,
	$PvdEstimatedCompletionTime,
	$PvdPercentDone,
	$PvdStage,
	$PvdUpdateStartTime,
	$RegistrationState,
	$ScheduledReboot,
	$SecureIcaRequired,
	$SessionAutonomouslyBrokered,
	$SessionClientAddress,
	$SessionClientName,
	$SessionClientVersion,
	$SessionConnectedViaHostName,
	$SessionConnectedViaIP,
	$SessionCount,
	$SessionDeviceId,
	$SessionHardwareId,
	$SessionHidden,
	$SessionKey,
	$SessionLaunchedViaHostName,
	$SessionLaunchedViaIP,
	$SessionProtocol,
	$SessionSecureIcaActive,
	$SessionsEstablished,
	$SessionSmartAccessTag,
	$SessionsPending,
	$SessionStartTime,
	$SessionState,
	$SessionStateChangeTime,
	$SessionSupport,
	$SessionType,
	$SessionUid,
	$SessionUserName,
	$SessionUserSID,
	$SID,
	$SummaryState,
	$SupportedPowerActions,
	$Tag,
	$UUID,
	$VMToolsState,
	$WillShutdownAfterUse,
	$WindowsConnectionSetting,
	$AssignedUserSID,
	$AdminAddress,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "MachineName")
	$desktops = @()
	$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt $MaxRecordCount)
	{
	$remainingPerc= [math]::round($desktops.count / $total * 100, 1)
	#Write-Host  "Getting Broker Machines. Retrieved $($desktops.count) of $total machines completed in $remainingPerc%"
	$last = $desktops[-1].MachineName
	$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { MachineName -gt $last } @argumentList
	}

	return $desktops
}
#>

Function Get-BrokerMachinesPaged
{
	param (
	$MachineName,
	$AgentVersion,
	$AllocationType,
	$ApplicationInUse,
	$AssignedClientName,
	$AssignedIPAddress,
	$AssociatedUserFullName,
	$AssociatedUserName,
	$AssociatedUserSID,
	$AssociatedUserUPN,
	$BrowserName,
	$CatalogName,
	$CatalogUid,
	$CatalogUUID,
	$ColorDepth,
	$ControllerDNSName,
	$DeliveryType,
	$Description,
	$DesktopCondition,
	$DesktopGroupName,
	$DesktopGroupUid,
	$DesktopGroupUUID,
	$DesktopKind,
	$DesktopUid,
	$DNSName,
	$FaultState,
	$FunctionalLevel,
	$HostedMachineId,
	$HostedMachineName,
	$HostingServerName,
	$HypervisorConnectionName,
	$HypervisorConnectionUid,
	$HypHypervisorConnectionUid,
	$IconUid,
	$ImageOutOfDate,
	$InMaintenanceMode,
	$IPAddress,
	$IsAssigned,
	$IsPhysical,
	$LastConnectionFailure,
	$LastConnectionTime,
	$LastConnectionUser,
	$LastDeregistrationReason,
	$LastDeregistrationTime,
	$LastErrorReason,
	$LastErrorTime,
	$LastHostingUpdateTime,
	$LastPvdErrorReason,
	$LastPvdErrorTime,
	$LoadIndex,
	$MachineInternalState,
	$Metadata,
	$OSType,
	$OSVersion,
	$PersistUserChanges,
	$PowerActionPending,
	$PowerState,
	$ProvisioningType,
	$PublishedApplication,
	$PublishedName,
	$PvdEstimatedCompletionTime,
	$PvdPercentDone,
	$PvdStage,
	$PvdUpdateStartTime,
	$RegistrationState,
	$ScheduledReboot,
	$SecureIcaRequired,
	$SessionAutonomouslyBrokered,
	$SessionClientAddress,
	$SessionClientName,
	$SessionClientVersion,
	$SessionConnectedViaHostName,
	$SessionConnectedViaIP,
	$SessionCount,
	$SessionDeviceId,
	$SessionHardwareId,
	$SessionHidden,
	$SessionKey,
	$SessionLaunchedViaHostName,
	$SessionLaunchedViaIP,
	$SessionProtocol,
	$SessionSecureIcaActive,
	$SessionsEstablished,
	$SessionSmartAccessTag,
	$SessionsPending,
	$SessionStartTime,
	$SessionState,
	$SessionStateChangeTime,
	$SessionSupport,
	$SessionType,
	$SessionUid,
	$SessionUserName,
	$SessionUserSID,
	$SID,
	$SummaryState,
	$SupportedPowerActions,
	$Tag,
	$UUID,
	$VMToolsState,
	$WillShutdownAfterUse,
	$WindowsConnectionSetting,
	$AssignedUserSID,
	$AdminAddress,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$desktops = @()
	$LastUid = 0;
	do
	{
		$NewDesktops = @(Get-BrokerMachine -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewDesktops.Count -gt 0)
		{
			$desktops += $NewDesktops
			$LastUid = $desktops[-1].Uid
		}
	} while ($NewDesktops.Count -ge $MaxRecordCount)
	return $desktops
}

Write-Host "[Connectors]"
$EdgeServers = Get-ConfigEdgeServer
if ($EdgeServers -ne $null)
{
	foreach ($EdgeServer in $EdgeServers)
	{
		$ips = @(([net.dns]::GetHostEntry($EdgeServer.MachineAddress)).AddressList)
		$tips = ""
		$count = 0
		if($? -and $ips)
		{
			foreach($ip in $ips)
			{
				if($ip.AddressFamily -eq "InterNetwork")
				{
					if($count -eq 0)
					{
						$tips = $ip.IPAddressToString
					}
					else
					{
						$tips = $tips + "," + $ip.IPAddressToString
					}
					$count = $count + 1
				}	
			}
			if ($tips.length -eq 0)
			{
				$tips = "-"
			}
			Write-Host $EdgeServer.MachineAddress"=" $tips
		}
		else
		{
			Write-Host $EdgeServer.MachineAddress"= -"
		}
		$ips = @{}
		$count = 0
		$tips = ""
	}
}
$ips = @{}
$count = 0
$tips = ""

Write-Host "[Controllers]"
$Controllers = Get-BrokerController
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		Write-Host $Controller.MachineName "=" $Controller.DNSName
	}
}

Write-Host "[Hypervisor_Details]"
$BrokerHypConns = Get-BrokerHypervisorConnection -State On
$HypConnsToBroker = @{}
if($BrokerHypConns -ne $null)
{
	foreach($BrokerHypConn in $BrokerHypConns)
	{
		$Value = $BrokerHypConn.HypHypervisorConnectionUid
		$Key = $BrokerHypConn.Uid
		$HypConnsToBroker[$Key] = $Value
	}
}

$HypConns = @{}
$HypConnPath = "XDHyp:\Connections\"
$HypConnections = Get-ChildItem $HypConnPath
if ($HypConnections -ne $null)
{
	foreach ($HypConn in $HypConnections)
	{
		$Key1 = $HypConn.HypervisorConnectionUid
		$Value1 = $HypConn.ConnectionType.toString() + ";" + $HypConn.HypervisorAddress[0].toString()
		$HypConns[$Key1] = $Value1
	}
}
if($BrokerToHypIDs -ne $null)
{
	$Keys = $BrokerToHypIDs.Keys
	foreach($key2 in $keys)
	{
		$Values = $BrokerToHypIDs[$key2]
		if($Values.Count -ge 1)
		{
			foreach($Value in $Values)
			{
				$hypuobj = Get-BrokerHypervisorConnection -Uid $Value
				if($hypuobj)
				{
					$hypuid = $hypuobj.HypHypervisorConnectionUid
					$output = $HypConns[$hypuid]
					$outputStr = $key2.toString() + "=" + $output
					Write-Host $outputStr
				}
			}
		}
	}
}

Write-Host "[Hypervisor_Type]"
#Support XenServer only
$HypConnPath = "XDHyp:\Connections\"
$HypConnections = Get-ChildItem $HypConnPath
if ($HypConnections -ne $null)
{
	foreach ($HypConn in $HypConnections)
	{
		$ConnUid = $HypConn.HypervisorConnectionUid
		$Value1 = $HypConn.ConnectionType.toString() + ";" + $HypConn.HypervisorAddress[0].toString()
		if ($HypConn.ConnectionType.toString() -ieq "XenServer")
		{
			#$machines = Get-BrokerMachine -SessionSupport 'MultiSession' -HypHypervisorConnectionUid $ConnUid
			$machines = Get-BrokerMachinesPaged -SessionSupport 'MultiSession' -HypHypervisorConnectionUid $ConnUid
			if ($machines -ne $null)
			{
				Write-Host $Value1"=Server"
			}
			else
			{
				Write-Host $Value1"=VDI"
			}
		}
		elseif ($HypConn.ConnectionType.toString() -ieq "vcenter")
		{
			Write-Host $Value1"=vCenter"
		}
	}
}

Write-Host "[ActiveDirectory_server]"
$d =  (gwmi win32_computersystem).domain
$ips = @(([net.dns]::GetHostEntry($d)).AddressList)
if($? -and $ips)
{
	foreach($ip in $ips)
	{
		if($ip.AddressFamily -eq "InterNetwork")
		{
			if($count -eq 0)
			{
				$tips = $ip.IPAddressToString
			}
			else
			{
				$tips = $tips +","+ $ip.IPAddressToString
			}
			$count = $count + 1
		}
	}
	Write-Host $d "=" $tips
}

$ips = ""
Write-Host "[XenApp]"
#$XenApps = Get-BrokerMachine -SessionSupport MultiSession -MaxRecordCount 2147483647
$BrokerDesktopGroup = Get-BrokerDesktopGroup -SessionSupport MultiSession -MaxRecordCount 999
if ($BrokerDesktopGroup  -ne $null)
{
	foreach ($DesktopGroup in $BrokerDesktopGroup)
	{
		$XenApps = Get-BrokerMachinesPaged -SessionSupport MultiSession -DesktopGroupUid $DesktopGroup.Uid
		if($XenApps -ne $null)
		{
			foreach($XenApp in $XenApps)
			{
				$ips = $XenApp.IPAddress
				if($? -and $ips)
				{
					Write-Host $XenApp.ControllerDNSName  "^^"  $XenApp.DNSName "^^" $XenApp.DesktopGroupName "="   $ips
				}
			}
		}
	}
}

Write-Host "[PVS]"
$PVSIPAddresses = @()
$Catalogs = Get-BrokerCatalog -MaxRecordCount 999
if($Catalogs -ne $null)
{
	foreach($Catalog in $Catalogs)
	{
		if ($Catalog.PvsAddress.length -gt 0)
		{
			$PvsAddress = $Catalog.PvsAddress
			if($PVSIPAddresses -Contains $PvsAddress)
			{
			}
			else
			{
				$PVSIPAddresses += $PvsAddress
			}
		}
	}
	Write-Host $PVSIPAddresses
}

Write-Host "[XD_Site]"
$Site = Get-BrokerSite
if ($Site -ne $null)
{

	$SiteName = $Site.Name
	$objIPProperties = [System.Net.NetworkInformation.IPGlobalProperties]::GetIPGlobalProperties()
	$fqdn = "{0}.{1}" -f $env:COMPUTERNAME, $objIPProperties.DomainName
	$ips = ""
	$ips = @(([net.dns]::GetHostEntry($env:COMPUTERNAME)).AddressList)
	$tips = ""
	$count = 0
	if($? -and $ips)
	{
		foreach($ip in $ips)
		{
			if($ip.AddressFamily -eq "InterNetwork")
			{
				if($count -eq 0)
				{
					$tips = $ip.IPAddressToString +":443"
				}
				else
				{
					$tips = $tips + "," + $ip.IPAddressToString+":443"
				}
				$count = $count + 1
			}
		}
		Write-Host $fqdn "^^" $SiteName "=" $tips
	}
}

Remove-PSSnapin citrix*



# SIG # Begin signature block
# MIIlTAYJKoZIhvcNAQcCoIIlPTCCJTkCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQU0gWhbjIUy1lcP5USZeLM34XD
# zEOggh5yMIIFMDCCBBigAwIBAgIQBAkYG1/Vu2Z1U0O1b5VQCDANBgkqhkiG9w0B
# AQsFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMTMxMDIyMTIwMDAwWhcNMjgxMDIyMTIwMDAwWjByMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQg
# Q29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
# +NOzHH8OEa9ndwfTCzFJGc/Q+0WZsTrbRPV/5aid2zLXcep2nQUut4/6kkPApfmJ
# 1DcZ17aq8JyGpdglrA55KDp+6dFn08b7KSfH03sjlOSRI5aQd4L5oYQjZhJUM1B0
# sSgmuyRpwsJS8hRniolF1C2ho+mILCCVrhxKhwjfDPXiTWAYvqrEsq5wMWYzcT6s
# cKKrzn/pfMuSoeU7MRzP6vIK5Fe7SrXpdOYr/mzLfnQ5Ng2Q7+S1TqSp6moKq4Tz
# rGdOtcT3jNEgJSPrCGQ+UpbB8g8S9MWOD8Gi6CxR93O8vYWxYoNzQYIH5DiLanMg
# 0A9kczyen6Yzqf0Z3yWT0QIDAQABo4IBzTCCAckwEgYDVR0TAQH/BAgwBgEB/wIB
# ADAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMweQYIKwYBBQUH
# AQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYI
# KwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcnQwgYEGA1UdHwR6MHgwOqA4oDaGNGh0dHA6Ly9jcmw0
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwTwYDVR0gBEgwRjA4BgpghkgBhv1sAAIEMCowKAYIKwYBBQUHAgEWHGh0
# dHBzOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwCgYIYIZIAYb9bAMwHQYDVR0OBBYE
# FFrEuXsqCqOl6nEDwGD5LfZldQ5YMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6en
# IZ3zbcgPMA0GCSqGSIb3DQEBCwUAA4IBAQA+7A1aJLPzItEVyCx8JSl2qB1dHC06
# GsTvMGHXfgtg/cM9D8Svi/3vKt8gVTew4fbRknUPUbRupY5a4l4kgU4QpO4/cY5j
# DhNLrddfRHnzNhQGivecRk5c/5CxGwcOkRX7uq+1UcKNJK4kxscnKqEpKBo6cSgC
# PC6Ro8AlEeKcFEehemhor5unXCBc2XGxDI+7qPjFEmifz0DLQESlE/DmZAwlCEIy
# sjaKJAL+L3J+HNdJRZboWR3p+nRka7LrZkPas7CM1ekN3fYBIM6ZMWM9CBoYs4Gb
# T8aTEAb8B4H6i9r5gkn3Ym6hU/oSlBiFLpKR6mhsRDKyZqHnGKSaZFHvMIIFjTCC
# BHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0Ew
# HhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEV
# MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29t
# MSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZ
# wuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4V
# pX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAd
# YyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3
# T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjU
# N6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNda
# SaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtm
# mnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyV
# w4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3
# AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYi
# Cd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmp
# sh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7Nfj
# gtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNt
# yA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYY
# aHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2Fj
# ZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUG
# A1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2Vy
# dEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3
# DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+Ica
# aVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096ww
# epqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcD
# x4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsg
# jTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37Y
# OtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMIIGMzCCBRugAwIBAgIQCJp0nrgt
# w+wn6mXq2/g1MTANBgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UE
# ChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYD
# VQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQgQ29kZSBTaWduaW5nIENBMB4X
# DTIxMDUyNTAwMDAwMFoXDTI0MDUyOTIzNTk1OVowcTELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCk5ldyBKZXJzZXkxDzANBgNVBAcTBklzZWxpbjEdMBsGA1UEChMUZUcg
# SW5ub3ZhdGlvbnMsIEluYy4xHTAbBgNVBAMTFGVHIElubm92YXRpb25zLCBJbmMu
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAxGUJ0prjbfxcmgZGjd7J
# DxOs7sySHvaQ3wGR2KbD3aZageSE+kG0tKkEsQ4na8bnNViY/zF4Pbo0ZkYtMzJI
# k0AZXDTMvXr/mEvmKxCbiJFTCpKkmZ4sa5BZfO7igIvcNSa0zII2a5jiQJFy85j+
# f9I4EgTo9OcdVeINXhgQ2xZ6TjEK+pbhqmVXvr8DB26JclOaed0L5Vs7+CbGzZuK
# ifgxL2i8d5FDzkhZSZfdCtGtEvE5pLesXSWfpzUddvCNRisIPGB7mg+Rln2XvUZy
# yaRURtlx11pYfJ/KNYAkUCL23rnh38/maxtSvdeioYBj4xLyt7poQBG+b0uzCjgR
# ADe3/k0NtiVwJIo0ZEsUyDANp7JIIjMePzsOMYzfMPyi8clAwfBYc2XhSuUcX3yK
# f4kpRFmClYfK5LstGQRNbONjpsCBTEgZuwEfkdOU0rmaMywZFVLHEpmpYZePz7M3
# VhN+aV56kR/efo8eD81E5VDQqmJiGWKq/s9jhbkYH7g2uhL2MXb/uRshVDSOkFRv
# IID/l+yDJogDBpae3x2ov5YvjY8Zo8RF0tXvyS0rfsDvANZw7mptzHn0Blmw5lug
# 3F2CVJpNbzb1HmAH6k0yFHmWYCoUANbBB0YR/k4JvgY2byttF7cQPJyl7UI9Cu4K
# TAg7ROmGkwdLhmXoPCBco0ECAwEAAaOCAcQwggHAMB8GA1UdIwQYMBaAFFrEuXsq
# CqOl6nEDwGD5LfZldQ5YMB0GA1UdDgQWBBT5yESaz9gTSS8aBNa1r34MMA42SjAO
# BgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYDVR0fBHAwbjA1
# oDOgMYYvaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1n
# MS5jcmwwNaAzoDGGL2h0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFzc3Vy
# ZWQtY3MtZzEuY3JsMEsGA1UdIAREMEIwNgYJYIZIAYb9bAMBMCkwJwYIKwYBBQUH
# AgEWG2h0dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAIBgZngQwBBAEwgYQGCCsG
# AQUFBwEBBHgwdjAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# ME4GCCsGAQUFBzAChkJodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRTSEEyQXNzdXJlZElEQ29kZVNpZ25pbmdDQS5jcnQwDAYDVR0TAQH/BAIwADAN
# BgkqhkiG9w0BAQsFAAOCAQEAlAjncFPvLKywT/4DECG7bHbtiVl+uumfRj2YrFuC
# hsgkv1PmQpgfypxC3g+ErV9yRV8+XXSCwOaKJ3v6RLnfphLhEwYc2+0Qs/Nlib5N
# AxDGuIczAIeXOc5kRRpvFsQ2XSNtM7XL1tLDm6p/VG7BoUAyqRXsMPdWbTkN/9nd
# CmGSsqcxjG2ud8O6Vhte9J5LaHBVk3lIZAMtH6ACdo5QTrM49nbIU8QGuRYNXZKR
# LAUu6IgD6WJKMVfZXWlyfD8dZ2r3ej6Q1uAO/Nbtd397T+BVQrDWMOG8+GeRiJwo
# evxbIWh0SenZOUrAq9vTJaSvFMSvctkJm/oxLUcUdEGS3zCCBq4wggSWoAMCAQIC
# EAc2N7ckVHzYR6z9KGYqXlswDQYJKoZIhvcNAQELBQAwYjELMAkGA1UEBhMCVVMx
# FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNv
# bTEhMB8GA1UEAxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIyMDMyMzAw
# MDAwMFoXDTM3MDMyMjIzNTk1OVowYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRp
# Z2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQw
# OTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQADggIP
# ADCCAgoCggIBAMaGNQZJs8E9cklRVcclA8TykTepl1Gh1tKD0Z5Mom2gsMyD+Vr2
# EaFEFUJfpIjzaPp985yJC3+dH54PMx9QEwsmc5Zt+FeoAn39Q7SE2hHxc7Gz7iuA
# hIoiGN/r2j3EF3+rGSs+QtxnjupRPfDWVtTnKC3r07G1decfBmWNlCnT2exp39mQ
# h0YAe9tEQYncfGpXevA3eZ9drMvohGS0UvJ2R/dhgxndX7RUCyFobjchu0CsX7Le
# Sn3O9TkSZ+8OpWNs5KbFHc02DVzV5huowWR0QKfAcsW6Th+xtVhNef7Xj3OTrCw5
# 4qVI1vCwMROpVymWJy71h6aPTnYVVSZwmCZ/oBpHIEPjQ2OAe3VuJyWQmDo4EbP2
# 9p7mO1vsgd4iFNmCKseSv6De4z6ic/rnH1pslPJSlRErWHRAKKtzQ87fSqEcazjF
# KfPKqpZzQmiftkaznTqj1QPgv/CiPMpC3BhIfxQ0z9JMq++bPf4OuGQq+nUoJEHt
# Qr8FnGZJUlD0UfM2SU2LINIsVzV5K6jzRWC8I41Y99xh3pP+OcD5sjClTNfpmEpY
# PtMDiP6zj9NeS3YSUZPJjAw7W4oiqMEmCPkUEBIDfV8ju2TjY+Cm4T72wnSyPx4J
# duyrXUZ14mCjWAkBKAAOhFTuzuldyF4wEr1GnrXTdrnSDmuZDNIztM2xAgMBAAGj
# ggFdMIIBWTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBS6FtltTYUvcyl2
# mi91jGogj57IbzAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNV
# HQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwgwdwYIKwYBBQUHAQEEazBp
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUH
# MAKGNWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRS
# b290RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMCAGA1UdIAQZMBcwCAYGZ4EM
# AQQCMAsGCWCGSAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAfVmOwJO2b5ipRCIB
# fmbW2CFC4bAYLhBNE88wU86/GPvHUF3iSyn7cIoNqilp/GnBzx0H6T5gyNgL5Vxb
# 122H+oQgJTQxZ822EpZvxFBMYh0MCIKoFr2pVs8Vc40BIiXOlWk/R3f7cnQU1/+r
# T4osequFzUNf7WC2qk+RZp4snuCKrOX9jLxkJodskr2dfNBwCnzvqLx1T7pa96kQ
# sl3p/yhUifDVinF2ZdrM8HKjI/rAJ4JErpknG6skHibBt94q6/aesXmZgaNWhqsK
# RcnfxI2g55j7+6adcq/Ex8HBanHZxhOACcS2n82HhyS7T6NJuXdmkfFynOlLAlKn
# N36TU6w7HQhJD5TNOXrd/yVjmScsPT9rp/Fmw0HNT7ZAmyEhQNC3EyTN3B14OuSe
# reU0cZLXJmvkOHOrpgFPvT87eK1MrfvElXvtCl8zOYdBeHo46Zzh3SP9HSjTx/no
# 8Zhf+yvYfvJGnXUsHicsJttvFXseGYs2uJPU5vIXmVnKcPA3v5gA3yAWTyf7YGcW
# oWa63VXAOimGsJigK+2VQbc61RWYMbRiCQ8KvYHZE/6/pNHzV9m8BPqC3jLfBInw
# AM1dwvnQI38AC+R2AibZ8GV2QqYphwlHK+Z/GqSFD/yYlvZVVCsfgPrA8g4r5db7
# qS9EFUrnEw4d2zc4GqEr9u3WfPwwggbAMIIEqKADAgECAhAMTWlyS5T6PCpKPSkH
# gD1aMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdp
# Q2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2
# IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcNMjIwOTIxMDAwMDAwWhcNMzMxMTIx
# MjM1OTU5WjBGMQswCQYDVQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQxJDAiBgNV
# BAMTG0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDIyIC0gMjCCAiIwDQYJKoZIhvcNAQEB
# BQADggIPADCCAgoCggIBAM/spSY6xqnya7uNwQ2a26HoFIV0MxomrNAcVR4eNm28
# klUMYfSdCXc9FZYIL2tkpP0GgxbXkZI4HDEClvtysZc6Va8z7GGK6aYo25BjXL2J
# U+A6LYyHQq4mpOS7eHi5ehbhVsbAumRTuyoW51BIu4hpDIjG8b7gL307scpTjUCD
# HufLckkoHkyAHoVW54Xt8mG8qjoHffarbuVm3eJc9S/tjdRNlYRo44DLannR0hCR
# RinrPibytIzNTLlmyLuqUDgN5YyUXRlav/V7QG5vFqianJVHhoV5PgxeZowaCiS+
# nKrSnLb3T254xCg/oxwPUAY3ugjZNaa1Htp4WB056PhMkRCWfk3h3cKtpX74LRsf
# 7CtGGKMZ9jn39cFPcS6JAxGiS7uYv/pP5Hs27wZE5FX/NurlfDHn88JSxOYWe1p+
# pSVz28BqmSEtY+VZ9U0vkB8nt9KrFOU4ZodRCGv7U0M50GT6Vs/g9ArmFG1keLuY
# /ZTDcyHzL8IuINeBrNPxB9ThvdldS24xlCmL5kGkZZTAWOXlLimQprdhZPrZIGwY
# UWC6poEPCSVT8b876asHDmoHOWIZydaFfxPZjXnPYsXs4Xu5zGcTB5rBeO3GiMiw
# bjJ5xwtZg43G7vUsfHuOy2SJ8bHEuOdTXl9V0n0ZKVkDTvpd6kVzHIR+187i1Dp3
# AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADAWBgNV
# HSUBAf8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgB
# hv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYDVR0OBBYE
# FGKK3tBh/I8xFO2XC809KpQU31KcMFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9j
# cmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZU
# aW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEBBIGDMIGAMCQGCCsGAQUFBzAB
# hhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9j
# YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEy
# NTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcNAQELBQADggIBAFWqKhrzRvN4
# Vzcw/HXjT9aFI/H8+ZU5myXm93KKmMN31GT8Ffs2wklRLHiIY1UJRjkA/GnUypsp
# +6M/wMkAmxMdsJiJ3HjyzXyFzVOdr2LiYWajFCpFh0qYQitQ/Bu1nggwCfrkLdcJ
# iXn5CeaIzn0buGqim8FTYAnoo7id160fHLjsmEHw9g6A++T/350Qp+sAul9Kjxo6
# UrTqvwlJFTU2WZoPVNKyG39+XgmtdlSKdG3K0gVnK3br/5iyJpU4GYhEFOUKWaJr
# 5yI+RCHSPxzAm+18SLLYkgyRTzxmlK9dAlPrnuKe5NMfhgFknADC6Vp0dQ094XmI
# vxwBl8kZI4DXNlpflhaxYwzGRkA7zl011Fk+Q5oYrsPJy8P7mxNfarXH4PMFw1nf
# J2Ir3kHJU7n/NBBn9iYymHv+XEKUgZSCnawKi8ZLFUrTmJBFYDOA4CPe+AOk9kVH
# 5c64A0JH6EE2cXet/aLol3ROLtoeHYxayB6a1cLwxiKoT5u92ByaUcQvmvZfpyeX
# upYuhVfAYOd4Vn9q78KVmksRAsiCnMkaBXy6cbVOepls9Oie1FqYyJ+/jbsYXEP1
# 0Cro4mLueATbvdH7WwqocH7wl4R44wgDXUcsY6glOJcB0j862uXl9uab3H4szP8X
# TE0AotjWAQ64i+7m4HJViSwnGWH2dwGMMYIGRDCCBkACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFHmxvNpFYFJPcqgm
# DrQIIVHmaEhpMA0GCSqGSIb3DQEBAQUABIICAHVCgfstZwelSQL+2QwhKmJZf2Ni
# 3HpEyPJO3xIQJ/uhtSdD0enETF/F9OTnfzlQCEjzcL0FJYjHd4P6atbIadzAS79J
# JCb6S8mhIMu5lyozm3JU8aCgPkEYkoRa2VDfaOnBIrkVhr85am9xQ38QfpS8oL8b
# 8kABMyaJB9YlqKhZ2sSN5gfAGRy0QCZRtPQ0bUq5r//uuvhMFvFzpxdfBQcuP70v
# k6EA3zi57q9YHM0vB7x1vNotozTbNJdcb/JWjZs22VfFI6fcvBiJYb7N3RrDHRPp
# TJYzgctfHZTPPLWnMZzGa3LOiSFs+op5ynF1SRaaxvNhWuFx+IP+JD6Svkeeaj/d
# fPaNU9Mj7pyTwIC+21oT2rTsm/1L+/aD74RSkDXoe1pdo1c+4MnRYmuZCGGk/i0i
# sQQeONQn8sUaF3jYKNIk7GVEbtzsjv+ZfAYLZRyg+tUZniHSPRVfeilP1zdVV2Uj
# /mCzbqqV/ePsikh6UGWlu2isGaV4T+DgBz6xFD4QyiWv3WSoFGz1zikFa1C1pmWz
# 4MK+ZTiH/wOuYmxYpavD9nHpErc16o3anA1CQA2F9DuS8DxVYarCPfzguzSzcPFR
# JhBnvYnaX4DKTayEb1XJdUKj7BpsRVuz5gtvT4qkxcj3F9szQviX4Tc6HAKerYIE
# opdS/3pQPFBVr2uaoYIDIDCCAxwGCSqGSIb3DQEJBjGCAw0wggMJAgEBMHcwYzEL
# MAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJE
# aWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBD
# QQIQDE1pckuU+jwqSj0pB4A9WjANBglghkgBZQMEAgEFAKBpMBgGCSqGSIb3DQEJ
# AzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTIyMTIwMjEwMjQxOFowLwYJ
# KoZIhvcNAQkEMSIEIJe2VN1zUf22SRSaj2ez8E2Zq8rghoe/icR5EOLBKHzpMA0G
# CSqGSIb3DQEBAQUABIICAA6YdW8/8CwVUtP+ryFraa378dPJwCFxHJj4El2l4wI2
# aNTMIrAlzLODrZJLs0DN07A2mkMemQdjJo4AeA7Aeqmfi65qz/Etqb+FOLSk50Xo
# hs85qEJyzECU5HSGExMtz08FaOiTZpnx2iwdvr90TJ/cFmskVvDaFbmG8KojLJbz
# po2VStw3ZMRZcS63i+KGrnNzy50woVDHKIwWHusUnPhaVO0BDWB5QBwYm7CNFp+K
# Vqaan1xFJML8KKP5H+dzBerKud9+HpCKgyXUraH+DPjS+M4/9jG9TDaf088zrhGk
# 0Axjb30jLA1Ebpzy10LCVFDybbGbLfgeSDOz0ngtNqD005k1ICBJKMZwoGBDZNPu
# DiVuDzq/i5qTqUq2mQNMLUqwaQ5IFQXbjNbwg+uHuEF91waEt8AKphCaKblJ6X51
# 2i0jtrS4T+N+Rx89DgRW45gFCb8mpX4I2fW+pwIOIhpWPCV8vtth5f6iINy5Mfok
# cAHRCu2tT2D0Pn+5md7v+7AqymK3mr4EQb+NX0vyNTZpkqOIhTNCkyynwByCR1/z
# Fe2l/Q6Qk6lFKDxnJXV73SfOxcrJOCxS9llfj9KMu2+sDni6cLFPMCJBPjZLoZLe
# sSJmlnMn6/HFjl1FMkDFC9SHuUY/ZIwVRp7CrBWcIsSg1T6Fa/Ge3+9Z1+GoTf7n
# SIG # End signature block
