$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$hvModuleHelper="VMware.HV.Helper.psm1"
[string]$eGInstallDir = $args[0]
$hvModule=$eGInstallDir+"\"+$hvModuleHelper

Import-Module -Name VMware.VimAutomation.HorizonView -EA SilentlyContinue
Import-Module -Name $hvModule -Force -EA SilentlyContinue

function Get-ViewAPIService {
  param(
    [Parameter(Mandatory = $false)]
    $HvServer
  )
  if ($null -ne $hvServer) {
    if ($hvServer.GetType().name -ne 'ViewServerImpl') {
      $type = $hvServer.GetType().name
      Write-Error "Expected hvServer type is ViewServerImpl, but received: [$type]"
      return $null
    }
    elseif ($hvServer.IsConnected) {
      return $hvServer.ExtensionData
    }
  } elseif ($global:DefaultHVServers.Length -gt 0) {
     $hvServer = $global:DefaultHVServers[0]
     return $hvServer.ExtensionData
  }
  return $null
}

$hzUser = $args[1]
$hzPass = $args[2]
$DirPath = $eGInstallDir + "\OE.exe"
& $DirPath $hzPass | Tee-Object -Variable pwd | Out-Null
$hzDomain = $args[3]
[string]$hzConnStr = $args[4]
$totalConnectionServers = New-Object -TypeName System.Collections.Generic.List[string]
$hzConnSvrList = $hzConnStr.Split(",")
$foundServer = ""
$Global:ErrorActionPreference="SilentlyContinue"
Set-PowerCLIConfiguration -Scope User -ParticipateInCEIP $false -confirm:$false |out-null
Set-PowerCLIConfiguration -InvalidCertificateAction Ignore -confirm:$false |out-null

if($hzConnSvrList.count -eq 1)
{
	try
	{
		# Establish connection to Connection Server
		$hvServer = Connect-HVServer -Server $hzConnStr -User $hzUser -Password $pwd -Domain $hzDomain
		$foundServer = $hzConnStr
	}
	catch
	{
		write-host $Error[0]
		write-host "Not able to connect to the server $hzConnStr"
		break
	}
}
else
{
	foreach($Server in $hzConnSvrList)
	{
		$totalConnectionServers.Add($Server)
	}
}

foreach($Server in $totalConnectionServers)
{
	try
	{
		$hvServer = Connect-HVServer -Server $Server -User $hzUser -Password $pwd -Domain $hzDomain
		$foundServer = $Server
		if($hvServer -ne $null)
		{
			break
		}
	}
	catch
	{
		write-host $Error[0]
		write-host "Not able to connect to the server $hzConnStr"
	}
}

if($foundServer -eq "" -or $foundServer.Length -eq 0)
{
	write-host "Not able to connect any of the given connection servers"
	break
}

$Global:hvServices = $hvServer.ExtensionData
$services = Get-ViewAPIService -hvServer $hvServer
$vc_service_helper = New-Object VMware.Hv.VirtualCenterService
$vcList = $vc_service_helper.VirtualCenter_List($services)

cls

write-host "POD"
write-host "------"
$Sites = $services.Site.Site_List()
if($Sites -ne $null)
{
	foreach($Site in $Sites)
	{
	    $SiteBase = $Site.Base
	    $SiteDisplayName = $SiteBase.DisplayName
	    $PodIDs = $Site.Pods
	    foreach($PodID in $PodIDs)
	    {
		    $PodInfo = $services.pod.pod_get($PodID)
		    $PodDisplayName = $PodInfo.DisplayName
		    if($PodInfo.LocalPod -eq $True)
		    {
			    $PodEndpointIDs = $PodInfo.Endpoints
			    foreach($PodEndpointID in $PodEndpointIDs)
			    {
				    $PodEndpoint = $services.PodEndpoint.PodEndpoint_Get($PodEndpointID)
				    Write-Host($SiteDisplayName,$PodDisplayName,$PodEndpoint.name,$PodEndpoint.serverAddress,$PodEndpoint.enabled) -Separator "#" 
			    }
		    }
	    }
	}
}
else
{
	$connection_service_helper = New-Object VMware.Hv.ConnectionServerService
	$connList = $connection_service_helper.ConnectionServer_List($services)
	$SiteDisplayName = "-"
	$PodDisplayName = "-"
    foreach($connsrv in $connList)
	{
        $SiteDisplayName = $connsrv.General.ClusterName
        $PodDisplayName = $connsrv.General.ClusterName
		Write-Host($SiteDisplayName,$PodDisplayName,$connsrv.General.name,$connsrv.General.serverAddress,$connsrv.General.enabled) -Separator "#"
	}
}

write-host

Write-Host "Pools"
Write-Host "--------"

$pools = get-hvpool
if($pools -ne $null)
{
	foreach($pool in $pools)
	{
		$headroomcount = -5
		$maximumcount = -5
		$minimumcount = -5
		$vcServerName = -5

		$DesktopBase = $pool.Base
		$id = $DesktopBase.Name
		$displayName = $DesktopBase.DisplayName

		$DesktopSettings = $pool.DesktopSettings
		$enabled = $DesktopSettings.Enabled
		$deliveryModel = $pool.Type
		$desktopSource = $pool.Source
		$EnableProvisioning = $False
		if($deliveryModel -eq "AUTOMATED")
		{
			$DesktopData = $pool.AutomatedDesktopData
			$EnableProvisioning = $DesktopData.VirtualCenterProvisioningSettings.EnableProvisioning
			$VmNamingSettings = $DesktopData.VmNamingSettings
			$PatternNamingSettings = $VmNamingSettings.PatternNamingSettings
			$maximumcount = $PatternNamingSettings.MaxNumberOfMachines
			$minimumcount = $PatternNamingSettings.MinNumberOfMachines
			$headroomcount = $PatternNamingSettings.NumberOfSpareMachines
			if($minimumcount -eq $null -or $minimumcount.length -eq 0) {$minimumcount = -5}
			if($maximumcount -eq $null -or $maximumcount.length -eq 0) {$maximumcount = -5}
			if($headroomcount -eq $null -or $headroomcount.length -eq 0) {$headroomcount = -5}
		}
		elseif($deliveryModel -eq "MANUAL")
		{
			$DesktopData = $pool.ManualDesktopData
		}

		if($DesktopData.VirtualCenter -ne $null)
		{
			$vcid = $DesktopData.VirtualCenter.Id
			foreach($vCenter in $vcList)
			{
				if($vCenter.Id.Id -eq $vcid)
				{
					$vcServerSpec = $vCenter.ServerSpec
					$vcServerName = $vcServerSpec.ServerName +":" +  $vcServerSpec.Port
					if($vcServerName -eq $null -or $vcServerName.length -eq 0)
					{
						$vcServerName = "-"
					}
					break
				}
			}
		}
		$poolType = $deliveryModel
		write-host ($id, $displayName,$enabled, $deliveryModel, $desktopSource,$vcServerName, $poolType, $headroomcount, $maximumcount, $minimumcount,$EnableProvisioning) -Separator "#"
	}
}

write-host
Write-Host "poolentitlement starts"
Write-host "-----------------------"
if($pools -ne $null)
{
	foreach($pool in $pools)
	{
        $outLines = @()
		$DesktopBase = $pool.Base
		$id = $DesktopBase.Name
		$GlobalEntitlementData = $pool.GlobalEntitlementData
		$geID = $GlobalEntitlementData.GlobalEntitlement
		if($geID -ne $null)
		{
			$AndFilter = Get-HVQueryFilter 'globalData.globalEntitlements' -Contains ([VMware.Hv.GlobalEntitlementId[]]$geID)
			$AndFilter = Get-HVQueryFilter -And -Filters $AndFilter
			$results1 = (Get-HVQueryResult -EntityType EntitledUserOrGroupGlobalSummaryView -Filter $AndFilter -HvServer $HvServer) 
			$results1 = $results1 | where {$_.globalData.globalEntitlements -ne $null}
			$outLines += $results1
		}
		$AndFilter = Get-HVQueryFilter 'localData.desktops' -Contains ([VMware.Hv.DesktopId[]]$pool.Id)
		$AndFilter = Get-HVQueryFilter -And -Filters $AndFilter
		$results2 = (Get-HVQueryResult -EntityType EntitledUserOrGroupLocalSummaryView -Filter $AndFilter -HvServer $HvServer)
		$results2 = $results2 | where {$_.localData.desktops -ne $null}
		$outLines += $results2

		if($outLines -ne $null)
		{
            foreach($results in $outLines)
            {
			    $users = $results.base
			    foreach($user in $users)
			    {
				    Write-Host ($id,$user.DisplayName) -Separator "#"
			    }
            }
		}
	}
}

write-host
Write-Host "Desktops starts"
write-host "-----------------"
if($pools -ne $null)
{
	foreach($pool in $pools)
	{
		$DesktopBase = $pool.Base
		$id = $DesktopBase.Name
		$desktopSource = $pool.Source
		if($desktopSource -eq "VIRTUAL_CENTER" -or $desktopSource -eq "VIEW_COMPOSER" -or $desktopSource -eq "INSTANT_CLONE_ENGINE")
		{
			$machines = Get-HVMachine -poolname $id
			if($machines -ne $null)
			{
				foreach($machine in $machines)
				{
					$machineBase = $machine.Base
					$Name = $machineBase.Name
					if($Name -eq $null -or $Name.length -eq 0)
					{
						$Name = "-"
					}
					$GuestFullName = $machineBase.OperatingSystem
					if($GuestFullName -eq $null -or $GuestFullName.length -eq 0)
					{
						$GuestFullName = "-"
					}
					$HostName = $machineBase.DnsName
					if($HostName -eq $null -or $HostName.length -eq 0)
					{
						$HostName = "-"
					}
					$IPAddress = $HostName
					$isInPool = $true
					$machine_id = $Name
					$ManagedMachineData = $machine.ManagedMachineData
					$VirtualCenterData = $ManagedMachineData.VirtualCenterData
					$Path = $VirtualCenterData.Path
					$ServerGroup = $id
					$BasicState = $machineBase.BasicState
					Write-Host ($id,$Name,$Path, $GuestFullName, $HostName, $IPAddress, $isInPool,$machine_id,$ServerGroup,$BasicState) -Separator "#"
				}
			}
		}
		elseif($desktopSource -eq "RDS")
		{
			$RdsDesktopData = $pool.RdsDesktopData
			if($RdsDesktopData -ne $null)
			{
				$FarmID = $RdsDesktopData.Farm
				$FarmInfo = $services.Farm.Farm_GetSummaryView($FarmID)
				if($FarmInfo -ne $null)
				{
					$farmData = $FarmInfo.Data
					$Path = "-"
					$GuestFullName = "-"
					$isInPool = $true
					$Enabled = $farmData.Enabled
					if($Enabled -eq $null -or $Enabled.length -eq 0)
					{
						$Enabled = 0	#0 means Enabled   1 means Disabled		
					}
					$MaxConns = $farmData.MaximumNumberOfSessions
					if($MaxConns -eq $null -or $MaxConns.length -eq 0)
					{
						$MaxConns = 150
					}
					$farmHealth = $services.FarmHealth.FarmHealth_Get($FarmID)
					if($farmHealth -ne $null)
					{
						$name = $farmHealth.Name
						$RDSServersHealth = $farmHealth.RDSServerHealth
						foreach($RDSServer in $RDSServersHealth)
						{
							$HostName = "-"
							$IPAddress = "-"
							$machine_id = "-"
							$ServerGroup = "-"
							$BasicState = "-"

							$IPAddress = $RDSServer.Name
							$machine_id = $RDSServer.Name
							$HostName = $RDSServer.Name  
							$BasicState = $RDSServer.Status

							Write-Host ($id,$name,$Path, $GuestFullName, $HostName, $IPAddress, $isInPool,$machine_id,$ServerGroup,$BasicState,$Enabled,$MaxConns) -Separator "#"
						}
					}
				}
			}
		}
	}
}


write-host
Write-Host "PoolsUsage"
Write-Host "----------"
$PodSessions = get-hvlocalsession
if($podSessions -ne $null)
{
	foreach($session in $podSessions)
	{
		$NamesData = $session.NamesData
		$mac_id = $NamesData.MachineOrRDSServerName
		$SessionData = $session.SessionData
		$userName = $NamesData.userName
		$SessUniqID = $session.id.id
		if($userName -eq $null -or $userName.length -eq 0)
		{
			$userName = "-"
		}
		$DNSName = $NamesData.MachineOrRDSServerDNS
		if($DNSName -eq $null -or $DNSName.length -eq 0)
		{
			$DNSName = "-"
		}
		$startTime = $SessionData.startTime
		if($startTime -eq $null -or $startTime.length -eq 0)
		{
			$startTime = "-"
		}
		$duration = $SessionData.LastSessionDurationMS
		if($duration -eq $null -or $duration.length -eq 0)
		{
			$duration = "-"
		}
		$state = $SessionData.SessionState
		if($state -eq $null -or $state.length -eq 0)
		{
			$state = "-"
		}
		$protocol = $SessionData.SessionProtocol
		if($protocol -eq $null -or $protocol.length -eq 0)
		{
			$protocol = "-"
		}
		$session_id = $SessionData.SessionType
		if($session_id -eq $null -or $session_id.length -eq 0)
		{
			$session_id = "-"
		}
		$pool_id = $NamesData.DesktopName
		if($pool_id -eq $null)
		{
			$pool_id = $NamesData.FarmName
		}
		Write-Host ($pool_id,$userName,$startTime, $DNSName, $duration, $mac_id, $state, $protocol, $session_id, $SessUniqID) -Separator "#"
	}
}

Write-Host

Write-Host "RDS Applications"
Write-Host "------------------"
$AppForRDSServers = New-Object -TypeName System.Collections.Generic.List[string]
$MissingAppForRDSServers = New-Object -TypeName System.Collections.Generic.List[string]
$query_service_helper = New-Object VMware.Hv.QueryServiceService
$defn = New-Object VMware.Hv.QueryDefinition
$defn.queryEntityType = 'ApplicationInfo'
$queryResults = $query_service_helper.QueryService_Query($services,$defn)
$AppsInfo = $queryResults.results

if($AppsInfo -ne $null)
{
	foreach($AppInfo in $AppsInfo)
	{
		$MissingAppExists = $false
		$AppForRDSServers.Clear()
		$MissingAppForRDSServers.Clear()
		$AppData = $AppInfo.data
		$AppDisplayName = $AppData.Name
		$AppEnabled = $AppData.enabled
		$farmID = $AppInfo.executionData.farm
		$farmHealth = $services.FarmHealth.FarmHealth_Get($farmID)
		if($farmHealth -ne $null)
		{
			$fname = $farmhealth.name
			$RDSServersHealth = $farmHealth.RDSServerHealth
			foreach($RDSServer in $RDSServersHealth)
			{
				$IPAddress = $RDSServer.Name
				$AppForRDSServers.Add($IPAddress)
				$MissingApplications = $RDSServer.MissingApplications
				if($MissingApplications -ne $null)
				{
					foreach($missingApps in $MissingApplications)
					{
						
						if($missingApps.Name -eq $AppDisplayName)
						{
							$MissingAppExists = $true
							$MissingAppForRDSServers.add($IPAddress)
						}
					}
				} 
			}
			write-host ($AppDisplayName,$fname, $AppForRDSServers,$AppEnabled,$MissingAppExists,$MissingAppForRDSServers) -Separator "#"  
		}
	}
}

write-host
write-host "RDS Farms"
Write-host "-----------"
$farms = get-hvfarm -SuppressInfo $true
if($farms -ne $null)
{
	foreach($farm in $farms)
	{
		$farmId = $farm.ID
		$farmtype = $farm.Type
		$FarmInfo = $services.Farm.Farm_GetSummaryView($FarmID)
		if($FarmInfo -ne $null)
		{
			$farmData = $FarmInfo.Data
			$Path = "-"
			$GuestFullName = "-"
			$isInPool = $true
			$Enabled = $farmData.Enabled
			if($Enabled -eq $null -or $Enabled.length -eq 0)
			{
				$Enabled = 0	#0 means Enabled   1 means Disabled		
			}
			$MaxConns = $farmData.MaximumNumberOfSessions
			if($MaxConns -eq $null -or $MaxConns.length -eq 0)
			{
				$MaxConns = 150
			}
			$farmHealth = $services.FarmHealth.FarmHealth_Get($FarmID)
			if($farmHealth -ne $null)
			{
				$name = $farmHealth.Name
				$RDSServersHealth = $farmHealth.RDSServerHealth
				foreach($RDSServer in $RDSServersHealth)
				{
					$HostName = "-"
					$IPAddress = "-"
					$machine_id = "-"
					$ServerGroup = "-"
					$BasicState = "-"

					$IPAddress = $RDSServer.Name
					$machine_id = $RDSServer.Name
					$HostName = $RDSServer.Name  
					$BasicState = $RDSServer.Status
					Write-Host ($name,$Path, $GuestFullName, $HostName, $IPAddress, $isInPool,$machine_id,$ServerGroup,$BasicState,$Enabled,$MaxConns) -Separator "#"
				}
			}
		}
	}
}
write-host
write-host "Date Format"
Write-host "-----------"
$f = (get-culture).datetimeformat.ShortDatePattern
$h = (get-culture).datetimeformat.LongTimePattern
Write-host "$f $h"


Disconnect-HVServer -confirm:$false

# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUqXYQ0r5HceprOnYa7MQqXiQ1
# Xg2gghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFNyrQ4s74n6tKnW8
# 9AFwzAzfMPRAMA0GCSqGSIb3DQEBAQUABIICABoVg8BXlrFGY7GEzCfRZVdL1NZL
# wmONdjHT4NoV8nR0Yilo2EixU0fdSCtp7CKktHSmNNcLM+OCakpD7MLDSkX6IjNH
# LCLq4Gn5nuXicj4DxZgA7q11uHfMjwS0zPxngiKBbL/4atLhO9xOGldb1G5gMYQW
# ayYLVLUrNxIdb42QvJ/yP5ioxtlyLAIbHC13Z8JkTwZzWeDK4ys0KXniV30IHTzu
# xE2sS3czRpUX8ULIk8xOxnFcE42q4Zzr2q061Z7pVHICvBgPZC2N6jWWybnxlmua
# mr7G4IUuhcBUK7tQioczpPm3b/Jc3wqKPLwN5b5POWmqlEZ6Csy+syzYKiWGh9WE
# Y1cc9xzDOwtWYFIZfCfF9ek2ovBcj7QheM5qi9oOKgNW1SEiIjxIMiAOYc5TxfbJ
# wS7DniTph+J0oV6E4wC7azVEugwKFDVXuTl6OS3F6kBvqA4uhvcK0/YaWodPBVco
# SliY9xF+FiDBCj4LhCtYb/quLCkaP8gpA22GzIcfHNh6UmpqGv+6GSNdB/ppwsYg
# F8FIIVejuG8UmB3cWtgNEpFBWlRjB4qnWfzXdenDRmbhFd8J6uUT8emzCllNly2b
# z0rd2L8Ygst5ZPht3/pdamrB3FL4QnfGXrckZOYchhWustfjbH+RXKEcRkimUwI/
# 8wiw1MxpvEmhWLZGoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MDExMjM3MDNaMC8GCSqGSIb3DQEJBDEiBCCi+aDJgsWSRsLWDL0imQwixma7
# 3KlZqZqAguVT21wyDjANBgkqhkiG9w0BAQEFAASCAQCd6TYWObYRWYFr2xtHGwal
# 4ONxrFd49zaVxOjmgQctz1S5ZkPsvARNfy+y4mMb2CTaoh7vOEHr8RYCTZrCmBsL
# VVdRBr2nuT9AOB2ayr5OIbKgpeP8WxX0M/FKg2WhP8/Ar2Wh3MbJH+zEHEjPr3Yv
# y1x+gigsn0KtQDRkwKWqPWJbtS2KjLftbjJ0XtaDdIGDBI2V2oqnJItuv2h0mSNr
# fr/AuQODjDFqE4JsnZjrWM1Bl4bD355vZdvkaemaS0UF646Nt4pI8Hb2zLxtmcM2
# uywWTGxakcTgEAp7whMKuq6Os7mUSY8d8cPrWNeYE70UjkGCWht8NICauLlo1r8m
# SIG # End signature block
