#Requires -Version 3.0
#Requires -Module dnsserver
#This File is in Unicode format.  Do not edit in an ASCII editor.

#region help text

<#
.SYNOPSIS
	Provides DNS server information in the customized format.
.DESCRIPTION

	To run the script from a workstation, RSAT is required.
	
	Remote Server Administration Tools for Windows 7 with Service Pack 1 (SP1)
		http://www.microsoft.com/en-us/download/details.aspx?id=7887
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520

.PARAMETER ComputerName
	Specifies a computer to use to run the script against.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	Default is localhost.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  
#>

#endregion

#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None") ]

Param(
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost"
	
	)
#endregion

#region initialize variables
$SaveEAPreference = $ErrorActionPreference
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DNS config data ***"
$scriptEndIdentifier = "*** End - DNS config data ***"
$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region reusable common functions
Function validObject( [object] $object, [string] $topLevel )
{
	If( $object )
	{
		If((Get-Member -Name $topLevel -InputObject $object))
		{
			Return $True
		}
	}
	Return $False
}

Function ResolveIPtoFQDN
{
	Param([string]$cname)
	$ip = $CName -as [System.Net.IpAddress]
	try
	{
		If($ip)
		{
			$Result = [System.Net.Dns]::gethostentry($ip)
			If($? -and $Null -ne $Result)
			{
				$CName = $Result.HostName
			}
			Else
			{
				$CName = 'Unable to resolve'
			}
		}	
	}
	catch
	{
		$CName = 'Unable to resolve'
		Write-Verbose "Catch Block is Executed : ResolveIPtoFQDN"
	}
	Return $CName
}
#endregion

#region script setup function
Function ProcessScriptStart
{
	$ComputerName = TestComputerName $ComputerName
}

Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		Write-Verbose "$(Get-Date): Testing to see if $CName is online and reachable"
		If(Test-Connection -ComputerName $CName -quiet)
		{
			Write-Verbose "$(Get-Date): Server $CName is online."
		}
		Else
		{
			Write-Verbose "$(Get-Date): Computer $CName is offline"
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $CName is offline.`nScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		Write-Verbose "$(Get-Date): Computer name has been renamed from localhost to $CName"
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip).AddressList.IPAddressToString
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
			Write-Verbose "$(Get-Date): Computer name has been renamed from $($ip) to $CName"
		}
		Else
		{
			Write-Warning "Unable to resolve $CName to a hostname"
		}
	}
	Else
	{
		#computer is online but for some reason $ComputerName cannot be converted to a System.Net.IpAddress
	}

	$Results = Get-DNSServer -ComputerName $CName -EA 0 3>$Null
	If($Null -ne $Results)
	{
		#the computer is a dns server
		Write-Verbose "$(Get-Date): Computer $CName is a DNS Server"
		Write-Verbose "$(Get-Date): "
		$Script:DNSServerData = $Results
		Return $CName
	}
	ElseIf($Null -eq $Results)
	{
		#the computer is not a dns server
		Write-Verbose "$(Get-Date): Computer $CName is not a DNS Server"
		$ErrorActionPreference = $SaveEAPreference
		Write-Error "`n`n`t`tComputer $CName is not a DNS Server.`n`n`t`tRerun the script using -ComputerName with a valid DNS server name.`n`n`t`tScript cannot continue.`n`n"
		Exit
	}
	Return $CName
}
#endregion

#region process lookup  zone data
Function OutputLookupZone
{
	Param([string] $zType, [object] $DNSZone)
	
	#General tab
	Write-Verbose "$(Get-Date): `t`tGeneral"
	$Status = "N/A"
	$ZoneType = "N/A"
	$Replication = "N/A"
	$DynamicUpdate = "N/A"
	$NorefreshInterval = "N/A"
	$SaRefreshInterval = "N/A"
	$RefreshInterval = "N/A"
	$EnableScavenging = "N/A"
	$ipNotifyServers = ""
	$Datastored = ""
	$SerialNumber= ""
	$PrimaryServer = ""
	$ResponsiblePerson = ""
	$RetryDelay = ""
	$ExpireLimit = ""
	$MinimumTTL = ""
	$TTLRecord = ""
	$WINSEnabled = ""
	$DomainNameAppend = ""
	$CacheTimeout = ""
	$LookupTimeout = ""
	$WINSReplicate = ""
	$ZoneTransfer = ""
	$Automaticallynotify = ""
	$IPAdrsFqdnNtSrvr = ""
	$IPAdrsFqdn = ""

	If($DNSZone.IsPaused -eq $False)
	{
		$Status = "Running"
	}
	Else
	{
		$Status = "Paused"
	}
	
	If($DNSZone.ZoneType -eq "Primary" -and $DNSZone.IsDsIntegrated -eq $True)
	{
		$ZoneType = "Active Directory-Integrated"
	}
	ElseIf($DNSZone.ZoneType -eq "Primary" -and $DNSZone.IsDsIntegrated -eq $False)
	{
		$ZoneType = "Primary"
	}
	ElseIf($DNSZone.ZoneType -eq "Secondary" -and $DNSZone.IsDsIntegrated -eq $False)
	{
		$ZoneType = "Secondary"
	}
	ElseIf($DNSZone.ZoneType -eq "Stub")
	{
		$ZoneType = "Stub"
	}
	
	Switch ($DNSZone.ReplicationScope)
	{
		"Forest" {$Replication = "All DNS servers in this forest"; break}
		"Domain" {$Replication = "All DNS servers in this domain"; break}
		"Legacy" {$Replication = "All domain controllers in this domain (for Windows 2000 compatibility"; break}
		"None" {$Replication = "Not an Active-Directory-Integrated zone"; break}
		Default {$Replication = "Unknown: $($DNSZone.ReplicationScope)"; break}
	}
	
	If( ( validObject $DNSZone DynamicUpdate ) )
	{
		Switch ($DNSZone.DynamicUpdate)
		{
			"Secure" {$DynamicUpdate = "Secure only"; break}
			"NonsecureAndSecure" {$DynamicUpdate = "Nonsecure and secure"; break}
			"None" {$DynamicUpdate = "None"; break}
			Default {$DynamicUpdate = "Unknown: $($DNSZone.DynamicUpdate)"; break}
		}
	}
	
	If($DNSZone.ZoneType -eq "Primary")
	{
		$ZoneAging = Get-DnsServerZoneAging -Name $DNSZone.ZoneName -ComputerName $ComputerName -EA 0
		
		If($Null -ne $ZoneAging)
		{
			If($ZoneAging.AgingEnabled)
			{
				$EnableScavenging = "Selected"
				If($ZoneAging.NoRefreshInterval.days -gt 0)
				{
					$NorefreshInterval = "$($ZoneAging.NoRefreshInterval.days) days"
				}
				ElseIf($ZoneAging.NoRefreshInterval.hours -gt 0)
				{
					$NorefreshInterval = "$($ZoneAging.NoRefreshInterval.hours) hours"
				}
				If($ZoneAging.RefreshInterval.days -gt 0)
				{
					$SaRefreshInterval = "$($ZoneAging.RefreshInterval.days) days"
				}
				ElseIf($ZoneAging.RefreshInterval.hours -gt 0)
				{
					$SaRefreshInterval = "$($ZoneAging.RefreshInterval.hours) hours"
				}
			}
			Else
			{
				$EnableScavenging = "Not Selected"
			}
		}
		Else
		{
			$EnableScavenging = "Unknown"
		}
		
		$ScavengeServers = @()
		
		If($ZoneAging.ScavengeServers -is [array])
		{
			ForEach($Item in $ZoneAging.ScavengeServers)
			{
				$ScavengeServers += $ZoneAging.ScavengeServers.IPAddressToString
			}
		}
		Else
		{
			$ScavengeServers += $ZoneAging.ScavengeServers.IPAddressToString
		}
		
		If($ScavengeServers.Count -eq 0)
		{
			$ScavengeServers += "Not Configured"
		}
	}
	
	If($Null -ne $DNSZone.ZoneFile)
	{
		$ZoneFile ="Zone file name : " + $DNSZone.ZoneFile
	}
	ElseIf($Null -eq $DNSZone.ZoneFile -and $DNSZone.IsDsIntegrated)
	{
		$Datastored  = " Yes " 
	}
	Else
	{
		$Datastored  = " No" 
	}
	
	If($DNSZone.ZoneType -eq "Primary")
	{
		
		$ScavengeServers = $ScavengeServers[0]
		
		$cnt = -1
		ForEach($ScavengeServer in $ScavengeServers)
		{
			$cnt++
			
			If($cnt -gt 0)
			{
				$ScavengeServer = "Scavenge server :" + $ScavengeServer
			}
		}
	}
	
	#Start of Authority (SOA) tab
	$Results = Get-DnsServerResourceRecord -zonename $DNSZone.ZoneName -rrtype soa -ComputerName $ComputerName -EA 0
	If($? -and $Null -ne $Results)
	{
		$SOA = $Results[0]
		
		If($SOA.RecordData.RefreshInterval.Days -gt 0)
		{
			$RefreshInterval = "$($SOA.RecordData.RefreshInterval.Days) days"
		}
		ElseIf($SOA.RecordData.RefreshInterval.Hours -gt 0)
		{
			$RefreshInterval = "$($SOA.RecordData.RefreshInterval.Hours) hours"
		}
		ElseIf($SOA.RecordData.RefreshInterval.Minutes -gt 0)
		{
			$RefreshInterval = "$($SOA.RecordData.RefreshInterval.Minutes) minutes"
		}
		ElseIf($SOA.RecordData.RefreshInterval.Seconds -gt 0)
		{
			$RefreshInterval = "$($SOA.RecordData.RefreshInterval.Seconds) seconds"
		}
		Else
		{
			$RefreshInterval = "Unknown"
		}
		
		If($SOA.RecordData.RetryDelay.Days -gt 0)
		{
			$RetryDelay = "$($SOA.RecordData.RetryDelay.Days) days"
		}
		ElseIf($SOA.RecordData.RetryDelay.Hours -gt 0)
		{
			$RetryDelay = "$($SOA.RecordData.RetryDelay.Hours) hours"
		}
		ElseIf($SOA.RecordData.RetryDelay.Minutes -gt 0)
		{
			$RetryDelay = "$($SOA.RecordData.RetryDelay.Minutes) minutes"
		}
		ElseIf($SOA.RecordData.RetryDelay.Seconds -gt 0)
		{
			$RetryDelay = "$($SOA.RecordData.RetryDelay.Seconds) seconds"
		}
		Else
		{
			$RetryDelay = "Unknown"
		}
		
		If($SOA.RecordData.ExpireLimit.Days -gt 0)
		{
			$ExpireLimit = "$($SOA.RecordData.ExpireLimit.Days) days"
		}
		ElseIf($SOA.RecordData.ExpireLimit.Hours -gt 0)
		{
			$ExpireLimit = "$($SOA.RecordData.ExpireLimit.Hours) hours"
		}
		ElseIf($SOA.RecordData.ExpireLimit.Minutes -gt 0)
		{
			$ExpireLimit = "$($SOA.RecordData.ExpireLimit.Minutes) minutes"
		}
		ElseIf($SOA.RecordData.ExpireLimit.Seconds -gt 0)
		{
			$ExpireLimit = "$($SOA.RecordData.ExpireLimit.Seconds) seconds"
		}
		Else
		{
			$ExpireLimit = "Unknown"
		}
		
		If($SOA.RecordData.MinimumTimeToLive.Days -gt 0)
		{
			$MinimumTTL = "$($SOA.RecordData.MinimumTimeToLive.Days) days"
		}
		ElseIf($SOA.RecordData.MinimumTimeToLive.Hours -gt 0)
		{
			$MinimumTTL = "$($SOA.RecordData.MinimumTimeToLive.Hours) hours"
		}
		ElseIf($SOA.RecordData.MinimumTimeToLive.Minutes -gt 0)
		{
			$MinimumTTL = "$($SOA.RecordData.MinimumTimeToLive.Minutes) minutes"
		}
		ElseIf($SOA.RecordData.MinimumTimeToLive.Seconds -gt 0)
		{
			$MinimumTTL = "$($SOA.RecordData.MinimumTimeToLive.Seconds) seconds"
		}
		Else
		{
			$MinimumTTL = "Unknown"
		}
	}
	Else
	{
		Write-Host "Start of Authority data could not be retrieved"
	}

	If($zType -eq "Reverse")
	{
		#WINS-R tab
		#Write-Verbose "$(Get-Date): `t`tWINS-R"
		If( ( validObject $DNSZone IsWinsEnabled ) )
		{
			If($DNSZone.IsWinsEnabled)
			{
				$WINSEnabled = "Selected"
				
				$WINS = Get-DnsServerResourceRecord -zonename $DNSZone.ZoneName -rrtype winsr -ComputerName $ComputerName -EA 0
				
				If($? -and $Null -ne $WINS)
				{
					If($WINS.RecordData.Replicate)
					{
						$WINSReplicate = "Selected"
					}
					Else
					{
						$WINSReplicate = "Not selected"
					}

					If($WINS.RecordData.CacheTimeout.Days -gt 0)
					{
						$CacheTimeout = "$($WINS.RecordData.CacheTimeout.Days) days"
					}
					ElseIf($WINS.RecordData.CacheTimeout.Hours -gt 0)
					{
						$CacheTimeout = "$($WINS.RecordData.CacheTimeout.Hours) hours"
					}
					ElseIf($WINS.RecordData.CacheTimeout.Minutes -gt 0)
					{
						$CacheTimeout = "$($WINS.RecordData.CacheTimeout.Minutes) minutes"
					}
					ElseIf($WINS.RecordData.CacheTimeout.Seconds -gt 0)
					{
						$CacheTimeout = "$($WINS.RecordData.CacheTimeout.Seconds) seconds"
					}
					Else
					{
						$CacheTimeout = "Unknown"
					}

					If($WINS.RecordData.LookupTimeout.Days -gt 0)
					{
						$LookupTimeout = "$($WINS.RecordData.LookupTimeout.Days) days"
					}
					ElseIf($WINS.RecordData.LookupTimeout.Hours -gt 0)
					{
						$LookupTimeout = "$($WINS.RecordData.LookupTimeout.Hours) hours"
					}
					ElseIf($WINS.RecordData.LookupTimeout.Minutes -gt 0)
					{
						$LookupTimeout = "$($WINS.RecordData.LookupTimeout.Minutes) minutes"
					}
					ElseIf($WINS.RecordData.LookupTimeout.Seconds -gt 0)
					{
						$LookupTimeout = "$($WINS.RecordData.LookupTimeout.Seconds) seconds"
					}
					Else
					{
						$LookupTimeout = "Unknown"
					}
				}
				Else
				{
					#$txt1 = "WINS"
					Write-Host "Use WINS forward lookup: $WINSEnabled"
					Write-Host "Unable to retrieve WINS details"
				}
			}
			Else
			{
				$WINSEnabled = "Not selected"
			}
		}
	}

	If( (validObject $DNSZone SecureSecondaries) )
	{
		If($DNSZone.SecureSecondaries -ne "NoTransfer")
		{
			If($DNSZone.SecureSecondaries -eq "TransferAnyServer")
			{
				$ZoneTransfer = "To any server"
			}
			ElseIf($DNSZone.SecureSecondaries -eq "TransferToZoneNameServer")
			{
				$ZoneTransfer = "Only to servers listed on the Name Servers tab"
			}
			ElseIf($DNSZone.SecureSecondaries -eq "TransferToSecureServers")
			{
				$ZoneTransfer = "Only to the following servers"
			}
			Else
			{
				$ZoneTransfer = "Unknown"
			}
			
			If($ZoneTransfer -eq "Only to the following servers")
			{
				If($? -and $Null -ne $DNSZone.SecondaryServers)
				{
					ForEach($ipAddressZnTra in $DNSZone.SecondaryServers)
					{
						$ResolvedZnTra = ResolveIPtoFQDN $ipAddressZnTra
						$IPAdrsFqdn += "$($ipAddressZnTra)($($ResolvedZnTra)); "
					}
				}
			}
			If($DNSZone.Notify -eq "NotifyServers")
			{
				If($? -and $Null -ne $DNSZone.NotifyServers)
				{
					ForEach($ipAddressZnTra in $DNSZone.NotifyServers)
					{
						$ResolvedNtSrvr = ResolveIPtoFQDN $ipAddressZnTra
						$IPAdrsFqdnNtSrvr += "$($ipAddressZnTra)($($ResolvedNtSrvr)); "
					}
				}
			}
			
			If($DNSZone.Notify -eq "NoNotify")
			{
				$Automaticallynotify = "Not selected"
			}
			ElseIf($DNSZone.Notify -eq "Notify")
			{
				$Automaticallynotify =  " Servers listed on the Name Servers tab"
			}
			ElseIf($DNSZone.Notify -eq "NotifyServers")
			{
				$Automaticallynotify = " The following servers " 
			}
		}
		Else
		{
			$ZoneTransfer = "Not selected"
		}
	}
	$SerialNumber = $($SOA.RecordData.SerialNumber.ToString())
	$PrimaryServer = $($SOA.RecordData.PrimaryServer)
	$ResponsiblePerson = $($SOA.RecordData.ResponsiblePerson)
	$TTLRecord = $($SOA.TimeToLive.ToString())
	$DomainNameAppend = $($WINS.RecordData.ResultDomain)

	Write-Host "server name :$($env:ComputerName) $($delimiter)  Reverse Lookup Zone Name : $($DNSZone.ZoneName) $($delimiter) Status :$($Status) $($delimiter) Type :$($ZoneType) $($delimiter) Replication :$($Replication) $($delimiter) Data stored in Active Directory : $($Datastored) $($delimiter) Dynamic updates : $($DynamicUpdate) $($delimiter) Scavenge stale resource records : $($EnableScavenging) $($delimiter)  No-refresh interval:  $($NorefreshInterval) $($delimiter) General Refresh interval :  $($SaRefreshInterval )  $($delimiter) Serial number : $($SerialNumber) $($delimiter) Primary server : $($PrimaryServer) $($delimiter) Responsible person : $($ResponsiblePerson) $($delimiter) Refresh interval : $($RefreshInterval) $($delimiter) Retry interval : $($RetryDelay) $($delimiter) Expires after : $($ExpireLimit) $($delimiter) Minimum (default) TTL : $($MinimumTTL) $($delimiter) TTL for this record : $($TTLRecord) $($delimiter) Use WINS-R lookup : $($WINSEnabled) $($delimiter) Domain name to append :$($DomainNameAppend) $($delimiter) Cache time-out :$($CacheTimeout) $($delimiter) Lookup time-out :$($LookupTimeout) $($delimiter) Submit DNS domain as NetBIOS scope :$($WINSReplicate) $($delimiter)  Allow zone transfers :  $($ZoneTransfer) $($delimiter) AutomaticallyNotify  :  $($Automaticallynotify)  $($delimiter) Notify IPAddress and Server FQDN : $($IPAdrsFqdnNtSrvr) $($delimiter) Zone Transfers IPAddress and Server FQDN :  $($IPAdrsFqdn)	"
}

Function ProcessReverseLookupZones
{
	$DNSZones = $Script:DNSServerData.ServerZone | Where-Object {$_.IsReverseLookupZone -eq $True}
	ForEach($DNSZone in $DNSZones)
	{
		OutputLookupZone "Reverse" $DNSZone
	}
}
#endregion


#region script core
#Script begins
Write-Host $scriptStartIdentifier 
ProcessScriptStart
try
{
	ProcessReverseLookupZones
}
catch 
{
	$global:isErrorOccured = $True
	Write-Verbose "catch Blcok is Executed :"
}

If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion
