#requires -Version 3.0
#requires -Module DHCPServer
#This File is in Unicode format.  Do not edit in an ASCII editor.

#region help text
<#
.SYNOPSIS
	To get statistics (Scope Name, No. of addresses in use, No. of free addresses, No. of total addresses, Scope status) of all 
	the IPV4 scopes (scope, super scope, multicast scope) and IPV6 scopes
.DESCRIPTION
	To get statistics (Scope Name, No. of addresses in use, No. of free addresses, No. of total addresses, Scope status) of all 
	the IPV4 scopes (scope, super scope, multicast scope) and IPV6 scopes
	
	Script requires at least PowerShell version 3 but runs best in version 5.
	
	Requires the DHCPServer module.
	Can be run on a DHCP server or on a Windows 8.x or Windows 10 computer with RSAT installed.
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520
	
	For Windows Server 2003, 2008 and 2008 R2, use the following to export and import the 
	DHCP data:
		Export from the 2003, 2008 or 2008 R2 server:
			netsh dhcp server export C:\DHCPExport.txt all
			
			Copy the C:\DHCPExport.txt file to the 2012+ server.
			
		Import on the 2012+ server:
			netsh dhcp server import c:\DHCPExport.txt all
			
		The script can now be run on the 2012+ DHCP server to document the older DHCP 
		information.

	For Windows Server 2008 and Server 2008 R2, the 2012+ DHCP Server PowerShell cmdlets 
	can be used for the export and import.
		From the 2012+ DHCP server:
			Export-DhcpServer -ComputerName 2008R2Server.domain.tld -Leases -File 
			C:\DHCPExport.xml 
			
			Import-DhcpServer -ComputerName 2012Server.domain.tld -Leases -File 
			C:\DHCPExport.xml -BackupPath C:\dhcp\backup\ 
			
			Note: The c:\dhcp\backup path must exist before the Import-DhcpServer 
			cmdlet is run.
	
	Using netsh is much faster than using the PowerShell export and import cmdlets.
	
	Processing of IPv4 Multicast Scopes is only available with Server 2012 R2 DHCP.
	
.PARAMETER ComputerName
	DHCP server to run the script against.
	The computername is used for the report title.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
.PARAMETER AllDHCPServers
	The script will process all Authorized DHCP servers that are online.
	"All DHCP Servers" is used for the report title.
	This parameter is disabled by default.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
	This parameter has an alias of ALL.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  Customized text output with delimeter.
#>
#endregion

#region initialize variables
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False

$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region DHCP script functions
Function ProcessIPv4SuperScopeStatistics
{
	Write-Verbose "$(Get-Date): Getting IPv4 Superscopes"
	$IPv4Superscopes = Get-DHCPServerV4Superscope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv4Superscopes)
	{
		ForEach($IPv4Superscope in $IPv4Superscopes)
		{
			If(![string]::IsNullOrEmpty($IPv4Superscope.SuperscopeName))
			{
				$Statistics = Get-DHCPServerV4SuperscopeStatistics -ComputerName $Script:DHCPServerName -Name $IPv4Superscope.SuperscopeName -EA 0
				
				[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				$AddressesFree = $Statistics.AddressesFree
				$AddressesInUse = $Statistics.AddressesInUse
				if( $IPv4Superscope.status -eq "Active")
				{
					$IPv4Superscope.status = 1;
				}
				elseif( $IPv4Superscope.status -eq "Inactive")
				{
					$IPv4Superscope.status = 0;
				}
				Write-Host "Scopename : $($IPv4Superscope.SuperscopeName) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($IPv4Superscope.status)"
			}
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Verbose "Error retrieving IPv4 Superscopes"
	}
	
	$IPv4Superscopes = $Null
	[gc]::collect() 
}

Function ProcessIPv4ScopeStatistics
{
	$IPv4Scopes = Get-DHCPServerV4Scope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv4Scopes)
	{
		ForEach($IPv4Scope in $IPv4Scopes)
		{
			$ScopeIPScopeName = "[$($IPv4Scope.ScopeId)] $($IPv4Scope.Name)"
			$Ipv4ScopeState = $IPv4Scope.state
		   
			$Statistics   =  Get-DhcpServerv4ScopeStatistics -ComputerName $Script:DHCPServerName -ScopeId $IPv4Scope.ScopeId -EA 0
			
			If($? -and $Null -ne $Statistics)
			{
				[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				$AddressesFree = $Statistics.AddressesFree
				$AddressesInUse = $Statistics.AddressesInUse
				if( $Ipv4ScopeState -eq "Active")
				{
					$Ipv4ScopeState = 1;
				}
				elseif( $Ipv4ScopeState -eq "Inactive")
				{
					$Ipv4ScopeState = 0;
				}
				Write-Host "Scopename: $($ScopeIPScopeName) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($Ipv4ScopeState)"
			}
			ElseIf(!$?)
			{
				$global:isErrorOccured = $True
				Write-Verbose "Error retrieving scope statistics : $ScopeIPScopeName"
			}					
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Verbose "Error retrieving IPv4 scopes "
	}
	
	$IPv4Scopes = $Null
	[gc]::collect() 
}
Function ProcessIPv4MulticastScopesStatistics
{
	$CmdletName = "Get-DHCPServerV4MulticastScope"
	If(Get-Command $CmdletName -Module "DHCPServer" -EA 0)
	{
		$IPv4MulticastScopes = Get-DHCPServerV4MulticastScope -ComputerName $Script:DHCPServerName -EA 0

		If($? -and $Null -ne $IPv4MulticastScopes)
		{
			ForEach($IPv4MulticastScope in $IPv4MulticastScopes)
			{	
				$multicastScopeState = $IPv4MulticastScope.state	
				$Statistics = Get-DHCPServerV4MulticastScopeStatistics -ComputerName $Script:DHCPServerName -Name $IPv4MulticastScope.Name -EA 0
				
				If($? -and $Null -ne $Statistics)
				{
					[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
					$AddressesFree = $Statistics.AddressesFree
					$AddressesInUse = $Statistics.AddressesInUse
					if( $multicastScopeState -eq "Active")
					{
						$multicastScopeState = 1;
					}
					elseif( $multicastScopeState -eq "Inactive")
					{
						$multicastScopeState = 0;
					}
					Write-Host "Scopename : $($IPv4MulticastScope.Name) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($multicastScopeState)"
				}
				ElseIf(!$?)
				{
					$global:isErrorOccured = $True
					Write-Verbose "Error retrieving scope statistics"
				}	
			}
		}
		ElseIf(!$?)
		{
			$global:isErrorOccured = $True
			Write-Verbose "Error retrieving IPv4 Multicast scopes"
		}
		
		$IPv4MulticastScopes = $Null
	}
	[gc]::collect() 
}
Function ProcessIPv6ScopesStatistics
{	
	$IPv6Scopes = Get-DHCPServerV6Scope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv6Scopes)
	{
		ForEach($IPv6Scope in $IPv6Scopes)
		{
			$scopename  = "[$($IPv6Scope.Prefix)] $($IPv6Scope.Name)"
			$Ipv6State = $IPv6Scope.state
			$Statistics = Get-DHCPServerV6ScopeStatistics -ComputerName $Script:DHCPServerName -Prefix $IPv6Scope.Prefix -EA 0
			
			If($? -and $Null -ne $Statistics)
			{		
				[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				$AddressesFree = $Statistics.AddressesFree
				$AddressesInUse = $Statistics.AddressesInUse
				if( $Ipv6State -eq "Active")
				{
					$Ipv6State = 1;
				}
				elseif( $Ipv6State -eq "Inactive")
				{
					$Ipv6State = 0;
				}
				
				Write-Host "Scopename : $($scopename ) $($delimiter) AddressesInUse : $($AddressesInUse) $($delimiter) AddressesFree : $($AddressesFree)  $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) state : $($Ipv6State)"
			}
			ElseIf(!$?)
			{
				$global:isErrorOccured = $True
				Write-Verbose "Error retrieving IPv6 scope statistics"
			}
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Verbose "Error retrieving IPv6 scopes"
	}
	$IPv6Scopes = $Null
	[gc]::collect() 
}
#endregion

#region script core
#Script begins
	$Script:DHCPServerName = "LocalHost"	
	ProcessIPv4SuperScopeStatistics
	ProcessIPv4ScopeStatistics
	ProcessIPv4MulticastScopesStatistics
	ProcessIPv6ScopesStatistics
#endregion