

const Queue = require('./queue');
const logger = require('../logger');
const Socket = require('./socket-client');
const utils = require('./collector-utils');
const Config = require('./../default_config');

const appendMsg = 'Communicator:';

function Communicator(collector) {
  this.queue = {};
  this.timer = null;
  this.isStarted = false;
  this.socket = new Socket();
  this.collector = collector;

  this.socket.event.on('stopTracing', _ => {
    if (this.queue.size() > 0) {
      logger.debug(appendMsg, `${this.queue.size()} data is going to discard`);
      this.queue.reset();
    }
    
    this.collector.agent.stopTracing();
    _stopTimer.call(this);
  });

  this.socket.event.on('startTracing', _ => {
    this.collector.agent.startTracing();
    _startTimer.call(this);
  });

  this.socket.event.on('configChanged', newConfig => {
    this.collector.onConfigChange(newConfig);
  });

  this.socket.event.on('onNewComponent', newComponentDetails => {
    this.collector.onNewComponent(newComponentDetails);
  });
}
module.exports = Communicator;

Communicator.prototype.start = function (config, runTimeInfo) {
  this.config = config;
  this.ip = config && config.agent_host;
  this.port = config && config.agent_port;

  if (!this.ip || !this.port) {
    logger.error(appendMsg, 'not a valid socket ip or port');
    return;
  }

  this.queue = new Queue(config);
  this.socket.start(config, runTimeInfo);
  _startTimer.call(this);
  this.isStarted = true;
}

function _startTimer() {
  if (this.timer) return;
  const interval = this.config.flush_interval || Config.flush_interval;
  this.timer = setInterval(() => {
    if (this.queue.size() <= 0) return;
    _sendData.call(this);
  }, interval);

  this.timer.unref();
}

function _stopTimer() {
  if (this.timer) {
    clearInterval(this.timer);
    this.timer = null;
  }
}

function _sendData() {
  utils.sendDataToSocket(this.socket, this.queue, this.collector.agent.config, appendMsg);
}

Communicator.prototype.add = function (data) {
  if (!this.isStarted) return;
  if (data.sendImmediately) {
    if (!this.socket || !this.socket.isConnected) {

      return;
    }

    if (data.type === 'String' && data.message) {
      this.socket.send(data.message, true, true);
    } else {
      delete data.sendImmediately;
      this.socket.send([data], true);
    }
    return true;
  }

  this.queue.enqueue(data);

  /** if no batch size is given then the app will send one by one immediattly  */
  if (!this.config.max_batch_size) {
    _sendData.call(this);
  }
}

Communicator.prototype.sendFile = function (type, payload) {
  this.socket.sendFile(type, payload);
}

Communicator.prototype.stop = function () {
  this.isStarted = false;
  _stopTimer.call(this);
  this.socket.stop();
}

Communicator.prototype.isReadyToTransmit = function () {
  if (!this.isStarted) return false;
  return this.socket.isReadyToTransmit();
}

Communicator.prototype.resetConnection = function () {
  if (!this.isStarted) return;
  this.socket.resetConnection();
}

Communicator.prototype.setConfig = function (config) {
  this.socket.setConfig(config);
}

Communicator.prototype.setTimeout = function () {
  this.socket.setTimeout();
}