// ignore_for_file: constant_identifier_names

import '../model/log_level.dart';
import 'string_utils.dart';

/// Type of the log message
enum LogType {
  /// Info Log level
  Info,

  /// Error log level
  Error,

  /// Warning log level
  Warning
}

/// Internal logger class
class Logger {
  static Logger? _instance;
  LogLevel? _definedLogLevel;

  /// Factory for internal logger class
  factory Logger() {
    _instance ??= Logger.private();

    return _instance!;
  }

  /// Private constructor for internal logger class
  Logger.private() {
    _definedLogLevel = LogLevel.Info;
  }

  /// Set the global [logLevel]
  set logLevel(LogLevel logLevel) => _definedLogLevel = logLevel;

  /// Is logging a debug log. This will only be logged if
  /// the customer activated the debug logs
  void d(String message, {LogType logType = LogType.Info}) {
    if (_definedLogLevel == LogLevel.Debug) {
      _logMessage(logType, message);
    }
  }

  /// Is logging a info log. This log will always be displayed.
  /// So be careful when using it
  void i(String? message, {LogType logType = LogType.Info}) {
    _logMessage(logType, message);
  }

  /// Logging a [message] with a certain [logLevel]
  void _logMessage(LogType logType, String? message) {
    if (!StringUtils.isStringNullEmptyOrWhitespace(message)) {
      // ignore: avoid_print
      print(
          "[${_currentDate()}][${_getLogTypeString(logType)}][DYNATRACE]: $message");
    }
  }

  String _getLogTypeString(LogType logType) {
    if (logType == LogType.Error) {
      return "ERROR";
    } else if (logType == LogType.Warning) {
      return "WARNING";
    } else {
      return "INFO";
    }
  }

  String _currentDate() {
    String date = DateTime.now().toIso8601String().replaceFirst("T", " ");
    return date.substring(0, date.indexOf("."));
  }
}
