import 'dart:io' show Platform, HttpOverrides;

import 'package:eginnovations/src/http/http_overrides.dart';
import 'package:eginnovations/src/monitors.dart';
import 'package:eginnovations/src/request_receiver.dart';
import 'package:flutter/services.dart' show MethodCall, MethodChannel;
import 'package:eginnovations/src/navigation_observer.dart';

/// This class needed to aggregate eG properties and arguments.
class Config {
  /// API_KEY of your iOS Application.
  ///
  /// Property is optional if your app don't support iOS.
  /// Your can find API_KEY on settings page of your Application
  ///
  /// If you don't specify it and try to launch the app on iOS
  /// SDK will throw ArgumentError
  String? iosApiKey;

  /// API_KEY of your Android Application.
  ///
  /// Property is optional if your app don't support Android.
  /// Your can find API_KEY on settings page of your Application
  ///
  /// If you don't specify it and try to launch the app on Android
  /// SDK will throw ArgumentError
  String? androidApiKey;

  /// Collection of metadata information
  ///
  /// Property is optional. It allows to attach some additional
  /// information to session. For example, you can specify device name by
  /// putting it with MetaDataKeys.deviceName key
  Map<String, String>? metadata;

  /// List of monitor which will be enabled
  ///
  /// Property is optional. By default all available monitors will be enabled.
  /// E.g. to enable necessary monitors you need to provide list
  /// like [Monitors.http, Monitors.screenshot]
  List<Monitor>? monitors;
}

/// This is the main class for using eG. eG captures various
/// types of data to assist in debugging, analyzing application state and
/// understanding user behavior.
/// <p>
/// <p>Here is an example of how eG is used:
/// <pre>
/// void main() {
///   runeGInnovaions();
///   runApp(MyApp());
/// }
///
/// void runeGInnovaions() {
///   var config = new Config();
///   config.iosApiKey = "Your iOS API_KEY";
///   config.androidApiKey = "Your Android API_KEY";
///   EginnovationsPlugin.run(config);
/// }
/// </pre></p>

class EginnovationsPlugin {
  static EginnovationsPlugin _eGPlugin =
      EginnovationsPlugin._privateConstructor();

  static const MethodChannel _channl = MethodChannel('eginnovations_plugin');
  final MethodChannel _channel = const MethodChannel('eginnovations_plugin');
  final RequestReceiver _requestReceiver = new RequestReceiver();
  Function(String)? _sessionUrlListener;

  set sessionUrlListener(Function(String)? listener) {
    _sessionUrlListener = listener;
  }

  EginnovationsPlugin._privateConstructor();

  EginnovationsPlugin._withConfig(
      Config config, Function(String)? sessionUrlListener) {
    HttpOverrides.global = eGHttpOverrides();
    _requestReceiver.observeChannel();
    _channel.setMethodCallHandler(_handlePluginCalls);
    _sessionUrlListener = sessionUrlListener;
    _eGPlugin = this;
  }

  Future<dynamic> _init(Config config) {
    final monitors = config.monitors ?? Monitors.all();
    if (Platform.isAndroid) {
      ArgumentError.checkNotNull(config.androidApiKey, "androidApiKey");
      return _initeG(
          config.androidApiKey,
          _filterByPlatform(monitors, SupportedPlatform.android),
          config.metadata);
    } else if (Platform.isIOS) {
      ArgumentError.checkNotNull(config.iosApiKey, "iosApiKey");
      return _initeG(config.iosApiKey,
          _filterByPlatform(monitors, SupportedPlatform.ios), config.metadata);
    } else {
      return Future.error("eG doesn't support current platform");
    }
  }

  Future<dynamic> _handlePluginCalls(MethodCall methodCall) async {
    if (methodCall.method == "onSessionUrl") {
      _sessionUrlListener?.call(methodCall.arguments);
    }
  }

  /// Returns shared instance of SDK plugin
  static EginnovationsPlugin shared() => _eGPlugin;

  /// Method for starting eG with supplied configs
  static Future<dynamic> run(Config config) async {
    final sharedInstance = shared();
    final isStarted = await sharedInstance.isStarted();
    if (!isStarted) {
      return new EginnovationsPlugin._withConfig(
              config, sharedInstance._sessionUrlListener)
          ._init(config);
    }
  }

  _initeG(String? apiKey, Iterable<Monitor> monitors,
          Map<String, String>? metadata) =>
      _channel.invokeMethod("run", {
        "apiKey": apiKey,
        "enabledMonitors": monitors.map((m) => m.id).toList(),
        "metadata": metadata
      });

  static Future<void> reportScreenChange(String screenName) async {
    final List<dynamic> params = <dynamic>[screenName];
    await _channl.invokeMethod<Object>('reportScreenChange', params);
  }

  // static Future<void> reportScreenChange(String screenName) =>
  //     _channels.invokeMethod("reportScreenChange:", screenName);

  /// Stop all monitors and events sending
  Future<void> stop() => _channel.invokeMethod("stop");

  /// Resume sending events and work of all monitors
  Future<void> start() => _channel.invokeMethod("start");

  /// Returns true if sdk is started
  Future<bool> isStarted() =>
      _channel.invokeMethod("isStarted").then((value) => value ?? false);

  /// Set metadata value
  Future<void> setMetadataValue(String key, String value) =>
      _channel.invokeMethod("setMetadata", {"key": key, "value": value});

  /// Remove metadata value
  Future<void> removeMetadataValue(String key) =>
      _channel.invokeMethod("removeMetadata", {"key": key});

  ///Reports that the screen has been changed (repro steps)
  ///[screenName] String containing the screen name
  // Future<void> reportScreenChange(String key) =>
  // {
  //   final List<dynamic> params = <dynamic>[screenName];
  //   await _channel.invokeMethod<Object>('reportScreenChange:', params);
  // _channel.invokeMethod("reportScreenChange", {"key": key});
  // }

  Iterable<Monitor> _filterByPlatform(
      List<Monitor> monitors, SupportedPlatform platform) {
    return monitors.where((m) => m.platforms.contains(platform));
  }
}

/// Identifiers for supported metadata keys
/// Sdk provides opportunity to send additional session information
///
/// For more information see Config.metadata method
class MetadataKeys {
  MetadataKeys._();

  /// Supported key to change device name
  static const deviceName = "userSpecifiedDeviceName";
}
