'use strict'

const fs = require('fs');
const os = require('os');
const path = require('path');

const na = 'Not available';
const appendMsg = 'Start:';
global.preStartupLogs = global.preStartupLogs || [];

let info = {
  pid: process.pid,
  targetAppPackageJson: null,
  systemUptime: process.uptime(),

  // NodeJS info
  nodeVersion: process.versions && process.versions.node || na,
  nodejsLTS: process.release && process.release.lts || na,
  v8Version: process.versions && process.versions.v8 || na,
  libuvVersion: process.versions && process.versions.uv || na,
};

try {
  // OS info
  info.osType = os.type && os.type() || na; //'Windows_NT || Linux || Darwin' 
  info.osArchitecture = os.arch && os.arch() || na; //64 or 32 bit
  info.osPlatform = os.platform && os.platform() || na; //'darwin' 'freebsd' 'linux' 'openbsd' 'sunos' 'win32'
  info.hostName = os.hostname();
  info.ip = _getIPAddress();
} catch (e) {
  console.error(e);
}

function setBasePath(basePath) {
  global.preStartupLogs.push({
    appendMsg,
    msg: 'Initial Base Path: ' + basePath,
  });
  _setTagetAppDetails(basePath);
  basePath = info.targetAppRootPath;

  if (basePath) {
    try {
      fs.accessSync(basePath, fs.W_OK);
      info.homeDirectory = path.normalize(basePath + '/eGNodeMonitor');
      info.logDirectory = path.normalize(info.homeDirectory + '/logs');
      _createNecessaryFolders();
      return;
    } catch (e1) {
      global.preStartupLogs.push({
        appendMsg,
        type: 'error',
        msg: "Don't have the read permission to the Node.js Application Folder",
      });
    }
  }

  try {
    const _path = path.normalize(__dirname + '/..');
    fs.accessSync(_path, fs.constants.F_OK);
    info.homeDirectory = path.normalize(_path + '/home-dir');
    info.logDirectory = path.normalize(info.homeDirectory + '/logs');
    _createNecessaryFolders();
    return;
  } catch (e) {
    global.preStartupLogs.push({
      appendMsg,
      type: 'warn',
      msg: "Don't have the read permission to the Node.js Monitor Folder",
    });
  }

  try {
    info.homeDirectory = fs.realpathSync(os.tmpdir());
    if (!info.homeDirectory) throw e;

    info.homeDirectory = path.normalize(info.homeDirectory + '/eGNodeMonitor');
    info.logDirectory = path.normalize(info.homeDirectory + '/logs');
    _createNecessaryFolders();
  } catch (e) {
    console.error(e);
    throw new Error("Can't able to set a eG home directive path");
  }
}

function _getIPAddress() {
  const interfaces = os.networkInterfaces();
  for (let devName in interfaces) {
    const iface = interfaces[devName];

    for (let i = 0; i < iface.length; i++) {
      const alias = iface[i];
      if (alias.family === 'IPv4' && alias.address !== '127.0.0.1' && !alias.internal)
        return alias.address;
    }
  }

  return null;
}

function _setTagetAppDetails(basePath) {
  let targetAppPackageJson = _getPackageJson(basePath);
  if (targetAppPackageJson && targetAppPackageJson.name === 'eg-node-monitor') return null;
  if (targetAppPackageJson) return _setValues();

  basePath = path.normalize(basePath + path.sep + '..');
  targetAppPackageJson = _getPackageJson(basePath);
  if (targetAppPackageJson) return _setValues();

  basePath = path.normalize(basePath + path.sep + '..');
  targetAppPackageJson = _getPackageJson(basePath);
  if (targetAppPackageJson) return _setValues();

  function _setValues() {
    info.targetAppPackageJson = targetAppPackageJson;
    info.targetAppRootPath = basePath;
  }
}

function _getPackageJson(_path) {
  try {
    return require(_path + '/package.json');
  } catch (e) { }
}

function _createNecessaryFolders() {
  _ensureDirectoryExistence(info.logDirectory);

  global.preStartupLogs.push({
    appendMsg,
    msg: `${info.homeDirectory} is used for eG logs and data.`,
  });
}

function _ensureDirectoryExistence(filePath) {
  if (!filePath) throw new Error("File Path not found...!");
  if (fs.existsSync(filePath)) return true;
  fs.mkdirSync(filePath, { recursive: true });
}

info.setBasePath = setBasePath;
module.exports = info;