'use strict'

const semver = require('semver');
var { URL, urlToHttpOptions } = require('url');

const logger = require('../logger');
const appConstant = require('../app-constants');

const appendMsg = 'Http-Outwards:';
const eGGUID = appConstant.GUID_NAME;
const nodeHttpRequestSupportsSeparateUrlArg = semver.gte(process.version, '10.9.0');

/**
 * safeUrlToHttpOptions is a version of `urlToHttpOptions` -- available in
 * later Node.js versions (https://nodejs.org/api/all.html#all_url_urlurltohttpoptionsurl)
 * -- where the returned object is made "safe" to use as the `options` argument
 * to `http.request()` and `https.request()`.
 *
 * By "safe" here we mean that it will not accidentally be considered a `url`
 * argument. This matters in the instrumentation below because the following are
 * handled differently:
 *      http.request(<options>, 'this is a bogus callback')
 *      http.request(<url>, 'this is a bogus callback')
 */
let safeUrlToHttpOptions;
if (!urlToHttpOptions) {
  // Adapted from https://github.com/nodejs/node/blob/v18.13.0/lib/internal/url.js#L1408-L1431
  // Added in: v15.7.0, v14.18.0.
  safeUrlToHttpOptions = function (url) {
    const options = {
      protocol: url.protocol,
      hostname: typeof url.hostname === 'string' &&
        String.prototype.startsWith(url.hostname, '[')
        ? String.prototype.slice(url.hostname, 1, -1)
        : url.hostname,
      hash: url.hash,
      search: url.search,
      pathname: url.pathname,
      path: `${url.pathname || ''}${url.search || ''}`,
      href: url.href
    }
    if (url.port !== '') {
      options.port = Number(url.port)
    }
    if (url.username || url.password) {
      options.auth = `${decodeURIComponent(url.username)}:${decodeURIComponent(url.password)}`
    }
    return options
  }
} else if (semver.satisfies(process.version, '>=19.9.0 <20')) {
  // Starting in node v19.9.0 (as of https://github.com/nodejs/node/pull/46989)
  // `urlToHttpOptions(url)` returns an object which is considered a `url`
  // argument by `http.request()` -- because of the internal `isURL(url)` test.
  safeUrlToHttpOptions = function (url) {
    const options = urlToHttpOptions(url)
    // Specically we are dropping the `Symbol(context)` field.
    Object.getOwnPropertySymbols(options).forEach(sym => {
      delete options[sym]
    })
    return options
  }
} else if (semver.satisfies(process.version, '>=20', { includePrerelease: true })) {
  // This only works for versions of node v20 after
  // https://github.com/nodejs/node/pull/47339 which changed the internal
  // `isURL()` to duck-type test for the `href` field. `href` isn't an option
  // to `http.request()` so there is no harm in dropping it.
  safeUrlToHttpOptions = function (url) {
    const options = urlToHttpOptions(url)
    delete options.href
    return options
  }
} else {
  safeUrlToHttpOptions = urlToHttpOptions
}

exports.trace = function (agent, moduleName, method) {
  const ins = agent._instrumentation

  return function (orig) {
    return function (input, options, cb) {
      let id = null, span = null, newArgs = null;
      logger.debug(appendMsg, `new ${moduleName}.${method} Oubound call`);

      try {

        // Reproduce the argument handling from node/lib/_http_client.js#ClientRequest().
        //
        // The `new URL(...)` calls in this block *could* throw INVALID_URL, but
        // that would happen anyway when calling `orig(...)`. The only slight
        // downside is that the Error stack won't originate inside "_http_client.js".
        if (!nodeHttpRequestSupportsSeparateUrlArg) {
          // Signature from node <10.9.0:
          //    http.request(options[, callback])
          //        options <Object> | <string> | <URL>
          cb = options
          options = input
          if (typeof options === 'string') {
            options = safeUrlToHttpOptions(new URL(options))
          } else if (options instanceof URL) {
            options = safeUrlToHttpOptions(options)
          } else {
            options = Object.assign({}, options)
          }
        } else {
          // Signature from node >=10.9.0:
          //    http.request(options[, callback])
          //    http.request(url[, options][, callback])
          //        url <string> | <URL>
          //        options <Object>
          if (typeof input === 'string') {
            input = safeUrlToHttpOptions(new URL(input))
          } else if (input instanceof URL) {
            input = safeUrlToHttpOptions(input)
          } else {
            cb = options
            options = input
            input = null
          }

          if (typeof options === 'function') {
            cb = options
            options = input || {}
          } else {
            options = Object.assign(input || {}, options)
          }
        }

        newArgs = [options]
        if (cb !== undefined) {
          newArgs.push(cb)
        }

        let port = options.port;
        port = port ? ':' + port : '';
        const protocol = moduleName !== 'http' ? 'https' : 'http';
        const uri = (options.protocol || protocol) + '://' + (options.host || options.hostname || 'localhost') + port + options.path;
        span = agent.startSpan(options.method + ' ' + uri, 'http:outbound', 'Http');
        if (!span) return orig.apply(this, arguments);

        id = span.transaction.id;
        if (id) {
          options.headers = options.headers || {}
          options.headers[eGGUID] = span.getNeweGGUID();
        }

        span.options = {};
        span.options.uri = uri;
        span.options.nodeOrder = span.nodeOrder;
        span.options.resTime = 0;
      } catch (e) {
        logger.error(appendMsg, 'trace', e);
      }

      let req = null;
      try {
        req = orig.apply(this, newArgs);
      } catch (e) {
        onError(e);
        throw e;
      }
      if (!span) return req;

      span.options && (span.options.method = req.method);
      ins.bindEmitter(req);
      const emit = req.emit;

      req.emit = function wrappedEmit(type, res) {
        if (type === 'response') onResponse(res);
        if (type === 'close') onClose();
        if (type === 'error') onError(res);
        return emit.apply(req, arguments);
      }

      return req;

      //to capture external error
      function onError(err) {
        if (err) {
          logger.debug(appendMsg, 'error captured at ' + method);
          span.captureError(err);
        }

        span.end();
      }

      function onClose() {
        if (span.ended) return;
        span.end();
      }

      function onResponse(res) {
        logger.debug(appendMsg, 'Response came for http oubound request', {
          id: id,
          url: span && span.options && span.options.uri
        });
        ins.bindEmitter(res);

        if (res.prependListener) {
          // Added in Node.js 6.0.0
          res.prependListener('end', onEnd);
        } else {
          var existing = res._events && res._events.end;
          if (!existing) {
            res.on('end', onEnd);
          } else {
            if (typeof existing === 'function') {
              res._events.end = [onEnd, existing];
            } else {
              existing.unshift(onEnd);
            }
          }
        }

        function onEnd() {
          const headGuid = appConstant.RES_HEADER_GUID;
          const hostGuid = res.headers ? res.headers[headGuid] || res.headers[headGuid.toLowerCase()] : '';
          if (hostGuid) {
            logger.debug(appendMsg, 'The eG is enabled in the destination server and Guid ' + hostGuid);
            span.options.tftKey = hostGuid;
          }

          logger.debug(appendMsg, 'http outbound request event is completed id', id);
          span.options.statusCode = res.statusCode || null;
          span.end();
        }
      }
    }
  }
}