'use strict'

const shimmer = require('../shimmer');
const logger = require('../../logger');
const httpShared = require('../http-shared');
const httpOutwards = require('../http-outwards');

const appendMsg = 'Http:'

function getSafeHeaders(res) {
  return res.getHeaders ? res.getHeaders() : res._headers;
}

exports.start = function (http, agent, version, enabled) {
  if (!enabled) return http;

  try {
    if (http && http.Server && http.Server.prototype) {
      logger.debug(appendMsg, 'wrapping http.Server.prototype.emit function');
      shimmer.wrap(http.Server.prototype, 'emit', httpShared.instrumentRequest(agent, 'http'));
    }

    logger.debug(appendMsg, 'wrapping http.request function')
    shimmer.wrap(http, 'request', httpOutwards.trace(agent, 'http', 'request'))

    logger.debug(appendMsg, 'wrapping http.get function')
    shimmer.wrap(http, 'get', httpOutwards.trace(agent, 'http', 'get'))

    if (http && http.ServerResponse && http.ServerResponse.prototype) {
      logger.debug(appendMsg, 'wrapping http.ServerResponse.prototype.writeHead function');
      shimmer.wrap(http.ServerResponse.prototype, 'writeHead', wrapWriteHead);
    }

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return http;

  function wrapWriteHead(original) {
    return function wrappedWriteHead() {
      let result = null;
      try {
        const headers = arguments.length === 1
          ? getSafeHeaders(this) // might be because of implicit headers
          : arguments[arguments.length - 1]
        const trans = httpShared.getTransaction(this);
        result = original.apply(this, arguments);

        if (trans) {
          // It shouldn't be possible for the statusCode to be falsy, but just in
          // case we're in a bad state we should avoid throwing
          trans.requestIdentifier.statusCode = this.statusCode || null;
          trans.requestIdentifier.statusMessage = this.statusMessage || "";

          // End transacton early in case of SSE
          if (_isValidHeader(headers)) {
            Object.keys(headers).some(function (key) {
              if (key.toLowerCase() !== 'content-type') return false;
              if (String(headers[key]).toLowerCase().indexOf('text/event-stream') !== 0) return false;
              logger.debug(appendMsg, 'detected SSE response - ending transaction', { id: trans.id });
              trans.end();
              return true;
            });
          }
        }
      } catch (e) {
        logger.error(appendMsg, 'wrapWriteHead', e);
      }
      return result;
    }
  }

  function _isValidHeader(headers) {
    return headers && typeof headers === 'object' && !Array.isArray(headers);
  }
}

exports.stop = function (http, version) {
  if (http && http.Server && http.Server.prototype) {
    shimmer.unwrap(http.Server.prototype, 'emit');
  }

  shimmer.unwrap(http, 'request')
  shimmer.unwrap(http, 'get')

  if (http && http.Server && http.Server.prototype) {
    shimmer.unwrap(http.ServerResponse.prototype, 'writeHead');
  }

  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}
