'use strict'

const url = require('url');
const symbols = require('./symbols');

const basicAuth = require('basic-auth');

function parseError(err, maxCauseDepth, captureCause) {
  const error = getErrorObj(err, maxCauseDepth, captureCause);
  if (!error || !error.err) return;

  error.message = error.err.message ? format(error.err.message) : '';
  error.stack = error.err.stack ? error.err.stack.toString() : '';
  return error;
}

function getErrorObj(err, maxCauseDepth, captureCause) {
  if (!err) return;

  if (err[symbols.errorParsedSymbol] === true) {
    return { err };
  }

  if (err instanceof Error) {
    const type = err.type = _getErrorType(err);
    const causes = getCauses(err, maxCauseDepth, captureCause);
    return { err, type, causes };
  }

  const frames = (err.stack || err).toString().split(' at ');
  if (frames.length === 1) {
    return {
      type: err.type || 'Error',
      err: { message: err.toString() },
    };
  }

  const newError = {
    message: frames[0],
    type: err.type || 'Error',
    stack: (err.stack || err).toString(),
  };
  return {
    err: newError,
    type: newError.type,
    causes: getCauses(newError, maxCauseDepth, captureCause),
  };
}

function _getErrorType(err) {
  return err.constructor && err.constructor.name || err.name;
}

function getCauses(err, maxCauseDepth, captureCause) {
  const causes = _getCauses(err, maxCauseDepth, captureCause);
  causes.size = Object.keys(causes).length;
  return causes;
}

function _getCauses(err, maxCauseDepth, captureCause) {
  const causes = {};
  _set('rfqcn', err);
  const tempCauses = [];
  let bottomError = err;

  for (let i = 0; i < maxCauseDepth - 1; i++) {
    if (!bottomError.cause) break;
    bottomError = bottomError.cause;

    tempCauses.push({
      type: _getErrorType(bottomError),
      message: bottomError.message
    });
  }

  if (!tempCauses.length) {
    _set('bfqcn', err);
    return causes;
  }

  const lastIndex = tempCauses.length - 1;
  _set('bfqcn', tempCauses[lastIndex]);
  if (!captureCause || !lastIndex) return causes;

  if (tempCauses[0]) _set('rcause_1', tempCauses[0]);
  if (tempCauses[1]) _set('rcause_2', tempCauses[1]);
  if (lastIndex > 3) _set('bcause_2', tempCauses[lastIndex - 2]);
  if (lastIndex > 2) _set('bcause_1', tempCauses[lastIndex - 1]);
  return causes;

  function _set(key, value) {
    causes[key] = value.type;
    causes[key + '_msg'] = value.message;
  }
}

function format(num) {
  return (num < 10 ? '0' : '') + num;
};

function getUserContextFromReq(req) {
  if (!req) return;
  const user = req.user || basicAuth(req) || req.session;
  if (!user && !req.sessionID) return;

  const context = {};
  if (isNumOrStr(req.sessionID)) context.sessionID = req.sessionID;

  if (!user) return context;
  if (isNumOrStr(user.id)) context.id = user.id;
  else if (isNumOrStr(user._id)) context.id = user._id;

  if (isStr(user.username)) context.username = user.username;
  else if (isStr(user.name)) context.username = user.name;

  isStr(user.email) && (context.email = user.email);
  return context;
}

function isNumOrStr(data) {
  return isStr(data) || (data != null && typeof data === 'number');
}

function isStr(data) {
  return data && typeof data === 'string';
}

function parseUrl(urlStr) {
  return new url.URL(urlStr, 'relative:///')
}

exports.parseUrl = parseUrl;
exports.parseError = parseError;
exports.getUserContextFromReq = getUserContextFromReq;