'use strict'

const path = require('path');
const threads = require('./lib/utils/threads');

// Don't inject our agent for workers thread.
if (!threads.isMainThread) return;

let pm2Details = null;
const startUpFiles = [];
const appendMsg = 'Start:';
global.preStartupLogs = [];
let appFileToIgnore = process.env.EG_APP_FILE_TO_IGNORE;
appFileToIgnore = appFileToIgnore && appFileToIgnore.split(',') || [];

if (process.argv && process.argv.length) {
  const pm2Path = path.sep + 'pm2';
  const npmPath = path.sep + 'npm-cli.js';
  const pm2Daemon = `${path.sep}pm2${path.sep}lib${path.sep}Daemon.js`;

  for (let i = 0; i < process.argv.length; i++) {
    const arg = process.argv[i];
    const len = arg.length;
    startUpFiles.push(path.basename(arg));

    if (arg.substring(len - npmPath.length, len) === npmPath) {
      // exit immediately if the process is npm
      return;
    }

    if (appFileToIgnore.indexOf(path.basename(arg)) > -1) {
      // exit immediately if the app needs to ignore
      return;
    }

    if (arg.substring(len - 4, len) === pm2Path) {
      pm2Details = {};
      pm2Details.path = arg;
      break;
    }

    if (arg.substring(len - 18, len) === pm2Daemon) {
      pm2Details = {};
      pm2Details.path = arg;
      pm2Details.isDaemon = true;
      break;
    }
  }
}

let isStartedUsingPm2 = !!process.env.pm_exec_path || !!process.env.pm2_env;

if (!isStartedUsingPm2) {
  if (module.parent.paths && module.parent.paths.length) {
    global.basePath = path.dirname(module.parent.paths[0]);
  }

  if (global.basePath && process.argv && process.argv.length) {
    for (let i = 0; i < process.argv.length; i++) {
      if (process.argv[i].indexOf(global.basePath) > -1) {
        global.startingFile = process.argv[i];
        break;
      }
    }
  }

  if (global.startingFile.indexOf('/node_modules/') > -1) {
    // exit immediately if the process is started by some node_modules because it could be some child process.
    return;
  }
} else {
  global.basePath = process.env.pm_cwd;
  global.startingFile = process.env.pm_exec_path;
  pm2Details = {};
  pm2Details.isTragetApp = true;
  pm2Details.id = process.env.pm_id;
  pm2Details.errLogPath = process.env.pm_err_log_path;
}

const appInfo = require('./lib/app-info');
appInfo.pm2Details = pm2Details;
const agent = require('./index.js');
if (!agent || !agent.start) return;

const logger = require('./lib/logger');
const config = require('./lib/default_config');
const configKeys = Object.keys(config);

const options = {};
logger.info(appendMsg, 'Eg profiler is starting without snippet code..!');
_setOptionFromEnvVariable(process.env);
_logOptions('Options from the Enviroment variables');

if (isStartedUsingPm2 && process.env.pm2_env) {
  logger.info('The application is started using PM2');
  let pm2Env = null;
  try {
    pm2Env = JSON.parse(process.env.pm2_env);
  } catch (e) {
    logger.error(appendMsg, 'PM2 Enviroment variables parse error', e);
  }

  if (pm2Env.env) {
    _setOptionFromEnvVariable(pm2Env.env);
    _logOptions('Options from the PM2 Enviroment variables');
  }

  if (!global.startingFile && pm2Env.env.pm_exec_path) {
    global.startingFile = pm2Env.env.pm_exec_path;
  }
}

if (process.argv && process.argv.length) {
  process.argv.forEach(argv => {
    if (argv.substring(0, 5) !== '--eg_') return;
    const keys = argv.split("=");
    _setOptions(keys[0], keys[1], 5);
  });
  logger.info('Node.js application startup argument', process.argv);
}

agent.start(options);

function _logOptions(msg) {
  if (Object.keys(options).length < 1) return;
  logger.info(msg, options);
}

function _setOptionFromEnvVariable(env) {
  Object.keys(env).forEach(key => {
    if (key.substring(0, 3) !== 'EG_') return;
    _setOptions(key, env[key], 3);
  });
}

function _setOptions(key, value, prefix) {
  const okey = key.substring(prefix, 100).toLocaleLowerCase();
  if (configKeys.indexOf(okey) === -1) return;
  options[okey] = value;
  if (!isNaN(options[okey])) options[okey] = parseInt(options[okey]);
  else if (options[okey] == 'true') options[okey] = true;
  else if (options[okey] == 'false') options[okey] = false;
}

module.exports = agent;