$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Set-StrictMode -Version 2
$ErrorActionPreference = 'SilentlyContinue'
Function YesNo ([boolean] $condition)
{
    if ($condition -eq $True)
    {
	    return "Yes"
    }
    Else
    {
	    return "No"
    }
}
Function TranslatePasswordOptions($pwKey)
{
	Switch($pwKey)
    {
		"Always" 	{$pwValue = "At any time"}
		"ExpiredOnly" 	{$pwValue = "When expired"}
		"Never" 	{$pwValue = "Never"}
		Default		{$pwValue = $pwKey}
	}
	return $pwValue
}
Function TranslateMethod($methodKey)
{
	Switch($methodKey)
	{
		"ExplicitForms" 	{$methodValue = "User name and password"}
		"IntegratedWindows"	{$methodValue = "Domain pass-through"}
		"CitrixAGBasic"		{$methodValue = "Pass-through from NetScaler Gateway"}
		"HttpBasic"		{$methodValue = "HTTP Basic"}
		"Certificate"		{$methodValue = "Smart card"}
		Default			{$methodValue = "(Unknown authentication method)"}
	}
	return $methodValue
}
Function TranslateNSVersion($versionKey)
{
	Switch($versionKey)
	{
		"Version10_0_69_4" 	{$versionValue = "10.0 (Build 69.4) or later"}
		"Version9x"		{$versionValue = "9.x"}
		"Version5x"		{$versionValue = "5.x"}
		Default			{$versionValue = $versionKey}
	}
	return $versionValue
}

Function TranslateHTML5Deployment($HTML5Key)
{
	Switch($HTML5Key)
	{
		"Fallback" 	{$HTML5Value = "Use Receiver for HTML5 if local install fails"}
		"Always"	{$HTML5Value = "Always use Receiver for HTML5"}
		"Off"		{$HTML5Value = "Citrix Receiver installed locally"}
		Default		{$HTML5Value = $HTML5Key}
	}
	return $HTML5Value
}

Function TranslateLogonType($logonKey)
{
	Switch($logonKey)
	{
		"DomainAndRSA" 	{$logonValue = "Domain and security token"}
		"Domain"	{$logonValue = "Domain"}
		"RSA"		{$logonValue = "Security token"}
		"SMS"		{$logonValue = "SMS authentication"}
		"SmartCard"	{$logonValue = "Smart card"}
		"None"		{$logonValue = "None"}
		Default		{$logonValue = $logonKey}
	}
	return $logonValue
}

#Add-snapins
#Exclude these snaps to avoid clash with modules
$excludedSnapins = @("Citrix.DeliveryServices.ConfigurationProvider","Citrix.DeliveryServices.ClusteredCredentialWallet.Install","Citrix.DeliveryServices.Workflow.WCF.Install" )
$availableSnapins = Get-PSSnapin -Name "Citrix.DeliveryServices.*" -Registered | Select -ExpandProperty "Name"
$loadedSnapins = Get-PSSnapin -Name "Citrix.DeliveryServices.*" -ErrorAction SilentlyContinue | Select -ExpandProperty "Name"
foreach ($snapin in $availableSnapins)
{
	if (($excludedSnapins -notcontains $snapin) -and ($loadedSnapins -notcontains $snapin))
	{
		Add-PSSnapin -Name $snapin
	}
}


#Import-modules
$dsInstallProp = Get-ItemProperty -Path HKLM:\SOFTWARE\Citrix\DeliveryServicesManagement -Name InstallDir
$dsInstallDir = $dsInstallProp.InstallDir
$dsModules = Get-ChildItem -Path "$dsInstallDir\Cmdlets" | Where { $_.FullName.EndsWith('psm1') } | foreach { $_.FullName }
foreach ($dsModule in $dsModules)
{
    Import-Module $dsModule
}

$outputfile = $args[0]

$servername = $env:computername
"Server Info" | out-file $outputfile
"#------------" | out-file $outputfile -append
"ServerName: " + $servername | out-file $outputfile -append
$serverversion = (get-DSVersion).StoreFrontVersion
"ServerVersion: " + $serverversion  | out-file $outputfile -append
$SFVersion = $serverversion
# server group
"#-------------------------------" | out-file $outputfile -append
"Processing server group starts" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
$baseURL = $(get-DSHostBaseUrl).hostBaseUrl
"Base URL: " + $baseURL | out-file $outputfile -append
$clusterMemberCount = $(get-DSClusterMembersCount).MembersCount
"Number of servers: " + $clusterMemberCount | out-file $outputfile -append
if (($clusterMemberCount -gt 1) -and (($SFVersion) -ge 2.5) )
{
	$lastSource = (Get-DSClusterConfigurationUpdateState).UpdateState.LastSourceServerName
	"Last Source: " + $lastSource | out-file $outputfile -append
}
$clusterMemberNames = $(get-DSClusterMembersName).HostNames

ForEach($member in $clusterMemberNames) 
{
	If ($clusterMemberCount -gt 1)
	{
		$lastSync = $(Get-DSXdServerGroupConfigurationUpdateState $member).LastEndTime.ToString()
		"Server Name: " + $member + "`t`tSynchronization Status: " + $lastSync | out-file $outputfile -append
	}
}
"#-------------------------------" | out-file $outputfile -append
"Processing server group end" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append

# authentication
"Processing authentication starts" | out-file $outputfile -append
"#---------------------------------" | out-file $outputfile -append
$auth = $(Get-DSAuthenticationServicesSummary)
$tokenURL = $auth.TokenIssuerUrl + "/validate"
If($auth.UseHttps -eq $True)
{
	$status = "StoreFront using HTTPS."
	$daysToExpire = (New-TimeSpan -End $auth.IISCertificate.NotAfter).Days
}
Else
{
	$status = "StoreFront using HTTP not HTTPS."
	$daysToExpire = ""
}
"Authentication Methods" | out-file $outputfile -append
"~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

[int]$enabledMethods = 0
ForEach ($protocol in $auth.protocols)
{
	If($protocol.DisplayInConsole -eq $True)
	{
		$methodKey = $protocol.choice
		$method = TranslateMethod($methodKey)

		Switch($protocol.enabled)
		{
			$True	{$enabled = "Yes"; ++$enabledMethods}
			Default	{$enabled = "No"}
		}
		$method + ":`t"+ $enabled | out-file $outputfile -append
	}
}

"Authentication Service" | out-file $outputfile -append
"~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

"Token validation service: " + $tokenURL | out-file $outputfile -append
"Number of enabled methods: " + $enabledMethods | out-file $outputfile -append
"Status: " +$status | out-file $outputfile -append


"Configure Trusted Domains" | out-file $outputfile -append
"~~~~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

$domainInfo = Get-DSExplicitCommonDataModel ($auth.SiteID) $auth.VirtualPath
$defDomain = ($domaininfo.DefaultDomain).DefaultDomain
$changePW = TranslatePasswordOptions($domainInfo.AllowUserToChangePassword)
"Domain count: " + $domainInfo.Domains.Count | out-file $outputfile -append
$TrustedDomain = "-"
If($domainInfo.Domains.Count -gt 0)
{
	"Allow users to log on from: Trusted domains only" | out-file $outputfile -append
	"Default domain: " +$defDomain | out-file $outputfile -append
	if (($SFVersion) -ge 2.6)
	{
		$showDomains = YesNo($domainInfo.ShowDomainField)
		"ShowDomains: " + $showDomains | out-file $outputfile -append
	}
	else
	{
		"ShowDomains: -" | out-file $outputfile -append
	}
	ForEach($domain in $domainInfo.Domains)
	{
		$TrustedDomain = $TrustedDomain + "," + $domain
	}
	if($TrustedDomain.startsWith("-,"))
	{
		$TrustedDomain = $TrustedDomain.substring(2)
	}
}
Else
{
	"Allow users to logon from:  Any domain" | out-file $outputfile -append
	"Default domain: -" | out-file $outputfile -append
	"ShowDomains: -" | out-file $outputfile -append
}
"Trusted domains:" + $TrustedDomain | out-file $outputfile -append

"Allow users to change passwords: "+$changePW | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
"Processing authentication end" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append

# stores
"Processing stores starts" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
$accounts = @(((Get-DSGlobalAccounts).Accounts) | Sort Name)
foreach ($account in $accounts)
{
	$advertised = YesNo($account.Published)
	$store = Get-DSStoreServicesSummary | where {$_.FriendlyName -eq $account.Name}
	$friendlyName = $store.FriendlyName
	$URL = $store.Url
	if ($store.GatewayKeys.Count -gt 0)
	{
		$access = "Internal and external networks"
	}
	Else
	{
		$access = "Internal networks only"
	}

	if ($store.UseHttps -eq $True)
	{
		$status = "StoreFront using HTTPS"
	}
	Else
	{
		$status = "StoreFront using HTTP not HTTPS"
	}

	$locked = "-"
	$authenticated = "-"
	$filterTypes = "-"
	$filterKeywords = "-"
	$includeKeywords = "-"
	$excludeKeywords = "-"

	if (($SFVersion) -ge 2.5)
	{
		$locked = YesNo($store.IsLockedDown)
		$authenticated = YesNo(!$store.IsAnonymous)
		$filterTypes = Get-DSResourceFilterType $store.SiteID $store.VirtualPath
		$filterKeywords = Get-DSResourceFilterkeyword $store.SiteID $store.VirtualPath
		$includeKeywords = @($filterKeywords.Include)
		$excludeKeywords = @($filterKeywords.Exclude)
	}
	
	"General Details" | out-file $outputfile -append
	"~~~~~~~~~~~~~~~~" | out-file $outputfile -append
	"Store Name: " + $friendlyName | out-file $outputfile -append
	"Authenticated: " + $authenticated | out-file $outputfile -append
	"Advertised: " +$advertised | out-file $outputfile -append
	"Store URL: " + $URL | out-file $outputfile -append
	"Access: " +$access | out-file $outputfile -append
	"Status: " +$status | out-file $outputfile -append
	"Locked: " +$locked | out-file $outputfile -append

	" " | out-file $outputfile -append
	"Delivery Controllers" | out-file $outputfile -append
	"~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append
	$farmsets = @($store.Farmsets)
	foreach ($farmset in $farmsets)
	{
		$farms = @($farmset.Farms)
		foreach ($farm in $farms)
		{
			$farmName = $farm.FarmName 
			$farmType = $farm.FarmType
			$farmServers = $farm.Servers
			$transportType = $farm.TransportType
			$port = $farm.ServicePort
			$sslRelayPort = $farm.SSLRelayPort
			$loadBalance = YesNo($farm.LoadBalance)

			"Delivery Controller Name: " +$farmName | out-file $outputfile -append
			"Type: " +$farmType | out-file $outputfile -append
			"Servers: " + $farmServers | out-file $outputfile -append
			"XMLPort: " +$port | out-file $outputfile -append
		}
	}

	" " | out-file $outputfile -append
	"Integrate with Citrix Online" | out-file $outputfile -append
	"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

	$GoToMeeting = YesNo($store.IsGoToMeetingEnabled)
	$GotoWebinar = YesNo($store.IsGoToWebinarEnabled)
	$GoToTraining = YesNo($store.IsGoToTrainingEnabled)
	"GoToMeeting: " +$GoToMeeting | out-file $outputfile -append
	"GoToWebinar: " +$GotoWebinar | out-file $outputfile -append
	"GoToTraining: " +$GoToTraining | out-file $outputfile -append

	# remote access
	$defaultGW = ""
	$fullVPN = $False
	$vpnService = Get-DSGlobalService -ServiceRef "VPN_$($store.ServiceRef)"
	$NSGatewayAppliances = "-"
	if($vpnService.Service)
	{
		$allGws = (Get-DSGlobalGateways).Gateways
		foreach($gw in $vpnService.Service.GatewayRefs)
		{
			$vpnGws = $allGws | Where-Object { $_.ID -eq $gw.RefId }
			$Script:fullVPN = $vpnService.Service.ServiceType -eq "VPN"
			if ($gw.Default) {$defaultGW = $vpnGws.Name}
			$NSGatewayAppliances = $NSGatewayAppliances + ", " + $($vpnGws.Name)
		}
	}
	else
	{
		$service = Get-DSGlobalService -ServiceRef $store.ServiceRef
		$allGws = (Get-DSGlobalGateways).Gateways
		foreach($gw in $service.Service.GatewayRefs)
		{
			$vpnGws = $allGws | Where-Object { $_.ID -eq $gw.RefId }
			$Script:fullVPN = $service.Service.ServiceType -eq "VPN"
			if ($gw.Default) {$defaultGW = $vpnGws.Name}
			$NSGatewayAppliances = $NSGatewayAppliances + ", " + $($vpnGws.Name)
		}
	}

	if($NSGatewayAppliances.startsWith("-,"))
	{
		$NSGatewayAppliances = $NSGatewayAppliances.substring(2)
	}

	"Netscaler Gateway Appliances: " + $NSGatewayAppliances | out-file $outputfile -append

	switch ($defaultGW)
	{
		""
		{
			"Default appliance: -" | out-file $outputfile -append
			"Remote access: None" | out-file $outputfile -append
		}
		default
		{
			"Default appliance: " +$defaultGW | out-file $outputfile -append
			if ($fullVPN)
			{
				"Remote access: Full VPN Tunnel" | out-file $outputfile -append
			}
			else
			{
				"Remote access: No VPN Tunnel"| out-file $outputfile -append
			}
		}
	}
	"XXXX----------------------------------------XXXX" | out-file $outputfile -append
}
"#------------------------------------------" | out-file $outputfile -append
"Processing stores end" | out-file $outputfile -append
"#------------------------------------------" | out-file $outputfile -append
# Receiver for Web
"Processing Receiver for Web sites starts" | out-file $outputfile -append
"#------------------------------------------" | out-file $outputfile -append
$receivers = @(Get-DSWebReceiversSummary)
foreach ($receiver in $receivers)
{
	$name = $receiver.FriendlyName
	$WebUrl  = $receiver.Url

	$ClassicExpr = "-"
	$authenticated = "-"
	$HTML5version = "-"
	$authMethods = "-"
	if (($SFVersion) -ge 2.5)
	{
		$authenticated = YesNo(!$receiver.IsAnonymousStore)
		$HTML5version = $receiver.HTML5Version
		$authMethods = @($receiver.AllowedAuthMethods)
	}

	if($receiver.ClassicEnabled)
	{
		$ClassicExpr = $receiver.ClassicEnabled
	}

	$storeURL = $receiver.StoreUrl
	$aStore = $storeURL -split "/"
	$store = $aStore[$aStore.Count-1]
	$deployment = TranslateHTML5Deployment($receiver.HTML5Configuration)
	$shortcuts = Get-DSAppShortcutsTrustedUrls $receiver.SiteId $receiver.VirtualPath

	"Name: " +$name | out-file $outputfile -append
	"Website URL: " +$WebUrl | out-file $outputfile -append
	"Classic Experience Enabled: "+ $ClassicExpr | out-file $outputfile -append
	"Store: " +$store | out-file $outputfile -append
	"Store URL: " + $storeURL | out-file $outputfile -append
	"Store Authenticated: "+$authenticated | out-file $outputfile -append
	"HTML5 Version: " +$HTML5version | out-file $outputfile -append

	# "------------------ " | out-file $outputfile -append
	# $name +"#~#"+ $WebUrl +"#~#"+ $store +"#~#"+ $storeURL +"#~#"+ $authenticated +"#~#"+ $HTML5version | out-file $outputfile -append
	# "------------------ " | out-file $outputfile -append

	if(($authMethods -ne $Null) -and ($authMethods.length -gt 0) -and ($authMethods -ne "-"))
	{
		$methods = ""
		foreach ($authMethod in $authMethods)
		{
			$method = TranslateMethod($authMethod)
			$methods = $methods +", "+$method
		}
		if($methods.startsWith(","))
		{
			$methods = $methods.substring(1)
		}
		"Authentication methods: " + $methods | out-file $outputfile -append
	}
	else
	{
		"Authentication methods: -" | out-file $outputfile -append
	}

	# $finalOutputLine = $finalOutputLine+ "#~#" + $methods
	# $finalOutputLine | out-file $outputfile -append

	if (($shortcuts -ne $Null))
	{
		$TrustedURLs = @($shortcuts.TrustedUrls)
		if($TrustedURLs)
		{
			"Shortcuts to Websites: " + $TrustedURLs | out-file $outputfile -append
		}else
		{
			"Shortcuts to Websites: -"| out-file $outputfile -append
		}

		if($deployment)
		{
			"Deployment: " +$deployment | out-file $outputfile -append
		}else
		{
			"Deployment: -" | out-file $outputfile -append
		}
		# $finalOutputLine = $finalOutputLine+ "#~#" + $TrustedURLs + "#~#" +$deployment
	}
	else
	{
		"Shortcuts equal to Null" | out-file $outputfile -append
		"Shortcuts to Websites: -" | out-file $outputfile -append
		"Deployment: -" | out-file $outputfile -append
	}
	"XXXX----------------------------------------XXXX" | out-file $outputfile -append
}
"#----------------------------------------" | out-file $outputfile -append
"Processing Receiver for Web sites ends" | out-file $outputfile -append
"#----------------------------------------" | out-file $outputfile -append

# NetScaler Gateways
"Processing NetScaler Gateways starts" | out-file $outputfile -append
"#--------------------------------------" | out-file $outputfile -append
$gateways = @((Get-DSGlobalGateways).Gateways)
foreach ($gateway in $gateways)
{
	$name = $gateway.Name
	$used = "Yes"
	$url = $gateway.Address
	$NSversion = TranslateNSVersion($gateway.AccessGatewayVersion)
	$callbackURL = $gateway.CallbackURL
	$deploymentMode = $gateway.DeploymentType
	$STAs = $gateway.SecureTicketAuthorityURLs
	if ($gateway.SessionReliability -eq $True) {$sessionReliability="Yes"} Else {$sessionReliability="No"}
	if ($gateway.RequestTicketTwoSTA -eq $True) {$request2STATickets="Yes"} Else {$request2STATickets="No"}
	"Display Name: "+$name | out-file $outputfile -append
	"Used by Services: " + $used | out-file $outputfile -append

	"URL: " +$url | out-file $outputfile -append
	"Version: " +$NSversion | out-file $outputfile -append

	if($NSversion -ne "5.x")
	{
		$logonType = TranslateLogonType($gateway.Logon)
		$smartCardFallback = TranslateLogonType($gateway.SmartCardFallback)
		$subnetIPaddress = $gateway.IPAddress
		"Subnet IP address: " + $subnetIPaddress | out-file $outputfile -append
		"Logon type: " + $logonType | out-file $outputfile -append
		"Smart card fallback:" + $smartCardFallback | out-file $outputfile -append
	}
	else
	{
		"Subnet IP address: -"| out-file $outputfile -append
		"Logon type: -" | out-file $outputfile -append
		"Smart card fallback: -" | out-file $outputfile -append
	}

	"Callback URL:" +$callbackURL | out-file $outputfile -append
	"Secure Ticket Authority URLs: " + $STAs | out-file $outputfile -append	
	"Enable session reliability : " + $sessionReliability | out-file $outputfile -append
	"Request tickets from two STAs, where available: " +$request2STATickets | out-file $outputfile -append
	"XXXX----------------------------------------XXXX" | out-file $outputfile -append
}
"#------------------------------------" | out-file $outputfile -append
"Processing NetScaler Gateways ends" | out-file $outputfile -append
"#------------------------------------" | out-file $outputfile -append

# Beacons
"Processing Beacons starts" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
$internalBeacons = @((Get-DSGlobalBeacons "Internal").Beacons)
$externalBeacons = @((Get-DSGlobalBeacons "External").Beacons)

if ($internalBeacons.Count -gt 0)
{
	$intrBeacons = ""
	foreach ($beacon in $internalBeacons)
	{
		$beaconAddress = ($beacon).Address
		$intrBeacons = $intrBeacons+", "+$beaconAddress
	}
	if ($intrBeacons.startsWith(","))
	{
		$intrBeacons = $intrBeacons.substring(1)
	}
	"Internal: "+$intrBeacons | out-file $outputfile -append
}

if ($externalBeacons.Count -gt 0)
{
	$extBeacons = ""
	foreach ($beacon in $externalBeacons)
	{
		$beaconAddress = ($beacon).Address
		$extBeacons = $extBeacons+", "+$beaconAddress
	}
	if ($extBeacons.startsWith(","))
	{
		$extBeacons = $extBeacons.substring(1)
	}
	"External: " +$extBeacons | out-file $outputfile -append
}
"#-------------------------------" | out-file $outputfile -append
"Processing Beacons ends" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append

"#EG_SCRIPT_DONE#" | out-file $outputfile -append







# SIG # Begin signature block
# MIIlTAYJKoZIhvcNAQcCoIIlPTCCJTkCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUynnozkR9dCYqnPkDf1Of4m+F
# LNuggh5yMIIFMDCCBBigAwIBAgIQBAkYG1/Vu2Z1U0O1b5VQCDANBgkqhkiG9w0B
# AQsFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMTMxMDIyMTIwMDAwWhcNMjgxMDIyMTIwMDAwWjByMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQg
# Q29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
# +NOzHH8OEa9ndwfTCzFJGc/Q+0WZsTrbRPV/5aid2zLXcep2nQUut4/6kkPApfmJ
# 1DcZ17aq8JyGpdglrA55KDp+6dFn08b7KSfH03sjlOSRI5aQd4L5oYQjZhJUM1B0
# sSgmuyRpwsJS8hRniolF1C2ho+mILCCVrhxKhwjfDPXiTWAYvqrEsq5wMWYzcT6s
# cKKrzn/pfMuSoeU7MRzP6vIK5Fe7SrXpdOYr/mzLfnQ5Ng2Q7+S1TqSp6moKq4Tz
# rGdOtcT3jNEgJSPrCGQ+UpbB8g8S9MWOD8Gi6CxR93O8vYWxYoNzQYIH5DiLanMg
# 0A9kczyen6Yzqf0Z3yWT0QIDAQABo4IBzTCCAckwEgYDVR0TAQH/BAgwBgEB/wIB
# ADAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMweQYIKwYBBQUH
# AQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYI
# KwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcnQwgYEGA1UdHwR6MHgwOqA4oDaGNGh0dHA6Ly9jcmw0
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwTwYDVR0gBEgwRjA4BgpghkgBhv1sAAIEMCowKAYIKwYBBQUHAgEWHGh0
# dHBzOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwCgYIYIZIAYb9bAMwHQYDVR0OBBYE
# FFrEuXsqCqOl6nEDwGD5LfZldQ5YMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6en
# IZ3zbcgPMA0GCSqGSIb3DQEBCwUAA4IBAQA+7A1aJLPzItEVyCx8JSl2qB1dHC06
# GsTvMGHXfgtg/cM9D8Svi/3vKt8gVTew4fbRknUPUbRupY5a4l4kgU4QpO4/cY5j
# DhNLrddfRHnzNhQGivecRk5c/5CxGwcOkRX7uq+1UcKNJK4kxscnKqEpKBo6cSgC
# PC6Ro8AlEeKcFEehemhor5unXCBc2XGxDI+7qPjFEmifz0DLQESlE/DmZAwlCEIy
# sjaKJAL+L3J+HNdJRZboWR3p+nRka7LrZkPas7CM1ekN3fYBIM6ZMWM9CBoYs4Gb
# T8aTEAb8B4H6i9r5gkn3Ym6hU/oSlBiFLpKR6mhsRDKyZqHnGKSaZFHvMIIFjTCC
# BHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0Ew
# HhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEV
# MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29t
# MSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZ
# wuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4V
# pX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAd
# YyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3
# T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjU
# N6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNda
# SaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtm
# mnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyV
# w4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3
# AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYi
# Cd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmp
# sh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7Nfj
# gtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNt
# yA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYY
# aHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2Fj
# ZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUG
# A1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2Vy
# dEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3
# DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+Ica
# aVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096ww
# epqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcD
# x4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsg
# jTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37Y
# OtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMIIGMzCCBRugAwIBAgIQCJp0nrgt
# w+wn6mXq2/g1MTANBgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UE
# ChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYD
# VQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQgQ29kZSBTaWduaW5nIENBMB4X
# DTIxMDUyNTAwMDAwMFoXDTI0MDUyOTIzNTk1OVowcTELMAkGA1UEBhMCVVMxEzAR
# BgNVBAgTCk5ldyBKZXJzZXkxDzANBgNVBAcTBklzZWxpbjEdMBsGA1UEChMUZUcg
# SW5ub3ZhdGlvbnMsIEluYy4xHTAbBgNVBAMTFGVHIElubm92YXRpb25zLCBJbmMu
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAxGUJ0prjbfxcmgZGjd7J
# DxOs7sySHvaQ3wGR2KbD3aZageSE+kG0tKkEsQ4na8bnNViY/zF4Pbo0ZkYtMzJI
# k0AZXDTMvXr/mEvmKxCbiJFTCpKkmZ4sa5BZfO7igIvcNSa0zII2a5jiQJFy85j+
# f9I4EgTo9OcdVeINXhgQ2xZ6TjEK+pbhqmVXvr8DB26JclOaed0L5Vs7+CbGzZuK
# ifgxL2i8d5FDzkhZSZfdCtGtEvE5pLesXSWfpzUddvCNRisIPGB7mg+Rln2XvUZy
# yaRURtlx11pYfJ/KNYAkUCL23rnh38/maxtSvdeioYBj4xLyt7poQBG+b0uzCjgR
# ADe3/k0NtiVwJIo0ZEsUyDANp7JIIjMePzsOMYzfMPyi8clAwfBYc2XhSuUcX3yK
# f4kpRFmClYfK5LstGQRNbONjpsCBTEgZuwEfkdOU0rmaMywZFVLHEpmpYZePz7M3
# VhN+aV56kR/efo8eD81E5VDQqmJiGWKq/s9jhbkYH7g2uhL2MXb/uRshVDSOkFRv
# IID/l+yDJogDBpae3x2ov5YvjY8Zo8RF0tXvyS0rfsDvANZw7mptzHn0Blmw5lug
# 3F2CVJpNbzb1HmAH6k0yFHmWYCoUANbBB0YR/k4JvgY2byttF7cQPJyl7UI9Cu4K
# TAg7ROmGkwdLhmXoPCBco0ECAwEAAaOCAcQwggHAMB8GA1UdIwQYMBaAFFrEuXsq
# CqOl6nEDwGD5LfZldQ5YMB0GA1UdDgQWBBT5yESaz9gTSS8aBNa1r34MMA42SjAO
# BgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYDVR0fBHAwbjA1
# oDOgMYYvaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1n
# MS5jcmwwNaAzoDGGL2h0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFzc3Vy
# ZWQtY3MtZzEuY3JsMEsGA1UdIAREMEIwNgYJYIZIAYb9bAMBMCkwJwYIKwYBBQUH
# AgEWG2h0dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAIBgZngQwBBAEwgYQGCCsG
# AQUFBwEBBHgwdjAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29t
# ME4GCCsGAQUFBzAChkJodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRTSEEyQXNzdXJlZElEQ29kZVNpZ25pbmdDQS5jcnQwDAYDVR0TAQH/BAIwADAN
# BgkqhkiG9w0BAQsFAAOCAQEAlAjncFPvLKywT/4DECG7bHbtiVl+uumfRj2YrFuC
# hsgkv1PmQpgfypxC3g+ErV9yRV8+XXSCwOaKJ3v6RLnfphLhEwYc2+0Qs/Nlib5N
# AxDGuIczAIeXOc5kRRpvFsQ2XSNtM7XL1tLDm6p/VG7BoUAyqRXsMPdWbTkN/9nd
# CmGSsqcxjG2ud8O6Vhte9J5LaHBVk3lIZAMtH6ACdo5QTrM49nbIU8QGuRYNXZKR
# LAUu6IgD6WJKMVfZXWlyfD8dZ2r3ej6Q1uAO/Nbtd397T+BVQrDWMOG8+GeRiJwo
# evxbIWh0SenZOUrAq9vTJaSvFMSvctkJm/oxLUcUdEGS3zCCBq4wggSWoAMCAQIC
# EAc2N7ckVHzYR6z9KGYqXlswDQYJKoZIhvcNAQELBQAwYjELMAkGA1UEBhMCVVMx
# FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNv
# bTEhMB8GA1UEAxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIyMDMyMzAw
# MDAwMFoXDTM3MDMyMjIzNTk1OVowYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRp
# Z2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQw
# OTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQADggIP
# ADCCAgoCggIBAMaGNQZJs8E9cklRVcclA8TykTepl1Gh1tKD0Z5Mom2gsMyD+Vr2
# EaFEFUJfpIjzaPp985yJC3+dH54PMx9QEwsmc5Zt+FeoAn39Q7SE2hHxc7Gz7iuA
# hIoiGN/r2j3EF3+rGSs+QtxnjupRPfDWVtTnKC3r07G1decfBmWNlCnT2exp39mQ
# h0YAe9tEQYncfGpXevA3eZ9drMvohGS0UvJ2R/dhgxndX7RUCyFobjchu0CsX7Le
# Sn3O9TkSZ+8OpWNs5KbFHc02DVzV5huowWR0QKfAcsW6Th+xtVhNef7Xj3OTrCw5
# 4qVI1vCwMROpVymWJy71h6aPTnYVVSZwmCZ/oBpHIEPjQ2OAe3VuJyWQmDo4EbP2
# 9p7mO1vsgd4iFNmCKseSv6De4z6ic/rnH1pslPJSlRErWHRAKKtzQ87fSqEcazjF
# KfPKqpZzQmiftkaznTqj1QPgv/CiPMpC3BhIfxQ0z9JMq++bPf4OuGQq+nUoJEHt
# Qr8FnGZJUlD0UfM2SU2LINIsVzV5K6jzRWC8I41Y99xh3pP+OcD5sjClTNfpmEpY
# PtMDiP6zj9NeS3YSUZPJjAw7W4oiqMEmCPkUEBIDfV8ju2TjY+Cm4T72wnSyPx4J
# duyrXUZ14mCjWAkBKAAOhFTuzuldyF4wEr1GnrXTdrnSDmuZDNIztM2xAgMBAAGj
# ggFdMIIBWTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBS6FtltTYUvcyl2
# mi91jGogj57IbzAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNV
# HQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwgwdwYIKwYBBQUHAQEEazBp
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUH
# MAKGNWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRS
# b290RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMCAGA1UdIAQZMBcwCAYGZ4EM
# AQQCMAsGCWCGSAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAfVmOwJO2b5ipRCIB
# fmbW2CFC4bAYLhBNE88wU86/GPvHUF3iSyn7cIoNqilp/GnBzx0H6T5gyNgL5Vxb
# 122H+oQgJTQxZ822EpZvxFBMYh0MCIKoFr2pVs8Vc40BIiXOlWk/R3f7cnQU1/+r
# T4osequFzUNf7WC2qk+RZp4snuCKrOX9jLxkJodskr2dfNBwCnzvqLx1T7pa96kQ
# sl3p/yhUifDVinF2ZdrM8HKjI/rAJ4JErpknG6skHibBt94q6/aesXmZgaNWhqsK
# RcnfxI2g55j7+6adcq/Ex8HBanHZxhOACcS2n82HhyS7T6NJuXdmkfFynOlLAlKn
# N36TU6w7HQhJD5TNOXrd/yVjmScsPT9rp/Fmw0HNT7ZAmyEhQNC3EyTN3B14OuSe
# reU0cZLXJmvkOHOrpgFPvT87eK1MrfvElXvtCl8zOYdBeHo46Zzh3SP9HSjTx/no
# 8Zhf+yvYfvJGnXUsHicsJttvFXseGYs2uJPU5vIXmVnKcPA3v5gA3yAWTyf7YGcW
# oWa63VXAOimGsJigK+2VQbc61RWYMbRiCQ8KvYHZE/6/pNHzV9m8BPqC3jLfBInw
# AM1dwvnQI38AC+R2AibZ8GV2QqYphwlHK+Z/GqSFD/yYlvZVVCsfgPrA8g4r5db7
# qS9EFUrnEw4d2zc4GqEr9u3WfPwwggbAMIIEqKADAgECAhAMTWlyS5T6PCpKPSkH
# gD1aMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdp
# Q2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2
# IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcNMjIwOTIxMDAwMDAwWhcNMzMxMTIx
# MjM1OTU5WjBGMQswCQYDVQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQxJDAiBgNV
# BAMTG0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDIyIC0gMjCCAiIwDQYJKoZIhvcNAQEB
# BQADggIPADCCAgoCggIBAM/spSY6xqnya7uNwQ2a26HoFIV0MxomrNAcVR4eNm28
# klUMYfSdCXc9FZYIL2tkpP0GgxbXkZI4HDEClvtysZc6Va8z7GGK6aYo25BjXL2J
# U+A6LYyHQq4mpOS7eHi5ehbhVsbAumRTuyoW51BIu4hpDIjG8b7gL307scpTjUCD
# HufLckkoHkyAHoVW54Xt8mG8qjoHffarbuVm3eJc9S/tjdRNlYRo44DLannR0hCR
# RinrPibytIzNTLlmyLuqUDgN5YyUXRlav/V7QG5vFqianJVHhoV5PgxeZowaCiS+
# nKrSnLb3T254xCg/oxwPUAY3ugjZNaa1Htp4WB056PhMkRCWfk3h3cKtpX74LRsf
# 7CtGGKMZ9jn39cFPcS6JAxGiS7uYv/pP5Hs27wZE5FX/NurlfDHn88JSxOYWe1p+
# pSVz28BqmSEtY+VZ9U0vkB8nt9KrFOU4ZodRCGv7U0M50GT6Vs/g9ArmFG1keLuY
# /ZTDcyHzL8IuINeBrNPxB9ThvdldS24xlCmL5kGkZZTAWOXlLimQprdhZPrZIGwY
# UWC6poEPCSVT8b876asHDmoHOWIZydaFfxPZjXnPYsXs4Xu5zGcTB5rBeO3GiMiw
# bjJ5xwtZg43G7vUsfHuOy2SJ8bHEuOdTXl9V0n0ZKVkDTvpd6kVzHIR+187i1Dp3
# AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADAWBgNV
# HSUBAf8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgB
# hv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYDVR0OBBYE
# FGKK3tBh/I8xFO2XC809KpQU31KcMFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9j
# cmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZU
# aW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEBBIGDMIGAMCQGCCsGAQUFBzAB
# hhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9j
# YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEy
# NTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcNAQELBQADggIBAFWqKhrzRvN4
# Vzcw/HXjT9aFI/H8+ZU5myXm93KKmMN31GT8Ffs2wklRLHiIY1UJRjkA/GnUypsp
# +6M/wMkAmxMdsJiJ3HjyzXyFzVOdr2LiYWajFCpFh0qYQitQ/Bu1nggwCfrkLdcJ
# iXn5CeaIzn0buGqim8FTYAnoo7id160fHLjsmEHw9g6A++T/350Qp+sAul9Kjxo6
# UrTqvwlJFTU2WZoPVNKyG39+XgmtdlSKdG3K0gVnK3br/5iyJpU4GYhEFOUKWaJr
# 5yI+RCHSPxzAm+18SLLYkgyRTzxmlK9dAlPrnuKe5NMfhgFknADC6Vp0dQ094XmI
# vxwBl8kZI4DXNlpflhaxYwzGRkA7zl011Fk+Q5oYrsPJy8P7mxNfarXH4PMFw1nf
# J2Ir3kHJU7n/NBBn9iYymHv+XEKUgZSCnawKi8ZLFUrTmJBFYDOA4CPe+AOk9kVH
# 5c64A0JH6EE2cXet/aLol3ROLtoeHYxayB6a1cLwxiKoT5u92ByaUcQvmvZfpyeX
# upYuhVfAYOd4Vn9q78KVmksRAsiCnMkaBXy6cbVOepls9Oie1FqYyJ+/jbsYXEP1
# 0Cro4mLueATbvdH7WwqocH7wl4R44wgDXUcsY6glOJcB0j862uXl9uab3H4szP8X
# TE0AotjWAQ64i+7m4HJViSwnGWH2dwGMMYIGRDCCBkACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFIFbp2DRai9yRKiZ
# dSSni0tf+VDbMA0GCSqGSIb3DQEBAQUABIICAF2eqlKsJUS70OsWFA7R5OkNFw/d
# ylLHwDyohIyXSbziLc9LFYdQ1EkD50W5JOE9vgxrLTuleWFJvtgV728Yqd7oxl01
# bENP7EpnO5DtWh65XPy0qW2MMMCtwuNHdAEPgGh1ljAwIjelDcJ2/Gi7y7B4+2V5
# UAsGLVhXblDVWwZOzLCLDF4ywhkZWquk/RMog8hzJZaAl3GLfEzjRbtVZfj1gVD4
# FVteV/pXgh/HQ+nvYaKT0f+18MM7OovHWm6QbGDanwKnE8UUqEKwtqM/HCx8O29K
# /wyNAnQvw1ZF1oFv2rCfMtKO8SqT1wNHai5T6TFE9KR2nVc7Vlj2geEGEgiChSCX
# adzDGrO68Rrkj4Lfw/Pe8ijI4SlyvYtVln0qRI91hmTZLcwj4F89ck2JxDLP7q82
# dRO2QiceSFd69p1FHuCVVpsFtMzdJpqrkRvG1MrQalRArZaFz6Knqwj6E9M9RIFg
# kD86NOmxPtqLM6SH+rmdzLCbvllVv0HqO6d6W0aTxb4Lr+2VDCOdgci2Llrz8CnY
# SPGB0fJ99FRfSCZkD/5MRxgTuBqRXppWOhg2xlBrRDHtPcbSPTA9PBGzhYLNVfzJ
# K258NRPS4KGVhsZbDa5j0nmfvNI+FAZA4Mf/Y8rnOVz6i3VuxxPTTsLnSZjCZPD4
# rFqUHvItH7mL2SiSoYIDIDCCAxwGCSqGSIb3DQEJBjGCAw0wggMJAgEBMHcwYzEL
# MAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJE
# aWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBD
# QQIQDE1pckuU+jwqSj0pB4A9WjANBglghkgBZQMEAgEFAKBpMBgGCSqGSIb3DQEJ
# AzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTIzMDQwNzA3NTQ0MVowLwYJ
# KoZIhvcNAQkEMSIEIJh8x8KFoP94gx3wedcX1KzRZ0zM/2YAGrzMySGDiighMA0G
# CSqGSIb3DQEBAQUABIICAC7ZRhbI94aip5ECEGbagGc/ychMq+8rkAJ1Wed7of5A
# nWb7maWceQYLwmqGlSp10TokrMkbzvZH4k2uA2GAMA4x850k9/QQGJ7F+sHxIVL1
# +wlD1adaXOj2Tohog92z/nAn4urMQzeEA+i11yoSs3uj/QY/WrxUpcdLJWTbaMGf
# iqN9OpOelTbQoFldPnhNHBBAtahzO+XaSjJSk7rNO3lUYx68ZHbj1zvpS0vrY/lJ
# URKRTlSG3oMbZhNlAj303ZfurCXXFeaDlkkhfi19KEQdm9wc+EPeseDpNMm8vZQ3
# eT8nP0M+eIzV/ZLI6r6SnItpfJi97KZp5j/Wy0eYIGxSia8k/NELTrSFJeHOOPi5
# eVY9HB36oGEEAWk8qRRsUzUXpII0RtNPRthwUvZnhUTaBwQG+bBBvaXfOrrThrTT
# XLZp9mRT0zMQtEPfhxUuY12oTzJ/L8BCp5DYXMpiydW0V02WM6mobPfBDC3k//bZ
# dy7zYyNYZrxa3FZwMfBZB5B84dF2PEPHXe5Q1w33+uXc7zdQrkApVVaJBS5i4tzI
# MwIvKa4xDHY9eCwJS/+dnelKVd/JaQbkA/XnwDHv8LzobhlDhQeHDVk6TIpARY0t
# paNWDc1Semn98Xw1JkcSerh0/D+LGWS7GJLkm++YviSn07Rz7vsIpzHBf+g1grU7
# SIG # End signature block
