'use strict'

const onHeaders = require('on-headers');
const Buffer = require('safe-buffer').Buffer;

const logger = require('../logger');
const parser = require('../parser');
const httpShared = require('./http-shared');
const appConstant = require('../app-constants');
const instrumentationUtils = require('./instrumentation-utils');

const appendMsg = 'Http-res-transformer:';
const preC = '\n\n  <!-- RUM Header Starts --> \n  ';
const postC = '\n  <!-- RUM Header Ends  --> \n';
const validStatusCode = [200].concat(appConstant.REDIRECT_STATUS_CODE);
const cacheControlNoTransformRegExp = /(?:^|,)\s*?no-transform\s*?(?:,|$)/;

exports.transform = function (agent, req, res, moduleName) {
  const _write = res.write;
  let isNeedToTrans = false;
  let isTransformed = false;
  let isTransformChecked = false;
  const rumSnippet = agent.getConfig('rum_snippet');
  const enableRumInjection = agent.getConfig('enable_rum_injection');
  const rumSnippetInjectionPlace = agent.getConfig('rum_snippet_injection_place') || [];

  res.write = function write(chunk, encoding) {
    // if (!this._header) this._implicitHeader();
    if (isTransformChecked && !isNeedToTrans) return _write.apply(this, arguments);
    if (!isTransformChecked && !_isNeedToTransform('write')) return _write.apply(this, arguments);

    isNeedToTrans = true;
    logger.debug(appendMsg, `Going to try to transform URL: ${req.url}, place: res.write`);
    chunk = resTransformer(chunk);
    if (!chunk) return _write.apply(this, arguments);
    isTransformed = true;
    return _write.call(this, toBuffer(chunk, encoding), encoding);
  }

  const _end = res.end;
  res.end = function (chunk, encoding) {
    if (!chunk || isTransformed || (isTransformChecked && !isNeedToTrans)) return _end.apply(this, arguments);
    if (!isTransformChecked && !_isNeedToTransform('end')) return _end.apply(this, arguments);

    isNeedToTrans = true;
    logger.debug(appendMsg, `Going to try to transform URL: ${req.url}, place: res.end`);
    chunk = resTransformer(chunk);
    if (!chunk) return _end.apply(this, arguments);

    return _end.call(this, toBuffer(chunk, encoding), encoding);
  }

  onHeaders(res, function () {
    try {
      const trans = httpShared.getTransaction(this);
      if (!trans) return;

      trans.userContext = parser.getUserContextFromReq(req);
      if (agent.getConfig('enable_rum_btm_transaction_mapping')) {
        const egrumHead = req.headers && (req.headers.EGRUM || req.headers.egrum);
        _setHeader.call(this, 'Server-Timing', `eg-btm-guid;desc="${trans.getGuid()}";dur=${trans.getServerTime()}`);

        if (egrumHead === 'isAjax:true') {
          _setHeader.call(this, appConstant.RUM_HEADER, trans.getDataForRUM());
        } else if (isValidHtmlRes()) {
          const reqCookie = req.headers && (req.headers.cookie || req.headers.Cookie);
          if (reqCookie && reqCookie.indexOf(appConstant.RUM_COOKIE + '=') > -1) {
            _pushCookie.call(this, '; Max-Age=1;');
          }

          let path = req.originalUrl || req.url;
          path && (path = path.split("?")[0]);
          path && (path = ` Path=${path};`);
          let cookie = `${trans.getDataForRUM()}; Max-Age=30;${path}; Secure;`;
          _pushCookie.call(this, cookie);
          logger.debug(appendMsg, appConstant.RUM_COOKIE + " cookie is dropped.");
        }
      }

      if (trans.egEnabledInReqOrigin) {
        // need to send unique_component_id in res header
        _setHeader.call(this, appConstant.RES_HEADER_GUID, agent.getConfig('unique_component_id'));
        logger.debug(appendMsg, 'Req is from eG enabled server, so Guid was set in the res header');
      }

      if (isNeedToTrans || (!isTransformChecked && _isNeedToTransform('onHeader'))) {
        isNeedToTrans = true;
        res.removeHeader('Content-Length');
        logger.debug(appendMsg, `May be transformed ${req.url}`);
      }
    } catch (e) {
      logger.error(appendMsg, 'onHeaders', e);
    }

    function _pushCookie(value) {
      const cookie = instrumentationUtils.getResHeader.call(this, 'set-cookie') || [];
      cookie.push(`${appConstant.RUM_COOKIE}=${value} SameSite=Lax`);
      _setHeader.call(this, 'set-cookie', cookie);
    }
  });

  function _isNeedToTransform(place) {
    isTransformChecked = true;
    return rumSnippet && enableRumInjection === true && rumSnippetInjectionPlace.length && isNeedToInjectRumSnippet(place);
  }

  function resTransformer(chunk) {
    try {
      let rumInjectedData = chunk && chunk.toString();
      const tagToInject = getValidTagToInject(rumInjectedData);

      if (tagToInject && tagToInject.tag) {
        let desc = preC + rumSnippet + postC;
        desc = tagToInject.isAfter ? tagToInject.tag + desc : desc + tagToInject.tag;
        chunk = rumInjectedData.replace(tagToInject.tag, desc);
        logger.info(appendMsg, `The RUM snippet is injected in to the response of URL ${req.url}`);
        return chunk;
      }

      return false;
    } catch (e) {
      logger.error(appendMsg, 'onHeaders', e);
      callback(null, chunk, encoding);
    }
  }

  function getValidTagToInject(html) {
    if (!html) return notAbleToTransForm('no body found', html);
    if (!html.match(/<!doctype html.*?>/gi) && !html.match(/<html.*?>/gi)) {
      // if html tag is not found then ignore it
      return notAbleToTransForm('html tag is not found', html);
    }

    for (let i = 0; i < rumSnippetInjectionPlace.length; i++) {
      const ele = rumSnippetInjectionPlace[i];
      if (!ele || !ele.regex || !ele.type) continue;
      const tags = html.match(ele.regex);
      if (!tags || !tags.length) continue;

      if (ele.type === 'AFTER_LAST') {
        return {
          tag: tags[tags.length - 1],
          isAfter: true,
        };
      } else if (ele.type === 'AFTER') {
        return {
          tag: tags[0],
          isAfter: true,
        };
      } else {
        return { tag: tags[0] };
      }
    }

    return notAbleToTransForm(`RUM Snippet Injection Place can't be found`, html);
  }

  function isNeedToInjectRumSnippet(place) {
    if (!isValidHtmlRes()) return noTransForm('it is not a valid html res', place);
    if (validStatusCode.indexOf(res.statusCode) === -1) return noTransForm('response status is ' + res.statusCode, place);
    if (req.method === 'HEAD') return noTransForm('HEAD request', place);
    if (!shouldTransform()) return noTransForm('Cache-Control: no-transform', place);
    return true;
  }

  function notAbleToTransForm(msg, data) {
    logger.debug(appendMsg, `Not able to transform ${req.url} because ${msg}`, data);
    return false;
  }

  function noTransForm(msg, place) {
    logger.debug(appendMsg, `No need to transform ${req.url} because ${msg}. place: ${place}`);
    return false;
  }

  function isValidHtmlRes() {
    const contentType = instrumentationUtils.getResHeader.call(res, 'Content-Type');
    return contentType && contentType.indexOf('text/html') > -1 && req.url.substr(-4) != '.map';
  }

  function shouldTransform() {
    const cacheControl = instrumentationUtils.getResHeader.call(res, 'Cache-Control');

    // Don't transform for Cache-Control: no-transform
    // https://tools.ietf.org/html/rfc7234#section-5.2.2.4
    return !cacheControl ||
      !cacheControlNoTransformRegExp.test(cacheControl);
  }
}

function _setHeader(key, value) {
  if (this.setHeader) {
    this.setHeader(key, value);
  } else if (this._headers) {
    this._headers[key] = value;
  }
}

function toBuffer(chunk, encoding) {
  return !Buffer.isBuffer(chunk)
    ? Buffer.from(chunk, encoding)
    : chunk;
}