'use strict'

const semver = require('semver');

const shimmer = require('../shimmer');
const logger = require('../../logger');
const utils = require('../instrumentation-utils');
const safeStringify = require('../../utils/stringify');

let listener = null;
let isRunning = false;
let isWrapped = false;
const appendMsg = 'Mongodb:';
const HOSTNAME_PORT_RE = /^([^:]+):(\d+)$/

exports.start = function (mongodb, agent, version, enabled) {
  if (!enabled) return mongodb;
  if (!semver.satisfies(version, '>=3.3 <5.0')) {
    logger.info(appendMsg, `version ${version} not instrumented (mongodb <3.3 is instrumented via mongodb-core)`)
    return mongodb;
  }

  isWrapped = true;
  const activeSpans = new Map();
  try {
    if (mongodb.instrument) {
      if (!listener) {
        listener = mongodb.instrument();
      } else {
        listener.instrument(mongodb.MongoClient);
      }

      logger.debug(appendMsg, 'listener added to the mongodb.instrument');
      listener.on('started', onStart);
      listener.on('succeeded', onEnd);
      listener.on('failed', onError);
      isRunning = true;
      logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
    } else if (mongodb.MongoClient) {
      // mongodb 4.0+ removed the instrument() method in favor of
      // listeners on the instantiated client objects. There are two mechanisms
      // to get a client:
      // 1. const client = new mongodb.MongoClient(...)
      // 2. const client = await MongoClient.connect(...)
      class MongoClientTraced extends mongodb.MongoClient {
        constructor() {
          // The `command*` events are only emitted if `options.monitorCommands: true`.
          const args = Array.prototype.slice.call(arguments);
          if (!args[1]) {
            args[1] = { monitorCommands: true };
          } else if (args[1].monitorCommands !== true) {
            args[1] = Object.assign({}, args[1], { monitorCommands: true });
          }
          super(...args);
          this.on('commandStarted', onStart);
          this.on('commandSucceeded', onEnd);
          this.on('commandFailed', onError);
        }
      }

      Object.defineProperty(
        mongodb,
        'MongoClient',
        {
          enumerable: true,
          get: function () {
            return MongoClientTraced
          }
        }
      );
      isRunning = true;
      shimmer.wrap(mongodb.MongoClient, 'connect', wrapConnect);
      logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
    } else {
      logger.warn(appendMsg, 'could not instrument mongodb', version);
    }
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }

  return mongodb;
  // Wrap the MongoClient.connect(url, options?, callback?) static method.
  // It calls back with `function (err, client)` or returns a Promise that
  // resolves to the client.
  // https://github.com/mongodb/node-mongodb-native/blob/v4.2.1/src/mongo_client.ts#L503-L511
  function wrapConnect(origConnect) {
    return function wrappedConnect(url, options, callback) {
      if (typeof options === 'function') {
        callback = options;
        options = {};
      }
      options = options || {}
      if (!options.monitorCommands) {
        options.monitorCommands = true;
      }
      if (typeof callback === 'function') {
        return origConnect.call(this, url, options, function wrappedCallback(err, client) {
          if (err) {
            callback(err);
          } else {
            client.on('commandStarted', onStart);
            client.on('commandSucceeded', onEnd);
            client.on('commandFailed', onError);
            callback(err, client);
          }
        });
      } else {
        const p = origConnect.call(this, url, options, callback);
        p.then(client => {
          client.on('commandStarted', onStart);
          client.on('commandSucceeded', onEnd);
          client.on('commandFailed', onError);
        });
        return p;
      }
    }
  }

  function onStart(event) {
    if (!isRunning || activeSpans.has(event.requestId)) return;
    const names = [
      getCollectionName(event),
      event.commandName
    ];
    const name = names.join('.');

    const span = agent.startSpan(name, 'mongodb', 'Mongodb');
    if (!span) return;
    activeSpans.set(event.requestId, span);
    const address = event.address || event.connectionId;
    let match;
    span.uid = event.requestId;
    span.options = {};
    span.options.dbName = event.databaseName;
    span.options.query = getQuery(event, name);
    span.options.entityName = names[0];
    span.options.queryType = names[1];
    span.options.serverType = 'mongodb';

    if (address && typeof (address) === 'string' &&
      (match = HOSTNAME_PORT_RE.exec(address))) {
      span.options.host = match[1];
      span.options.port = Number(match[2]);
    } else {
      logger.debug(appendMsg, 'could not set destination context on mongodb span from address=', address);
    }
  }

  function getQuery(event, name) {
    if (!event || !event.command) return;
    let query = {};
    query.name = name;
    const cmd = event.command;
    if (cmd.filter) query.filter = cmd.filter;
    if (cmd.projection) query.projection = cmd.projection;
    if (cmd.limit) query.limit = cmd.limit;
    if (cmd.indexes) query.indexes = cmd.indexes;
    if (cmd.pipeline) query.pipeline = cmd.pipeline;
    if (cmd.documents) query.documents = cmd.documents;
    query = safeStringify(query);
    const maxChar = agent.getConfig('max_nosql_query_length');
    return utils.subString(query, maxChar);
  }

  function onError(event) {
    if (!isRunning || !activeSpans.has(event.requestId)) return;
    if (event.failure) {
      const span = activeSpans.get(event.requestId);
      logger.debug(appendMsg, 'error captured');
      span.captureError(event.failure);
    }
    onEnd(event);
  }

  function onEnd(event) {
    if (!isRunning || !activeSpans.has(event.requestId)) return;
    const span = activeSpans.get(event.requestId);
    activeSpans.delete(event.requestId);
    span.end(event.duration);
  }

  function getCollectionName(event) {
    const collection = event.command[event.commandName];
    return typeof collection === 'string' ? collection : '$cmd';
  }
}

exports.stop = function (mongodb, version) {
  if (!isWrapped) return;
  if (mongodb.instrument) {
    listener.uninstrument();
  } else if (mongodb.MongoClient) {
    shimmer.unwrap(mongodb.MongoClient, 'connect');
  }
  isRunning = false;
  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}