'use strict'

const Cpu = require('./cpu');
const Memory = require('./memory');
const logger = require('../logger');
const GcStats = require('./gc-stats');
const symbols = require('../symbols');
const appUtils = require('../app-utils');
const EventLoop = require('.//eventloop');
const threads = require('../utils/threads');
const CpuProfile = require('./cpu-profiler');
const MemoryProfile = require('./memory-profiler');

const appendMsg = 'InfraManager:';

/**
 * To collect overall infra level metrics like
 * cpu, memory, eventloop, garbage collection
 * cpu and memory dump periodically
 * 
 * default collect in interval is 5 mins once
 */
function InfraManager(agent) {
  this.agent = agent;
  this.cpu = new Cpu();
  this.memory = new Memory();
  this.gcStats = new GcStats(this);
  this.eventLoop = new EventLoop();
  this.cpuProfile = new CpuProfile(agent);
  this.memoryProfile = new MemoryProfile(agent);
  this.timer = null;
  this.isStarted = false;
}

InfraManager.prototype.start = function () {
  if (this.isStarted) return;
  if (!this.agent.getConfig('enable_inframetrics')) {
    logger.info(appendMsg, 'Inframetrics is disabled');
    return;
  }

  logger.info(appendMsg, `starting and infra manager will report for every ${this.agent.getConfig('inframetrics_interval')} Seconds.`);
  const _self = this;
  this.isStarted = true;

  try {
    this.collectInterval = this.agent.getConfig('inframetrics_interval') * 1000;
    if (this.agent.getConfig('enable_cpu_metrics')) {
      this.cpu.start();
    }

    if (this.agent.getConfig('enable_memory_metrics')) {
      this.memory.start();
    }

    if (this.agent.getConfig('enable_cpu_profiling')) {
      this.cpuProfile.start();
    }

    if (this.agent.getConfig('enable_gc_metrics')) {
      this.gcStats.start();
    }

    if (this.agent.getConfig('enable_event_loop_metrics')) {
      this.eventLoop.start(this.collectInterval);
    }

    if (this.agent.runtime.type === 'pm2') {
      const data = this.agent.metadata.get("LAST_INFRA_COLLECTED_AT");

      if (data) {
        const nextMetricsTime = new Date(parseInt(data) + this.collectInterval);
        if (new Date() < nextMetricsTime) {
          const nextInterval = nextMetricsTime - new Date();
          logger.debug(appendMsg, `process is restarted. in order to sync with other process, the profiler will send metrics after ${nextInterval} ms, after that it will send every ${this.collectInterval} ms`);

          const timeout = setTimeout(_ => {
            _self.sendMetrics();
            _setTimer(_self);
          }, nextInterval);
          timeout.unref();
          return;
        }
      }
    }

    _setTimer.call(this);
    logger.info(appendMsg, `started successfully...!`);
  } catch (e) {
    logger.error(appendMsg, 'start error', e);
  }
}

function _setTimer() {
  if (this.timer) {
    clearInterval(this.timer);
    this.timer = null;
  }
  if (!this.isStarted) return;

  const _self = this;
  this.timer = setInterval(function () {
    logger.debug(appendMsg, 'Started collecting Inframetrics...');
    try {
      _self.sendMetrics();
    } catch (e) {
      logger.error(appendMsg, 'start timer', e);
    }
  }, this.collectInterval);
  this.timer.unref();
}

InfraManager.prototype.restart = function () {
  logger.info(appendMsg, `restarting and infra manager will report for every ${this.agent.getConfig('inframetrics_interval')} Seconds.`);
  this.isStarted = true;

  try {
    this.collectInterval = this.agent.getConfig('inframetrics_interval') * 1000;

    if (this.agent.getConfig('enable_event_loop_metrics')) {
      this.eventLoop.restart(this.collectInterval);
    }

    _setTimer.call(this);
    logger.info(appendMsg, `restarted successfully...!`);
  } catch (e) {
    logger.error(appendMsg, 'restart error', e);
  }
}

/**
 * Collect all the metrics and send it to queue
 */
InfraManager.prototype.sendMetrics = function () {
  if (!this.isStarted) return;
  const _self = this;
  const data = {};

  if (this.agent.getConfig('enable_worker_thread_metrics') && this.getWorkerThreadMetrics) {
    this.getWorkerThreadMetrics((err, workerThreads) => {
      data.workerThreads = workerThreads;
      _sendMetrics.call(_self, data);
    });
  } else {
    _sendMetrics.call(_self, data);
  }

  if (this.agent.getConfig('enable_cpu_profiling')) {
    this.cpuProfile.collectAndSendProfile();
  }
  logger.info('eG Profiler stats', this.agent._getStats());
}

function _sendMetrics(data) {
  try {
    if (this.agent.getConfig('enable_memory_metrics')) {
      data.memoryUsage = this.memory.getUsage();
    }

    if (this.agent.getConfig('enable_cpu_metrics')) {
      data.cpuUsage = this.cpu.getUsage();
      if (!data.workerThreads || !data.workerThreads.length) {
        delete data.cpuUsage.threadCpuPct;
      }
    }

    if (this.agent.getConfig('enable_gc_metrics')) {
      data.gc = this.gcStats.getStats();
    }

    if (this.agent.getConfig('enable_event_loop_metrics')) {
      data.eventloop = this.eventLoop.getStats();
    }

    if (this.agent.getConfig('enable_uptime_metrics')) {
      data.uptime = _getCollectionInverval(this);
      data.totalUptime = Number((process.uptime() * 1000).toFixed(2));
    }

    this.sendinfra(data);
    //this.errorReporter.reportIfAnyFatalError();
    this.lastMetricsCollectedAt = Date.now();

    if (this.agent.runtime.type === 'pm2') {
      // going to write this metrics send time in a file
      this.agent.metadata.set("LAST_INFRA_COLLECTED_AT", Date.now().toString());
    }

  } catch (e) {
    logger.error(appendMsg, 'sendMetrics', e)
  }
}

function _getCollectionInverval(_self) {
  const uptime = _self.lastMetricsCollectedAt ? Date.now() - _self.lastMetricsCollectedAt : (process.uptime() * 1000);
  return Number(uptime.toFixed(2));
}

InfraManager.prototype.sendinfra = function (data, option) {
  if (!data || !this.isStarted) return;
  option = option || {};
  data.tz = appUtils.getTimezone();
  data.utcOffset = appUtils.getUtcOffset();
  data.collectedAt = appUtils.formatDate();
  data.threadId = option.threadId || threads.threadId;

  if (option.pid && option.pid != process.pid) {
    // sending metrics for a disconnected process.
    data.processID = option.pid;
  } else {
    data.processID = process.pid;
    if (this.agent.runtime) {
      data = Object.assign(data, this.agent.runtime.getInfo());
    }
  }

  //Send the data to the queue
  this.agent.collector.add({
    infra: data
  });
}

InfraManager.prototype.sendError = function (err, option) {
  err.err[symbols.errorReportedSymbol] = true;
  option = option || {};
  const data = {
    error: {
      errorTime: option.time ? (new Date(option.time)).getTime() : Date.now(),
      message: err.message,
      stack: err.stack.toString(),
      details: err.details,
      codeError: err.causes,
    }
  };

  //data.uptime = null;
  this.sendinfra(data, option);
}

InfraManager.prototype.workerExitInfo = function (worker, code, signal) {
  const _self = this;

  if (worker && worker.process) {
    const option = {
      pid: worker.process.pid,
    };
    const data = {
      workerExit: {
        code,
        signal
      },
      workerId: worker.id,
    }

    //_self.sendinfra(data, option);
  }

  // setTimeout(_ => {
  //   _self.errorReporter.reportIfAnyFatalError();
  // }, 1000);
}

InfraManager.prototype.stop = function () {
  if (!this.isStarted) return;
  try {
    clearInterval(this.timer);
    this.timer = null;
    this.cpuProfile.stop();
    this.gcStats.stop();
    this.eventLoop.stop();
    this.isStarted = false;
    logger.info(appendMsg, ' stopped')
  } catch (e) {
    logger.error(appendMsg, ' stop:', e)
  }
}

/**
 * process._getActiveRequests() to get a list of active I/O requests 
 * and process._getActiveHandles() to get a list of open handles/file descriptors.
 * 
 * https://github.com/mafintosh/why-is-node-running
 * 
 * Note:-This one is not added in the infra metrics
 */

InfraManager.prototype.collectActiveMetrics = function () {

  var activeRequests = 0;
  var activeHandles = 0;
  var counter = 0;

  var interval = setInterval(function () {
    activeRequests += process._getActiveHandles().length;
    activeHandles += process._getActiveRequests().length;
    counter++;
  }, 10000)

  interval.unref();
}

module.exports = InfraManager;