'use strict'

const semver = require('semver');

const logger = require('../logger');
const appUtils = require('../app-utils');

const Transaction = require('./transaction');
const ModuleHooker = require('../module-hooker');
const HttpStatusCodeTest = require('./http-status-code-test');
const instrumentationUtils = require('./instrumentation-utils');
const {
  AsyncHooksRunContextManager,
  // AsyncLocalStorageRunContextManager
} = require('./run-context');

let asyncHooks = null;
const appendMsg = 'Instrumentation:';
// const nodeSupportsAsyncLocalStorage = semver.satisfies(process.versions.node, '>=14.5 || ^12.19.0')

const MODULES = {
  amqplib: {
    enabled: true,
    filesToHook: {
      'amqplib/lib/channel': { path: 'amqplib/channel' },
      'amqplib/lib/connect': { path: 'amqplib/connect' },
      'amqplib/lib/callback_model': { path: 'amqplib/model' },
      'amqplib/lib/channel_model': { path: 'amqplib/model' },
    }
  },
  express: { enabled: true },
  http: { enabled: true },
  https: { enabled: true },
  http2: { enabled: true },
  mysql: { enabled: true },
  mysql2: { enabled: true },
  mongodb: { enabled: true },
  mongoose: { enabled: true },
  tedious: { enabled: true },
}

try {
  asyncHooks = require('async_hooks');
} catch (e) {
  logger.info(appendMsg, "native async-hooks module is not supported");
}

module.exports = Instrumentation

function Instrumentation(agent) {
  this.timer = null;
  this._agent = agent;
  this._started = false;
  this._runCtxMgr = null;
  this.lastBtmSentAt = new Date();
  this.httpStatusCodeTest = new HttpStatusCodeTest(agent);
  this.moduleHooker = new ModuleHooker(MODULES, agent, './instrumentation/modules/');
}

Instrumentation.prototype.start = function () {
  if (this._started) return;
  if (!this._agent.getConfig('enable_btm')) {
    logger.info(appendMsg, 'BTM is disabled');
    return;
  }

  try {
    // if (nodeSupportsAsyncLocalStorage) {
    //   this._runCtxMgr = new AsyncLocalStorageRunContextManager()
    // } else 
    if (asyncHooks && semver.gte(process.version, '8.2.0')) {
      this._runCtxMgr = new AsyncHooksRunContextManager()
    } else {
      logger.info(appendMsg, 'BTM is not possible without async-hooks module');
      return;
    }

    this._started = true;
    appUtils.setBlackListedUrls(this._agent.config);
    if (this._agent.getConfig('enable_http_status_code_test')) {
      this.httpStatusCodeTest.start();
    }

    logger.debug(appendMsg, 'going to wrap Node.js modules');
    this._runCtxMgr.enable();
    this.moduleHooker.start();
    logger.info(appendMsg, 'started');
  } catch (e) {
    logger.error(appendMsg, 'error while starting instrumentation', e);
  }
}

Instrumentation.prototype.currTransaction = function () {
  if (!this._started) return null;
  return this._runCtxMgr.active().currTransaction();
}

Instrumentation.prototype.currSpan = function () {
  if (!this._started) return null;
  return this._runCtxMgr.active().currSpan();
}

Instrumentation.prototype.stop = function () {
  if (!this._started || !this._runCtxMgr) return;

  if (!this._agent.getConfig('enable_http_status_code_test')) {
    this.httpStatusCodeTest.stop();
  }

  this._runCtxMgr.disable();
  this.moduleHooker.stop();
  this._started = false;
  if (this.timer) clearInterval(this.timer);
  logger.info(appendMsg, 'stopped');
}

Instrumentation.prototype.testReset = function () {
  if (this._runCtxMgr) {
    this._runCtxMgr.testReset()
  }
}

/**
 * addEndTransactionIs the main file
 */
Instrumentation.prototype.addEndedTransaction = function (transaction) {
  if (!this._started) {
    logger.debug(appendMsg, 'ignoring transaction since the btm is off.', transaction._getInfo());
    return;
  }

  const rc = this._runCtxMgr.active();
  if (rc.currTransaction() === transaction) {
    // Replace the active run context with an empty one. I.e. there is now
    // no active transaction or span (at least in this async task).
    this._runCtxMgr.supersedeRunContext(this._runCtxMgr.root());
  }

  if (!this.httpStatusCodeTest.add(transaction)) return;
  const btm = instrumentationUtils.getBtmPayload.call(this, transaction);

  //Send the data to the agent  
  this._agent.collector.add({ btm });
  this.lastBtmSentAt = new Date();
  transaction = null;
}

Instrumentation.prototype.addEndedSpan = function (span) {
  if (!this._started) {
    agent.logger.debug('span is not started', { span: span.id, name: span.name, type: span.type });
    return;
  }

  // Replace the active run context with this span removed. Typically this
  // span is the top of stack (i.e. is the current span). However, it is
  // possible to have out-of-order span.end(), in which case the ended span
  // might not.
  const newRc = this._runCtxMgr.active().leaveSpan(span);
  if (newRc) this._runCtxMgr.supersedeRunContext(newRc);
  logger.debug(`addEndedSpan(${span.name})`);
}

Instrumentation.prototype.startTransaction = function (name, type, option) {
  if (!this._started) return;
  const trans = new Transaction(this._agent, name, type, option);
  this.supersedeWithTransRunContext(trans);
  return trans;
}

// Replace the current run context with one where the given transaction is
// current.
Instrumentation.prototype.supersedeWithTransRunContext = function (trans) {
  if (!this._started) return;
  const rc = this._runCtxMgr.root().enterTrans(trans);
  this._runCtxMgr.supersedeRunContext(rc);
}

// Replace the current run context with one where the given span is current.
Instrumentation.prototype.supersedeWithSpanRunContext = function (span) {
  if (!this._started || !span) return;
  const rc = this._runCtxMgr.active().enterSpan(span);
  this._runCtxMgr.supersedeRunContext(rc);
}

// Set the current run context to have *no* transaction. No spans will be
// created in this run context until a subsequent `startTransaction()`.
Instrumentation.prototype.supersedeWithEmptyRunContext = function () {
  if (!this._started) return;
  this._runCtxMgr.supersedeRunContext(this._runCtxMgr.root())
}

Instrumentation.prototype.endTransaction = function (result) {
  const trans = this.currTransaction()
  if (!trans) {
    logger.debug(appendMsg, 'cannot end transaction - no active transaction found');
    // this._agent.collector.add({ btm: null });
    return;
  }

  trans.end(result);
}

Instrumentation.prototype.startSpan = function (name, type, _pointCutName) {
  const trans = this.currTransaction();
  if (!trans) {
    logger.debug(appendMsg, `no active transaction found - can't start new span`, { name, type, _pointCutName });
    return null;
  }

  if (trans.ended) {
    logger.debug(appendMsg, `transaction is already ended - can't start a new span`, { name, type, _pointCutName, transUid: trans.id });
    return null;
  }

  const span = trans.buildSpan(type);
  if (!span) return;
  span.start(name, type, _pointCutName);
  this._agent._instrumentation.supersedeWithSpanRunContext(span);
  return span;
}

Instrumentation.prototype.currRunContext = function () {
  if (!this._started) return null;
  return this._runCtxMgr.active();
}

// Bind the given function to the current run context.
Instrumentation.prototype.bindFunction = function (fn) {
  if (!this._started) return fn;
  return this._runCtxMgr.bindFn(this._runCtxMgr.active(), fn);
}

// Bind the given function to a given run context.
Instrumentation.prototype.bindFunctionToRunContext = function (runContext, fn) {
  if (!this._started) return fn;
  return this._runCtxMgr.bindFn(runContext, fn);
}

// Bind the given function to an *empty* run context.
// This can be used to ensure `fn` does *not* run in the context of the current
// transaction or span.
Instrumentation.prototype.bindFunctionToEmptyRunContext = function (fn) {
  if (!this._started) return fn;
  return this._runCtxMgr.bindFn(this._runCtxMgr.root(), fn);
}

// Bind the given EventEmitter to the current run context.
//
// This wraps the emitter so that any added event handler function is bound
// as if `bindFunction` had been called on it. Note that `ee` need not
// inherit from EventEmitter -- it uses duck typing.
Instrumentation.prototype.bindEmitter = function (ee) {
  if (!this._started) return ee;
  return this._runCtxMgr.bindEE(this._runCtxMgr.active(), ee);
}

// Bind the given EventEmitter to a given run context.
Instrumentation.prototype.bindEmitterToRunContext = function (runContext, ee) {
  if (!this._started) return ee;
  return this._runCtxMgr.bindEE(runContext, ee);
}

// Return true iff the given EventEmitter is bound to a run context.
Instrumentation.prototype.isEventEmitterBound = function (ee) {
  if (!this._started) return false;
  return this._runCtxMgr.isEEBound(ee);
}

// Invoke the given function in the context of `runContext`.
Instrumentation.prototype.withRunContext = function (runContext, fn, thisArg, ...args) {
  if (!this._started) return fn.call(thisArg, ...args);
  return this._runCtxMgr.with(runContext, fn, thisArg, ...args);
}