'use strict'

const fs = require('fs');
const path = require('path');
const hook = require('require-in-the-middle');

const logger = require('./logger');
const appendMsg = 'Module-Hooker:';

class ModuleHooker {

  constructor(modules, agent, modulePath) {
    this.agent = agent;
    this.isRunning = false;
    this.modulesToHook = [];
    this.patchedModules = [];
    this.moduleToFileMap = {};
    this.modulePath = modulePath;

    Object.keys(modules).forEach(m => {
      const mod = modules[m];
      if (!mod || !mod.enabled) return;
      if (!mod.filesToHook) {
        this.modulesToHook.push(m);
        return;
      }

      Object.keys(mod.filesToHook).forEach(f => {
        this.modulesToHook.push(f);
        const file = mod.filesToHook[f];
        if (!file) return;

        if (file.path) {
          this.moduleToFileMap[f] = file.path;
        }
      });
    });
  }

  start() {
    this.isRunning = true;
    const disabled = new Set([...this.agent.getConfig('disabled_modules')]);

    if (this.patchedModules.length) {
      /** this block will execute if we stopTracing and startTracing */

      this.patchedModules.forEach(mod => {
        try {
          const enabled = !disabled.has(mod.name);
          this._patchModule(mod.module, mod.name, mod.version, enabled);
        } catch (e) {
          logger.error(appendMsg, 'error while wrap the Instrument - ' + mod.name, e);
        }
      });
      return;
    }

    const _self = this;
    hook(this.modulesToHook, function (exports, name, basedir) {
      logger.debug(appendMsg, 'going to wrap ' + name);
      if (_self.agent && _self.agent.stopedTracing) return exports;

      const match = _self.patchedModules.find(e => e.name == name);
      if (match) return match.module;

      const enabled = !disabled.has(name);
      let pkg, version;

      if (basedir) {
        pkg = path.join(basedir, 'package.json')
        try {
          version = JSON.parse(fs.readFileSync(pkg)).version;
        } catch (e) {
          logger.error(appendMsg, 'could not wrap ', name, ' module: ', e.message);
          return exports;
        }
      } else {
        version = process.versions.node;
      }

      let patchedModule = null;
      try {
        patchedModule = _self._patchModule(exports, name, version, enabled);
      } catch (e) {
        logger.error(appendMsg, 'error while wrapping the module :' + name, e);
      }

      return patchedModule || exports;
    });
  }

  _patchModule(exports, name, version, enabled) {
    this.patchedModules.push({ name, version, module: exports });
    if (this.isRunning) {
      logger.info(appendMsg, 'wrapping module name: ' + name, ', version: ' + version);
      return this._getModulePatcher(name).start(exports, this.agent, version, enabled);
    } else {
      logger.debug(appendMsg, 'instrumentation is stopped so not going to wrap ' + name);
      return exports;
    }
  }

  _getModulePatcher(name) {
    if (this.moduleToFileMap[name]) {
      return require(this.modulePath + this.moduleToFileMap[name]);
    } else {
      return require(this.modulePath + name);
    }
  }

  stop() {
    this.isRunning = false;
    this.patchedModules.forEach(mod => {
      try {
        this._getModulePatcher(mod.name).stop(mod.module, mod.version);
      } catch (e) {
        logger.error(appendMsg, 'error while unwrap the Instrument - ' + mod.name, e);
      }
    });
  }
}

module.exports = ModuleHooker;