/**
 * An internally used DataView for {@link Ext.form.field.ComboBox ComboBox}.
 */
Ext.define('Ext.ux.OptGroup.BoundList', {
    extend: 'Ext.view.View',
    alias: 'widget.optgroupboundlist',
    alternateClassName: 'Ext.BoundList',
    requires: ['Ext.layout.component.BoundList', 'Ext.toolbar.Paging'],

    /**
     * @cfg {Number} [pageSize=0]
     * If greater than `0`, a {@link Ext.toolbar.Paging} is displayed at the bottom of the list and store
     * queries will execute with page {@link Ext.data.Operation#start start} and
     * {@link Ext.data.Operation#limit limit} parameters.
     */
    pageSize: 0,
    
    /**
     * @cfg {String} [displayField=""]
     * The field from the store to show in the view.
     */

    /**
     * @property {Ext.toolbar.Paging} pagingToolbar
     * A reference to the PagingToolbar instance in this view. Only populated if {@link #pageSize} is greater
     * than zero and the BoundList has been rendered.
     */

    // private overrides
    baseCls: Ext.baseCSSPrefix + 'boundlist',
    itemCls: Ext.baseCSSPrefix + 'boundlist-item',
    listItemCls: '',
    shadow: false,
    trackOver: true,
    refreshed: 0,
    
    // This Component is used as a popup, not part of a complex layout. Display data immediately.
    deferInitialRefresh: false,

    componentLayout: 'boundlist',

    childEls: [
        'listEl'
    ],

    renderTpl: [
        '<div id="{id}-listEl" class="{baseCls}-list-ct" style="overflow:auto"></div>',
        '{%',
            'var me=values.$comp, pagingToolbar=me.pagingToolbar;',
            'if (pagingToolbar) {',
                'pagingToolbar.ownerLayout = me.componentLayout;',
                'Ext.DomHelper.generateMarkup(pagingToolbar.getRenderTree(), out);',
            '}',
        '%}',
        {
            disableFormats: true
        }
    ],

    /**
     * @cfg {String/Ext.XTemplate} tpl
     * A String or Ext.XTemplate instance to apply to inner template.
     *
     * {@link Ext.view.BoundList} is used for the dropdown list of {@link Ext.form.field.ComboBox}.
     * To customize the template you can do this:
     *
     *     Ext.create('Ext.form.field.ComboBox', {
     *         fieldLabel   : 'State',
     *         queryMode    : 'local',
     *         displayField : 'text',
     *         valueField   : 'abbr',
     *         store        : Ext.create('StateStore', {
     *             fields : ['abbr', 'text'],
     *             data   : [
     *                 {"abbr":"AL", "name":"Alabama"},
     *                 {"abbr":"AK", "name":"Alaska"},
     *                 {"abbr":"AZ", "name":"Arizona"}
     *                 //...
     *             ]
     *         }),
     *         listConfig : {
     *             tpl : '<tpl for="."><div class="x-boundlist-item">{abbr}</div></tpl>'
     *         }
     *     });
     *
     * Defaults to:
     *
     *     Ext.create('Ext.XTemplate',
     *         '<ul><tpl for=".">',
     *             '<li role="option" class="' + itemCls + '">' + me.getInnerTpl(me.displayField) + '</li>',
     *         '</tpl></ul>'
     *     );
     *
     */

    initComponent: function() {
        var me = this,
            baseCls = me.baseCls,
            itemCls = me.itemCls;
            
        me.selectedItemCls = baseCls + '-selected';
        me.overItemCls = baseCls + '-item-over';
        me.itemSelector = "." + itemCls;

        if (me.floating) {
            me.addCls(baseCls + '-floating');
        }
        if (!me.tpl) {
            // should be setting aria-posinset based on entire set of data
            // not filtered set
			//alert("me.groupFieldContainer "+Ext.isArray(me.groupField));
			if (me.groupField != 'undefined' && Ext.isArray(me.groupField)) {
				var tplData = [
					'<ul>',
						'<tpl for=".">'
				];
				var padding = 1;
				padding = me.groupField.length;
				var hasTopMostPadding = false;
				for (var i = 0; i < me.groupField.length; i++) {
					var checking = '';
					for(var j=i-1;j>=0;j--) {
						checking += ' || parent[xindex - 2][\'' + me.groupField[j] + '\'] != values[\'' + me.groupField[j] + '\']';
					}
					if (hasTopMostPadding == false)
					{
						hasTopMostPadding = true;
						tplData.push(
							'<tpl if="xindex == 1">',
								'<li class="x-combo-list-group" style="padding-left:' + (i==0?"3":(i * 16)) + 'px;padding-top:2px"><b>{[values["' + me.groupField[i] + '"]]}</b></li>',
							'</tpl>'
						);
						tplData.push(
							'<tpl if="xindex != 1 && parent[xindex - 2][\'' + me.groupField[i] + '\'] != values[\'' + me.groupField[i] + '\']'+checking+'">',
								'<li class="x-combo-list-group" style="padding-left:' + (i==0?"3":(i * 16)) + 'px;padding-top:8px"><b>{[values["' + me.groupField[i] + '"]]}</b></li>',
							'</tpl>'
						);
					}
					else
					{
						tplData.push(
							'<tpl if="xindex == 1 || parent[xindex - 2][\'' + me.groupField[i] + '\'] != values[\'' + me.groupField[i] + '\']'+checking+'">',
								'<li class="x-combo-list-group" style="padding-left:' + (i==0?"3":(i * 16)) + 'px;"><b>{[values["' + me.groupField[i] + '"]]}</b></li>',
							'</tpl>'
						);
					}
				}
				tplData.push(
						'<li role="option" class="' + itemCls + '" style="padding-left:' + (padding * 16) + 'px;">'+ me.getInnerTpl(me.displayField) +'</li>',
					'</tpl>',
				'</ul>');
				me.tpl = Ext.create('Ext.XTemplate', tplData);
			}
			else
			{
				me.tpl = new Ext.XTemplate(
					'<ul><tpl for=".">',
						'<tpl if="xindex == 1">',
							'<li class="x-combo-list-group" style="padding-top:2px;padding-left:3px;"><b>{[this.getGroupLabel(values)]}</b></li>',
						'</tpl>',
						'<tpl if="xindex != 1 && this.getGroupLabel(parent[xindex - 2]) != this.getGroupLabel(values)">',
							'<li class="x-combo-list-group" style="padding-top:0px;padding-left:3px;"><b>{[this.getGroupLabel(values)]}</b></li>',
						'</tpl>',
						'<li role="option" class="' + itemCls + '" style="padding-left:6px;">' + me.getInnerTpl(me.displayField) + '</li>',
					'</tpl></ul>',
					{
						getGroupLabel: function (values) {
						  return values.groupField
						}
					}
				);
			}

        } else if (Ext.isString(me.tpl)) {
            me.tpl = new Ext.XTemplate(me.tpl);
        }

        if (me.pageSize) {
            me.pagingToolbar = me.createPagingToolbar();
        }

        me.callParent();
    },

    beforeRender: function() {
        var me = this;

        me.callParent(arguments);

        // If there's a Menu among our ancestors, then add the menu class.
        // This is so that the MenuManager does not see a mousedown in this Component as a document mousedown, outside the Menu
        if (me.up('menu')) {
            me.addCls(Ext.baseCSSPrefix + 'menu');
        }
    },

    /**
     * @private
     * Boundlist-specific implementation of the getBubbleTarget used by {@link Ext.AbstractComponent#up} method.
     * This links to the owning input field so that the FocusManager, when receiving notification of a hide event,
     * can find a focusable parent.
     */
    getBubbleTarget: function() {
        return this.pickerField;
    },

    getRefItems: function() {
        return this.pagingToolbar ? [ this.pagingToolbar ] : [];
    },

    createPagingToolbar: function() {
        return Ext.widget('pagingtoolbar', {
            id: this.id + '-paging-toolbar',
            pageSize: this.pageSize,
            store: this.store,
            border: false,
            ownerCt: this,
            ownerLayout: this.getComponentLayout()
        });
    },

    // Do the job of a container layout at this point even though we are not a Container.
    // TODO: Refactor as a Container.
    finishRenderChildren: function () {
        var toolbar = this.pagingToolbar;

        this.callParent(arguments);

        if (toolbar) {
            toolbar.finishRender();
        }
    },
    
    refresh: function(){
        var me = this,
            toolbar = me.pagingToolbar;
        
        me.callParent();
        // The view removes the targetEl from the DOM before updating the template
        // Ensure the toolbar goes to the end
        if (me.rendered && toolbar && toolbar.rendered && !me.preserveScrollOnRefresh) {
            me.el.appendChild(toolbar.el);
        }  
    },

    bindStore : function(store, initial) {
        var toolbar = this.pagingToolbar;
            
        this.callParent(arguments);
        if (toolbar) {
            toolbar.bindStore(this.store, initial);
        }
    },

    getTargetEl: function() {
        return this.listEl || this.el;
    },

    /**
     * A method that returns the inner template for displaying items in the list.
     * This method is useful to override when using a more complex display value, for example
     * inserting an icon along with the text.
     * @param {String} displayField The {@link #displayField} for the BoundList.
     * @return {String} The inner template
     */
    getInnerTpl: function(displayField) {
        return '{' + displayField + '}';
    },

    onDestroy: function() {
        Ext.destroyMembers(this, 'pagingToolbar', 'listEl');
        this.callParent();
    }
});
