'use strict'

var net = require('net');
var fs = require('fs')
var zlib = require('zlib')

var log = require('../logger');
var safeStringify = require('../utils/stringify')
var profiler = require('../external_libs/profiler');

var MAX_CONNECTION_RETRY = 10;
var CONNECTION_RETRY_INTERVAL = 10000;

var KEY_VALUE_PAIR_SEP = ':-:';
var NEXT_LINE = '\n';
var appendMsg = 'SocketClient :'


/**
 * Socket is responsible for send the datas to the agent
 * as a string.
 * 
 * @param {Object} agent 
 */
function Socket(agent) {
    this.agent = agent;
    this.isConnected = false;
    this.client = '';
    this.retryCount = 0;
    this.isMaxReachOut = false;
    //Retry to write the interval default is 15000
    this.REWRITE_INTERVAL = this.agent.config.retryInterval || 15000;
    this.MAX_REWRITE = this.agent.config.inframetrics_interval;
    this.connectionRetryRef = null;
}
/**
 * Create a new socket connection
 * if socket connection fails, then
 * it will again retry to create after interval
 * @param {string} ip 
 * @param {string} port
 */
Socket.prototype.makeConnection = function (ip, port) {

    try {
        log.info(appendMsg,'Trying to make a socket connection to the Agent, hostname=>', ip, ':', port)
        var self = this;
        this.MAX_REWRITE = this.agent.config.inframetrics_interval / (this.REWRITE_INTERVAL * 2);

        this.client = net.createConnection({
            port: port,
            host: ip
        }, function () {

            CONNECTION_RETRY_INTERVAL = 10000;

            self.isConnected = true;
            self.isMaxReachOut = false;
            self.retryCount = 0
            log.info(appendMsg,'Connected to the Agent Server ' + ip + ':' + port + ' via socket successfully...')

            if (!self.agent.infraManager.isStarted && self.agent.config.inframetrics_enabled) {
                self.agent.infraManager.start();
            }

            // self.client.write("NodeID" + KEY_VALUE_PAIR_SEP +self.agent.config.component_id + NEXT_LINE + "OFFLOAD - PING" , "utf8");
            // self.client.write("OFFLOAD - PING", "utf8");
            // self.client.destroy();
        });

        this.client.setEncoding('utf8')
        this.client.setTimeout(60000 * 30)

        this.client.on('end', function (err) {
            self.isConnected = false;
            log.error(appendMsg,'connection closed, try to reconnect  after ' + CONNECTION_RETRY_INTERVAL + ' millis', err)            
            self.connectionRetry();
        });

        this.client.on('error', function (err) {
            log.error(appendMsg,'connection Error, trying to reconnect after ' + CONNECTION_RETRY_INTERVAL + ' millis', err)
            self.isConnected = false;
            self.connectionRetry();
        });

        this.client.on('data', function (data) {

            log.debug(appendMsg,'New response from the manager', data)
            //Here verify that nodejs socket is connected to the correct agent
            //intial response from the agent will be offload pong message
            if (data === "OFFLOAD - PONG") {
                log.info(appendMsg, "Validated it offload agent....")
                return;
            }
            //change the config data
            self.changeConfig(data)
        })

        this.client.on('timeout', function () {
            log.info(appendMsg, ' connection timeout')
            //  self.client.destroy();
        })

        this.client.on('close', function (error) {
            if (error) log.debug(appendMsg, 'connection closed. is because of transmission error ?' + error);
            //self.connectionRetry();
        });


        this.client.unref()

    } catch (e) {
        log.error(appendMsg,' makeConnection error :', e)
    }
}

/**
 * Write the data to socket
 */
Socket.prototype.write = function (msg) {

    try {
         
        //Just dropping the data , if connection is not available
        //Make rewrite option disable 
        if(!this.isConnected){
            log.info(appendMsg, "agent is not connected so data dropped")
               return;
        }

        var that = this;
        //Here we use safely stringfy the json https://www.npmjs.com/package/json-stringify-safe 
        msg = safeStringify(msg);
        msg = this.generateMsg(msg)
    
        log.debug(appendMsg, "pid:",process.pid , 'MSG', msg)
        //Optimize:- 
        //Note:- Need to compress the data??
        if (true) {
            //Right now there is no compression technique is followed
            that.client.write(msg);

        } else {
            //Dead code but this will used in future optimization
            //Can we use stream ?? for optimization
            zlib.gzip(msg, function (err, zipped) {

                if (err) {
                    log.error('Error while compress the payload in the socket', err);
                    return;
                }

                if (that.isConnected) {
                    that.client.write(zipped)
                    log.debug('posted data to the agent', msg)
                } else {
                    log.debug("not connected to agent so dropping the data")
                    //log.debug("Socket not connected, trying to write again after", that.REWRITE_INTERVAL, " secs")
                   // that.writeAgain(zipped, msg)
                }
            })
        }

    } catch (e) {
        log.error(appendMsg, 'client: while write data', e)
    }
}
/**
 * Retry to connect
 */
Socket.prototype.connectionRetry = function () {
    var self = this;
    this.retryCount++;

    if (this.retryCount >= MAX_CONNECTION_RETRY) {
        this.isMaxReachOut = true;
        log.error(appendMsg,'Exceeded maximum retry connection : Failed connect to the Agent ')
        CONNECTION_RETRY_INTERVAL = 10000;
        //Note:- here may be return instead of proceed to make connection
        //CONNECTION_RETRY_INTERVAL = CONNECTION_RETRY_INTERVAL * 2;
    }

    if(this.connectionRetryRef === null){

        this.connectionRetryRef = setTimeout(function () {
             clearTimeout( self.connectionRetryRef)
            self.connectionRetryRef = null;

            log.info(appendMsg, "client going try to connect server after : " + CONNECTION_RETRY_INTERVAL + " secs")
            self.makeConnection(self.agent.config.agent_host, self.agent.config.agent_port) 

        }, CONNECTION_RETRY_INTERVAL)
    }else{
        log.debug(appendMsg,new Date() , " connection retry timer is not cleared")
    }
}

/**
 * To destroy the socket connection 
 * Note:- this function is not used anywhere right now
 */
Socket.prototype.destroy = function () {
    if (this.client) {
        try {
            this.client.destroy();
        } catch (e) {
            log.error(appendMsg, 'connection destroy ', e)
        }
    }
}
/**
 * Write the data again after socket connected to the java agent 
 * successfully
 * Note:- not used right now.
 */
Socket.prototype.writeAgain = function (zipped, orig) {
    var retries = 0;
    var self = this;

    var timer = setInterval(function () {

        if (retries == self.MAX_REWRITE) {
            log.debug(appendMsg, 'discarded the data :', orig)
            clearInterval(timer)
            return;
        }

        retries++;

        if (self.isConnected) {
            self.client.write(zipped, "utf8");
            log.debug(appendMsg, 'writeAgain : Posted data to the agent', orig)
            clearInterval(timer)
            return;
        } else if (self.isMaxReachOut) {
            clearInterval(timer)
            return;
        }

    }, this.REWRITE_INTERVAL);

    timer.unref();
}


/**
 * Get the socket response and update the config dynamically
 */
Socket.prototype.changeConfig = function (data) {
    try {

       //log.debug("new response from the JAVA agent" , data )
       
        data = JSON.parse(data)

        if (data.hasOwnProperty('inframetrics_enabled')) {

            if (this.agent.config.inframetrics_enabled !== data.inframetrics_enabled) {

                if (!data.inframetrics_enabled) {
                    this.agent.infraManager.stop();
                    log.info(appendMsg, 'inframetrics stopped by manager')
                } else {
                    this.agent.infraManager.start();
                    log.info(appendMsg,'inframetrics started by manager')
                }
            }

            this.agent.config.inframetrics_enabled = data.inframetrics_enabled;
        }


        if (data.hasOwnProperty('inframetrics_interval')) {

            if (this.agent.config.inframetrics_interval != data.inframetrics_interval && this.agent.config.inframetrics_enabled) {
                this.agent.config.inframetrics_interval = parseInt(data.inframetrics_interval);
                this.agent.infraManager.stop();
                this.agent.infraManager.start();
                log.info(appendMsg,'inframetrics interval changed by manager')
            } else {
                log.info(appendMsg,'inframetrics interval and manager interval are both same')
            }
        }

        //{"event_loop_stack_collection":"false","event_stack_collection_threshold":"50"}

        if (data.hasOwnProperty('event_loop_stack_collection')) {

            // console.log("Value received from server event_loop_stack_collection : ", data["event_loop_stack_collection"])

            var enable = data["event_loop_stack_collection"] === 'true'

            data["event_loop_stack_collection"] = enable;

            if (this.agent.config.event_loop_stack_collection !== enable) {
                this.agent.config.event_loop_stack_collection = enable;
                if (enable) {
                    this.agent.infraManager.enableBlockedAt();
                } else {
                    this.agent.infraManager.disableBlockedAt();
                }
            }
        }

        if (data.hasOwnProperty('event_stack_collection_threshold')) {
            data["event_stack_collection_threshold"] = parseInt(data["event_stack_collection_threshold"])

            if (data["event_stack_collection_threshold"] !== this.agent.config.event_stack_collection_threshold) {
                this.agent.infraManager.setBlockedAtTheshold(data["event_stack_collection_threshold"]);
            }
        }


        //Need to take the cpu profiler periodically
        if (data.hasOwnProperty('enable_cpu_profiling')) {

            var enable = data["enable_cpu_profiling"] === 'true'

            data["enable_cpu_profiling"] = enable;

            if (this.agent.config.enable_cpu_profiling !== enable) {

                this.agent.config.enable_cpu_profiling = enable;

                if (enable) {
                    log.debug(appendMsg, "command received from manager to start CPU profiler")
                    this.agent.infraManager.startCPUProfiling();

                } else {
                    log.debug(appendMsg,"command received from manager to stop CPU profiler")
                    this.agent.infraManager.stopCPUProfiling();
                }
            }
        }



        //Need to take the memory profiler periodically
        if (data.hasOwnProperty('enable_memory_profiling')) {

            var enable = data["enable_memory_profiling"] === 'true'

            data["enable_memory_profiling"] = enable;

            if (this.agent.config.enable_memory_profiling !== enable) {

                this.agent.config.enable_memory_profiling = enable;

                if (enable) {
                    log.debug(appendMsg,"command received from manager to start memory profiler")
                    this.agent.infraManager.startMemoryProfiling();

                } else {
                    log.debug(appendMsg,"command received from manager to stop memory profiler")
                    this.agent.infraManager.stopMemoryProfiling();
                }
            }
        }

        // Here btm config params are updated

        if (data.hasOwnProperty('btm')) {

            var btm  = data['btm']
            /**
             * work items
             * 
             * "SQL_Exec_CutOff": "1",
             * "HTTP_HEADERS": "true",
             * HTTP_COOKIES
             * MAX_SQL_QUERY_COUNT
             * HTTP_STATUS_CODE // status code is need in agent side for error validation 
             */

             if(btm.hasOwnProperty('SQL_Exec_CutOff')){
                
                var sqlCutOff = btm['SQL_Exec_CutOff']
                if (this.agent.config.sql_cutoff_time !== sqlCutOff) {
                    this.agent.config.sql_cutoff_time = sqlCutOff
                }
             }

             
             if(btm.hasOwnProperty('HTTP_HEADERS')){
                var headerEnabled = (btm['HTTP_HEADERS'] === 'true' ? true : false );

                if (this.agent.config.show_http_headers !== headerEnabled) {
                    this.agent.config.show_http_headers = headerEnabled
                }
             }

             if(btm.hasOwnProperty('HTTP_COOKIES')){
                var cookieEnabled = (btm['HTTP_COOKIES'] === 'true' ? true : false );

                if (this.agent.config.show_http_cookies !== cookieEnabled) {
                    this.agent.config.show_http_cookies = cookieEnabled
                }
             }

             if(btm.hasOwnProperty('HTTP_STATUS_CODE')){
                var statusCodeEnabled = (btm['HTTP_STATUS_CODE'] === 'true' ? true : false );

                if (this.agent.config.show_http_statuscode !== statusCodeEnabled) {
                    this.agent.config.show_http_statuscode = statusCodeEnabled
                }
             }

        }

        var keys = Object.keys(data);
        for (var i = 0; i < keys.length; i++) {
            //we cant compare two arrays, nested objects so if mostly not needed
            if (this.agent.config[keys[i]] !== data[keys[i]]) {
                //ignoreFileTypes  //ignoredUrlPatterns
                /**if(){
                 *try{
                    this.agent.config[keys[i]] = new RegEx( data[keys[i]]);
                 }catch(e){
                    console.log("REG parse error", e )
                 }
                 
                }**/

                this.agent.config[keys[i]] = data[keys[i]];
            }
        }

    } catch (e) {
        log.error(appendMsg,'changeConfig :', e)
    }
}

/**
 * This one format the data like java agent will be supported
 * Note:-  offload code changes removed this format
 */
Socket.prototype.generateMsg = function (data) {
    var msg = ''
    try {
        msg = "NodeID" + KEY_VALUE_PAIR_SEP + this.agent.config.component_id + NEXT_LINE + "AgentType" + KEY_VALUE_PAIR_SEP + "NODEBTM" + NEXT_LINE + "TestType" + KEY_VALUE_PAIR_SEP + "NODETEST" + NEXT_LINE + "MetricType" + KEY_VALUE_PAIR_SEP + "METRIC NODETYPE" + NEXT_LINE + "MetricData" + KEY_VALUE_PAIR_SEP + data + NEXT_LINE;
        return msg;

    } catch (e) {
        log.error(appendMsg, ' while generating the message', e);
    }
}

/**
 * To Send a cpu and memory dump files to the agent 
 *  here we write the data as a stream 
 * we restricted max dump size <25 Mb
 * @param {string} type  memory || cpu dump
 * @param {string} filePath abssoute location of the dump file 
 */
Socket.prototype.sendFile = function (type, filePath, cb) {
    
    log.debug(appendMsg, "File name send it to the agent : ", filePath)

    //Just dropping the data , if connection is not available
    //Make rewrite data dropped
    if(!this.isConnected){
            log.warn(appendMsg, "agent is not connected so data dump file dropped")
            cb("failed")
            return;
     }

    //  filePath = "D:\\Share\\NODE_BTM_DEMO_APP\\node-mysql-app\\node_modules\\eg-node-monitor\\dump\\"  + "nodejs_app_3000_22_11_2018_15_59.heapsnapshot"
    //Note:- this should be optimized
    var msg = "NodeID" + KEY_VALUE_PAIR_SEP + this.agent.config.component_id + NEXT_LINE + "AgentType" + KEY_VALUE_PAIR_SEP + type + NEXT_LINE + "TestType" + KEY_VALUE_PAIR_SEP + "NODETEST" + NEXT_LINE + "MetricType" + KEY_VALUE_PAIR_SEP + "METRIC NODETYPE" + NEXT_LINE ;

    this.client.write(msg);
    var that = this;

    var readerStream = fs.createReadStream(filePath);

    // Set the encoding to be utf8. 
    readerStream.setEncoding('UTF8');

    // that.client.write(' { "type" : "' + type + '" , "date" : "' + new Date().toDateString() + '" , "data": "')

    // Handle stream events --> data, end, and error
    //Here we removed new line 
    readerStream.on('data', function (chunk) {

        if(type === "memory_dump"){
            chunk = chunk.replace(/\n|\r/g, "")
        }

        that.client.write(chunk)
    });

    readerStream.on('end', function () {
        that.client.write(NEXT_LINE)
        log.debug(appendMsg, "file write to server successfully")
        cb("success")
    });

    readerStream.on('error', function (err) {
        log.debug(appendMsg, "File write error =>", err);
        cb("error")
    });

}

module.exports = Socket;


