'use strict'
/**
 * This copied from npm install blocked-at
 * https://www.npmjs.com/package/blocked-at
 * Detects slow synchronous execution and reports where it started.
 * Requires Node 8+
   License : MIT
 */
var log = require('./../logger');
var semver = require('semver');
var isLowerVersion = false;
var appendMsg = 'Blocked-At :'

if (semver.gte(process.version, '8.2.0')) {
  var asyncHooks = require('async_hooks')
} else {
  log.info(appendMsg, "blocked at module not loaded because of lower version", process.version)
  isLowerVersion = true;
}

const cache = new Map() // WeakMap maybe?

const asyncHooksRegEx = /\((internal\/)?async_hooks\.js:/

function cleanStack(stack) {
  const frames = stack.split('\n')
  // this part is opinionated, but it's here to avoid confusing people with internals
  let i = frames.length - 1
  while (i && !asyncHooksRegEx.test(frames[i])) {
    i--
  }
  return frames.slice(i + 1, stack.length - 1)
}

module.exports = (callback, options) => {

  if (isLowerVersion) {
    return {
      disable: () => {},
      enable: function () {},
      setBlockedAtTheshold : function (threshold) {}
    }
  };

  let continuityId
  options = options || {}
  options.threshold = (options.threshold || 20)
  Error.stackTraceLimit = Infinity
  const asyncHook = asyncHooks.createHook({
    init,
    before,
    after,
    destroy
  })
  const dispatchCallback = (dt, stack) => setImmediate(callback, dt, stack)

  const debugLog = (title, message) => (options.debug && process._rawDebug(title, message))

  function init(asyncId, type, triggerAsyncId, resource) {
    const e = {}
    Error.captureStackTrace(e)
    debugLog('init', asyncId)
    cache.set(asyncId, {
      asyncId,
      type,
      stack: e.stack
    })
  }

  function before(asyncId) {
    debugLog('before', asyncId)
    if (options.trimFalsePositives) {
      continuityId = asyncId
    }
    const cached = cache.get(asyncId)
    if (!cached) {
      return
    }
    cached.t0 = hrtime()
  }

  function after(asyncId) {
    debugLog('after', asyncId)
    if (options.trimFalsePositives && continuityId !== asyncId) {
      // drop for interuptions
      return
    }
    const cached = cache.get(asyncId)
    if (!cached) {
      return
    }
    const t1 = hrtime()
    const dt = (t1 - cached.t0) / 1000
    // process._rawDebug(dt > options.threshold, options.threshold, dt, cached)
    if (dt > options.threshold) {
      debugLog('stack', cached.stack)
      dispatchCallback(dt, cleanStack(cached.stack))
    }
  }

  function destroy(asyncId) {
    cache.delete(asyncId)
  }


  return {
    disable: () => {
      if (isLowerVersion) return;
      asyncHook.disable();
    },
    enable: function () {
      if (isLowerVersion) return;
      asyncHook.enable()
    },

    setBlockedAtTheshold : function (threshold) {
      if (isLowerVersion) return;
      options.threshold =  threshold;
    },
  }
}

function hrtime() {
  const t = process.hrtime()
  return t[0] * 1000000 + t[1] / 1000
}