'use strict'

var fs = require('fs')
var path = require('path')

var hook = require('require-in-the-middle')
var semver = require('semver')

var Transaction = require('./transaction')
var shimmer = require('./shimmer')
var log = require('../logger')
var wrapCore = require('./modules/core')
var appendMsg = 'Instrumentation :'

var MODULES = [
  'bluebird',
  'http',
  'https',
  'http2',
  'mysql',
  'mysql2',
  'mongodb',
  'redis',
  'ioredis',
  'elasticsearch',
  'pg',
  'tedious',
  'memcached',
  'cassandra-driver',
  'kafka-node',
  'generic-pool'
]

module.exports = Instrumentation

function Instrumentation (agent) {
  this._agent = agent
  this._started = false
  this.currentTransaction = null
}

Instrumentation.modules = Object.freeze(MODULES)

Instrumentation.prototype.start = function () {

  var self = this
  this._started = true
  //Wrap core modules
  wrapCore(this)

  if (semver.gte(process.version, '8.2.0')) {
    log.info(appendMsg,"native async-hooks module is used for btm")
    require('./async-hooks')(this)
  } else {
    log.info(appendMsg, 'using async-listener module for btm')
    require('./patch-async')(this)
  }

  var disabled = new Set([])

  log.debug(appendMsg,'adding hook to Node.js module loader')

  hook(MODULES, function (exports, name, basedir) {
    var enabled = !disabled.has(name)
    var pkg, version

    if (basedir) {
      pkg = path.join(basedir, 'package.json')
      try {
        version = JSON.parse(fs.readFileSync(pkg)).version
      } catch (e) {
        log.debug(appendMsg, 'could not wrap ',name,' module: ', e.message)
        return exports
      }
    } else {
      version = process.versions.node
    }

    return self._patchModule(exports, name, version, enabled)
  })
}

Instrumentation.prototype._patchModule = function (exports, name, version, enabled) {
  log.debug(appendMsg, 'wrapping  module name:', name, 'version:' , version)
  return require('./modules/' + name)(exports, this._agent, version, enabled)
}


/**
 * addEndTransactionIs the main file
 */
Instrumentation.prototype.addEndedTransaction = function (transaction) {

  if (this._started) {

    var payload = {
      id: transaction.id,
      name: transaction.name,
      type: transaction.type,
      duration: transaction.duration(),
      timestamp: new Date(transaction._timer.start).toISOString(),
      result: String(transaction.result),
      sampled: transaction.sampled,
      context: null,
      spans: []
    }

    if(transaction._context) log.debug(appendMsg, "transaction context =>", transaction._context)

     //Send the data to the agent  
         this._agent.collector.add({
          'btm': filter(transaction)
      })

  } else {
    log.debug(appendMsg,'ignoring transaction id', {
      id: transaction.id
    })
  }
}


/**
 * Filter the httpoutbound and mysql calls from the transaction 
 * and attached those values to the transaction event_data
 * @param {} transaction 
 */
function filter(transaction) {

  var transaction_event_data = {}

 transaction_event_data = transaction.requestIdentifier || '';
 transaction_event_data.resTime = transaction._timer.durationInMs();
 transaction_event_data.callTrace = [];
 transaction_event_data.callTrace.push({name: transaction.method+' '+transaction.name, type:transaction.type, startTime:transaction._timer.start, duration: transaction._timer.durationInMs()});
  transaction_event_data.externalCallDetails = {
    http: [],
    sql: [],
    nosql : [],
    cache : []
  }
  
  for (var i = 0; i < transaction.spans.length; i++) {
    var span = transaction.spans[i];
    transaction_event_data.callTrace.push({name:span.name, type:span.type, startTime:span._timer.start, duration:span._timer.durationInMs()});
      if (transaction.spans[i].ended) {
          var trace = {}

          if (transaction.spans[i].options && typeof transaction.spans[i].options === 'object') {
              trace = transaction.spans[i].options
          }

          trace.resTime =transaction.spans[i]._timer.durationInMs()

    

      if (transaction.spans[i].type === 'http:outbound') {
        transaction_event_data.externalCallDetails.http.push(trace)
      } else if (transaction.spans[i].type === 'mysql') {
        transaction_event_data.externalCallDetails.sql.push(trace)
      } else if (transaction.spans[i].type === 'postgresql') {
        transaction_event_data.externalCallDetails.sql.push(trace)
      } else if (transaction.spans[i].type === 'cassandra') {
        transaction_event_data.externalCallDetails.sql.push(trace)
      } else if (transaction.spans[i].type === 'mongodb') {
        transaction_event_data.externalCallDetails.nosql.push(trace)
      } else if (transaction.spans[i].type === 'redis') {
        transaction_event_data.externalCallDetails.cache.push(trace)
      } else {
        log.debug(appendMsg, " Instrumentation , ignored others type", transaction.spans[i].type)
      }
    }
  }

  transaction_event_data.externalCallDetails = transaction_event_data.externalCallDetails;

  return transaction_event_data;

}

Instrumentation.prototype.startTransaction = function (name, type) {
  return new Transaction(this._agent, name, type)
}

Instrumentation.prototype.endTransaction = function (result) {
  if (!this.currentTransaction) {
    log.debug(appendMsg, 'cannot end transaction - no active transaction found')
    return
  }
  this.currentTransaction.end(result)
}

Instrumentation.prototype.setDefaultTransactionName = function (name) {
  var trans = this.currentTransaction
  if (!trans) {
    log.debug(appendMsg, 'no active transaction found - cannot set default transaction name')
    return
  }
  trans.setDefaultName(name)
}

Instrumentation.prototype.setTransactionName = function (name) {
  var trans = this.currentTransaction
  if (!trans) {
    log.debug(appendMsg, 'no active transaction found - cannot set transaction name')
    return
  }
  trans.name = name
}

Instrumentation.prototype.buildSpan = function () {
  if (!this.currentTransaction) {
    log.debug(appendMsg, 'no active transaction found - cannot build new span')
    return null
  }

  return this.currentTransaction.buildSpan()
}

Instrumentation.prototype.bindFunction = function (original) {
  if (typeof original !== 'function' || original.name === 'egAPMCallbackWrapper') return original

  var ins = this
  var trans = this.currentTransaction
  if (trans && !trans.sampled) {
    return original
  }

  return egAPMCallbackWrapper

  function egAPMCallbackWrapper () {
    var prev = ins.currentTransaction
    ins.currentTransaction = trans
    var result = original.apply(this, arguments)
    ins.currentTransaction = prev
    return result
  }
}

Instrumentation.prototype.bindEmitter = function (emitter) {
  var ins = this

  var methods = [
    'on',
    'addListener'
  ]

  if (semver.satisfies(process.versions.node, '>=6')) {
    methods.push('prependListener')
  }

  shimmer.massWrap(emitter, methods, (original) => function (name, handler) {
    return original.call(this, name, ins.bindFunction(handler))
  })
}

Instrumentation.prototype._recoverTransaction = function (trans) {
  if (this.currentTransaction === trans) return

  // log.debug('Instrumentation : recovering from wrong currentTransaction ', {
  //   wrong: this.currentTransaction ? this.currentTransaction.id : undefined,
  //   correct: trans.id
  // })

  this.currentTransaction = trans
}


// function hrToMills(hrtime) {
//   return parseInt(((hrtime[0] * 1e9) + hrtime[1]) / 1e6);
// }