'use strict'
var log = require('../logger');
var appendMsg = 'profiler :'
var path = require('path')
var pathSepartor = path.sep;

/**
 * Limitations
 * At a time, We cannot run profiler more than one 
 *   // v8-profiler does not compile on Node 7.x -> move to @risingstack/v8-profiler
 * Analyze cpu profiler
 * https://developers.google.com/web/tools/chrome-devtools/rendering-tools/js-execution#top_of_page
 */
var v8Profiler = null;
var profilerExists = false;
//Default
var samplingInterval = 1000;
var profileDuration = 10000;
var maxProfileDuration = 180000;
var timerHandler = null;


//https://stackimpact.com/docs/nodejs-profiling/
/**
 * CPU profiler is disabled by default for Node.js v7.0.0 to v8.9.3 due to memory leak in underlying V8’s CPU profiler.
 if version is greater than  8  we need to use v8-profiler-node8 npm module
 */

var semver = require('semver');
var isLowerVersion = false;


if (semver.gte(process.version, '8.2.0')) {

    try {
        v8Profiler = require("v8-profiler-node8");
        log.info(appendMsg + 'v8-profiler-node8 npm required successfully...')
        profilerExists = true;

    } catch (e) {
        log.error(appendMsg + "v8-profiler-node8  couldn't be required", e)
    }

} else {

    try {
        v8Profiler = require("v8-profiler");
        log.info(appendMsg + 'v8-profiler npm required successfully...')
        profilerExists = true;
    } catch (e) {
        log.error(appendMsg + "V8 Profiler couldn't be required", e)
    }

}



var fs = require('fs');
/**
 * To analyze the cpu profiler in the chrome browser
 * file format should be .cpuprofile
 * Timeout should be 10 secs
 * @param {*} timeout 
 * @param {*} cb 
 */
function cpuProfile(timeout, filePath, cb) {

    if (!profilerExists) return;

    if (timerHandler) {

        // clearTimeout(profilingTimeoutHandle);
        // profilingTimeoutHandle = null;
        // profiler.stopProfiling().delete();
        log.info(appendMsg + 'cannot cpu profile more than one at a time')
        return;
    }

    log.debug(appendMsg + 'Started cpu profiling ms : ', duration)

    v8Profiler.setSamplingInterval(samplingInterval);
    v8Profiler.startProfiling();

    var duration = timeout > maxProfileDuration ? maxProfileDuration : timeout

    timerHandler = setTimeout(function () {

        clearTimeout(timerHandler)
        timerHandler = null;

        var profile = v8Profiler.stopProfiling();
        log.debug(appendMsg + 'stopping cpu profiling')

        filePath = filePath + getTimeStamp() + ".cpuprofile";

        profile.export().pipe(fs.createWriteStream(filePath))
            .on('finish', function () {

                profile.delete();
                v8Profiler.deleteAllProfiles();

                cb("cpu_dump", filePath)

            }).on('error', function (err) {
                profile.delete();
                v8Profiler.deleteAllProfiles();
                log.error(appendMsg + "Error while exporting CPU profiler", err)
                cb("error")
            })

    }, duration)
}

/**
 * Note:- we can use  snapshot.export().pipe(fs.createWriteStream(filePath))
 * 
 * To analyze the heapsnap shot the file format will be
 * .heapsnapshot 
 * @param {*} cb 
 */
function memoryProfile(filePath, cb) {

    if (!profilerExists) {
        log.debug(appendMsg + 'Profiler npm not exists')
        return;
    }

    filePath = filePath + getTimeStamp() + ".heapsnapshot";


    var snapshot = v8Profiler.takeSnapshot();

    log.debug(appendMsg + 'Started MEMORY PROFILING profiling')


    snapshot.export().pipe(fs.createWriteStream(filePath))

        .on('finish', function () {

            log.info(appendMsg + 'write memory profile in a file successfully path: ', filePath);

            snapshot.delete();

            cb("memory_dump", filePath)

        }).on('error', function (err) {
            snapshot.delete();
            // v8Profiler.deleteAllProfiles();
            log.error(appendMsg + "Error while exporting memory profiler", err)
            cb("error")
        })

}


exports.toTreeWithTiming = function toTreeWithTiming(profile) {
    return processRawNode(profile.head);
};

/**
 * This function used for show the cpu trace in sampling method
 * @param {*} rawNode 
 */
function processRawNode(rawNode) {

    /**
      S= self time & total time
     self is how much time was spent doing work directly in that function.
       total is how much time was spent in that function, and in the functions it called.
     */

    //How to  calculate self time or method time and total time in V8 Profiler 
    //https://github.com/jlfwong/chrome2calltree
    /**
        
        "functionName": "(root)",
        "url": "",
        "lineNumber": 0,
        "callUID": 108,
        "bailoutReason": "",
        "id": 1,
        "scriptId": 0,
        "hitCount": 2,
        (%N = Name and line no. & Method name, %M = method time, %T = total time, %c = Class Name, %m = Method name
     */
    var node = {
        functionName: rawNode.functionName,
        url: rawNode.url,
        lineNumber: rawNode.lineNumber,
        sh: rawNode.hitCount,
        th: rawNode.hitCount,
        selfTime: 0,
        totalTime: 0,
        b: getBailoutReason(rawNode),
        children: []
    };

    var children = rawNode.children;
    for (var i = 0, len = children.length; i < len; i++) {
        var childNode = processRawNode(children[i], samplingIntervalMicros);
        node.children.push(childNode);
        node.th += childNode.th;
    }

    node.totalTime = node.th * samplingIntervalMicros;
    node.selfTime = node.sh * samplingIntervalMicros;

    return node;
}


function getBailoutReason(rawNode) {
    var reason = rawNode.bailoutReason;
    if (!reason || reason === 'no reason') {
        return undefined;
    }
    return reason;
}

function format(num) {
    return (num < 10 ? '0' : '') + num;
};

function getTimeStamp() {
    var now = new Date();

    return '_' + format(now.getDate()) +
        '_' + format(now.getMonth() + 1) +
        '_' + now.getFullYear() +
        '_' + format(now.getHours()) +
        '_' + format(now.getMinutes())
};


module.exports.cpuProfile = cpuProfile;
module.exports.memoryProfile = memoryProfile;
module.exports.isExists = profilerExists;