'use strict'

var cluster = require('cluster');
var pathSepartor = require('path').sep;
var path = require('path');
var fs = require('fs')

var log = require('./logger');
var gcStats = require('./external_libs/gc_stats');
var eventloop = require('./external_libs/eventloop');
var lag = require('./external_libs/eventloop-lag')(1000);
var blocked = require("./external_libs/blocked_at");
var profiler = require("./external_libs/profiler");
var appendMsg = 'InfraManager :'


var MICROS_TO_SECONDS = 1e6;
var NANOS_TO_SECONDS = 1e9;
var BYTES_TO_MB = 1048576;
var TO_PERCENTAGE = 100;
var CPU_COUNT = require('os').cpus().length;
//var SAMPLE_INTERVAL = 15e3;
var CHECK_CLUSTER_INTERVAL = 5000;
var MAX_BLOCKED_SIZE = 10;

log.info(appendMsg,'Number of cpu cores :', CPU_COUNT);


/**
 * We can implement Getload percentage (Works only linux env)
 * getLoad
 * Math.floor(os.loadavg()[0] * 100 / this.cpuCount),
 */
var cpuUsageExists;

if (process.cpuUsage) {
    cpuUsageExists = true;
} else {
    cpuUsageExists = false;
    log.error(appendMsg,'Process.cpuUsage is not a function, cpu usage supported from nodejs V6.1.0')
}


/**
 * To collect overall infra level metrics like
 * cpu, memory, eventloop, garbage collection , eventloop is blocked
 * cpu and memory dump periodically
 * 
 * default collect in interval is 5 mins once
 */
function InfraManager(agent) {
    this.agent = agent;
    this.collectInterval = this.agent.config.inframetrics_interval | 300000;
    this.socket = this.agent.socket;
    this.lastCpuTime;
    this.lastIntervalTime;
    this.timer = '';
    this.isStarted = false;
    this.gc = {
        tick: 0,
        time: 0,
        minor: 0,
        minorTime: 0,
        inc: 0,
        incTime: 0,
        full: 0,
        fullTime: 0,
        releasedMem: 0
    }

    this.cpuProfilerHandler = null;
    this.memoryProfilerHandler = null;
    this.deleteFilesHandler = null;
    var self = this;

    this.BlockedAt = []

    function sortNumber(a, b) {
        return a.execTime - b.execTime;
    }

    /*
     * This module blocked at gives whether event loop is blocked or not 
     *  supports from  node v8.1.0
     */
    this.monitBlockedAt = blocked((time, stack) => {
        // console.log(new Date().toLocaleString(),"Array is ===",  Array.isArray(stack))
        //  console.log(new Date().toLocaleString(), ` Blocked for ${time}ms, operation started here:`, stack)
        // console.log("Stack===", stack.join('').replace(/\n|\r/g, ""))
        if (this.BlockedAt.length > MAX_BLOCKED_SIZE) {

            //   console.log(new Date().toLocaleString(), ` Blocked for ${time}ms, operation started here:`, stack)

            this.BlockedAt.push({
                date: getTimeStamp(),
                execTime: time.toFixed(0),
                stack: stack.join('~;~').replace(/\n|\r/g, "") //stack.join('\r')
            })

            this.BlockedAt.sort(sortNumber);
            this.BlockedAt.splice(0, 1)

        } else {
            this.BlockedAt.push({
                date: getTimeStamp(),
                execTime: time.toFixed(0),
                stack: stack.join('~;~').replace(/\n|\r/g, "")
            })

        }
    }, {
        threshold: this.agent.config.blocked_threshold
    })

    //Reference
    //http://jayconrod.com/posts/55/a-tour-of-v8-garbage-collection
    //Collecting gc metrics
    gcStats.on('stats', function (stats) {
        // time is in nanoseconds
        try {
            self.gc.time += stats.pauseMS;

            switch (stats.gctype) {
                case 1:
                    self.gc.minor += 1;
                    self.gc.minorTime += stats.pauseMS;
                    break
                case 2:
                    self.gc.full += 1;
                    self.gc.fullTime += stats.pauseMS;
                    break
                case 3:
                    self.gc.minor += 1
                    self.gc.full += 1
                    break
                case 4:
                    self.gc.inc += 1;
                    self.gc.incTime += stats.pauseMS;
                    break
            }

            self.gc.tick += 1;

            //Math.abs
            self.gc.releasedMem += (Math.abs(stats.diff.usedHeapSize) / BYTES_TO_MB);

        } catch (e) {
            log.error(appendMsg,'GC stats emitter error:', e);
        }
    })
}

InfraManager.prototype.start = function () {
    try {
        log.info(appendMsg,'starting...');

        if (this.agent.config.inframetrics_interval) {
            this.collectInterval = this.agent.config.inframetrics_interval
        }

        if (this.agent.config.enable_cpu_profiling === true) {
            this.startCPUProfiling()
        }

        if (this.agent.config.enable_memory_profiling === true) {
            this.startMemoryProfiling()
        }

        //Collect cluster details
        this.isCluster()

        this.isStarted = true;
        var self = this;


        this.timer = setInterval(function () {

            log.debug(appendMsg,'Started collecting Inframetrics...')
            try {
                self.sendMetrics();
            } catch (e) {
                log.error(appendMsg,'start timer', e)
            }

        }, this.collectInterval)

        this.timer.unref();

    } catch (e) {
        log.error(appendMsg,' start', e)
    }
}

/**
 * To Get current cpu usage
 * Process.cpuUsage() supports from nodejs V6.1.0
 * @param  {Object} lastCpuTime contains user and system time in micros
 * @return {Object} cputime contains user and system time in micros
 */
InfraManager.prototype.getCpuTime = function (lastCpuTime) {
    try {
        return process.cpuUsage(lastCpuTime);
    } catch (e) {
        log.error(appendMsg,'getCpuTime ', e);
        return;
    }
}

InfraManager.prototype.getCpuUsage = function () {
    try {
        var currCpuTime = this.getCpuTime(this.lastCpuTime);
        this.lastCpuTime = this.getCpuTime();
        log.debug(appendMsg,'CPU time :', currCpuTime)
        return this.getCpuPercent(currCpuTime.user / MICROS_TO_SECONDS, currCpuTime.system / MICROS_TO_SECONDS);
    } catch (e) {
        log.error(appendMsg, 'getCpuUsage', e)
    }
}

/**
 * To get cpu percent
 * Get the elapsed time by  current time - prev time
 * Date.now() returns current time in millis 
 * @param {*} user user time by seconds 
 * @param {*} system system usage by seconds
 * @return {User, system, overall cpu usage}
 */
InfraManager.prototype.getCpuPercent = function (user, system) {

    var diffTime;
    var userPercent;
    var systemPercent;

    try {
        if (!this.lastIntervalTime) {
            diffTime = process.uptime();
        } else {
            // diffTime = process.hrtime(this.lastIntervalTime);
            // diffTime = diffTime[0] + (diffTime[1] / NANOS_TO_SECONDS);
            diffTime = (Date.now() - this.lastIntervalTime) / 1000;
        }

        // this.lastIntervalTime = process.hrtime();
        this.lastIntervalTime = Date.now()

        diffTime = diffTime * CPU_COUNT;

        userPercent = ((user) / diffTime) * 100 * CPU_COUNT;
        systemPercent = ((system) / diffTime) * 100 * CPU_COUNT;

        return {
            'userTime': user.toFixed(4),
            'systemTime': system.toFixed(4),
            'userPct': userPercent.toFixed(4),
            'systemPct': systemPercent.toFixed(4),
            'totalPct': (userPercent + systemPercent).toFixed(4)
        }

    } catch (e) {
        log.error(appendMsg, ' getCpuPercent', e)
    }
}

/**
 * To calculate  process memory usage
 * @return  {Object}  rss, heaptotal,nonheap,heapused in MBS
 */
InfraManager.prototype.memoryUsage = function () {
    try {
        var memory = process.memoryUsage();

        log.debug(appendMsg,'Memory usage info :', memory)

        return {
            'physical': (memory.rss / BYTES_TO_MB).toFixed(2),
            'max_heap': (memory.heapTotal / BYTES_TO_MB).toFixed(2),
            'free_heap': ((memory.heapTotal - memory.heapUsed) / BYTES_TO_MB).toFixed(2),
            'non_heap': ((memory.rss - memory.heapTotal) / BYTES_TO_MB).toFixed(2),
            'heap_used': (memory.heapUsed / BYTES_TO_MB).toFixed(2)
        }

    } catch (e) {
        log.error(appendMsg,'memoryUsage', e)
    }
}

/**
 * Collect all the metrics and send it to queue
 */
InfraManager.prototype.sendMetrics = function () {

    try {
        //Store  cpu and memory data's in object, then send it to collector 
        var data = {}

        if (cpuUsageExists) {
            data.cpuUsage = this.getCpuUsage();
        }

        data.memoryUsage = this.memoryUsage();

        this.gc.releasedMem = this.gc.releasedMem.toFixed(4);

        data.gc = this.gc;
    
        log.debug(appendMsg,'GC stats metrics :', data.gc);

        data.eventloop = eventloop.sense();
        data.eventloop.blockedAt = this.BlockedAt;
        log.debug(appendMsg,'Eventloop stats metrics :', data.eventloop.blockedAt.length);

        //Added with eventloop lag
        data.eventloop.lag = Math.round(lag() * 100) / 100;

        //Add these parameter for check if it is a nodejs cluster
        data.processID = this.agent.config.pid;
        data.isMaster = (this.agent.config.isMaster).toString();
        data.workerCount = this.agent.config.workerCount;

        //Send the data to the queue
        this.agent.collector.add({
            'infra': data
        })

        this.reset();

    } catch (e) {
        log.error(appendMsg,'sendMetrics', e)
    }
}

InfraManager.prototype.stop = function () {

    try {

        clearInterval(this.timer);

        // gcStats.on = function () {};

        // eventloop = {
        //     sense: function () {
        //         return {}
        //     }
        // }

        this.isStarted = false;
        log.info(appendMsg,' stopped')

    } catch (e) {
        log.error(appendMsg,' stop:', e)
    }
}

InfraManager.prototype.reset = function () {
    this.gc = {
        tick: 0,
        time: 0,
        minor: 0,
        minorTime: 0,
        inc: 0,
        incTime: 0,
        full: 0,
        fullTime: 0,
        releasedMem: 0
    }

    this.BlockedAt = []

}

InfraManager.prototype.enableBlockedAt = function () {
    log.info(appendMsg,"enabling blocked at module")
    this.monitBlockedAt.enable();
}

InfraManager.prototype.disableBlockedAt = function () {
    log.info(appendMsg,"disabling blocked at module")
    this.monitBlockedAt.disable();
}

InfraManager.prototype.setBlockedAtTheshold = function (threshold) {
    log.info(appendMsg,"setting blocked at threshold value=>" , threshold)
    this.monitBlockedAt.setBlockedAtTheshold(threshold);
}

/**
 * To get current process is master or not 
 */
InfraManager.prototype.isCluster = function () {

    var interval = setInterval(() => {
        if (cluster.isMaster) {

            this.agent.config.isMaster = true;

            if (this.agent.config.workerCount !== Object.keys(cluster.workers || {}).length) {
                this.agent.config.workerCount = Object.keys(cluster.workers || {}).length;
                log.debug(appendMsg,'cluster worker count changed :', this.agent.config.workerCount)
                // agent.socket.write( JSON.stringify({'workerCount' :  agent.config.workerCount }));
            }
        }

    }, CHECK_CLUSTER_INTERVAL);

    interval.unref();

}

/**
 * process._getActiveRequests() to get a list of active I/O requests 
 * and process._getActiveHandles() to get a list of open handles/file descriptors.
 * 
 * https://github.com/mafintosh/why-is-node-running
 * 
 * Note:-This one is not added in the infra metrics
 */

InfraManager.prototype.collectActiveMetrics = function () {

    var activeRequests = 0;
    var activeHandles = 0;
    var counter = 0;

    var interval = setInterval(function () {
        activeRequests += process._getActiveHandles().length;
        activeHandles += process._getActiveRequests().length;
        counter++;
    }, 10000)

    interval.unref();

}

/**
 * To start cpu dump
 */
InfraManager.prototype.startCPUProfiling = function () {

    if (!profiler.isExists) {

        log.debug(appendMsg,'CPU profiler not exists...')
        return;
    }

    log.debug(appendMsg,'CPU profiler enabled  by the manager...')
    //  if (this.agent.config.inframetrics_interval) {

    var interval = this.agent.config.inframetrics_interval
    //Remove special characters
    var filename = this.agent.config.component_id.replace(/^\[(.+)\]$/, '$1').replace(":", "_");

    var that = this;


    this.cpuProfilerHandler = setInterval(function () {

        profiler.cpuProfile(that.agent.config.cpu_profiling_duration, that.agent.config.cpu_dump_location + pathSepartor + filename, function (type, savedPath) {

            if (type == "error") return;

            fs.stat(savedPath, function (err, data) {

                if (err) {
                    log.error(appendMsg,"cpu profiler dumped file not found ", err)
                    return
                }

                if (bytesToMb(data["size"]) < that.agent.config.max_memory_dump_size) {

                    that.agent.socket.sendFile(type, savedPath, function(){
                        fs.unlink(savedPath , function(err){
                            if (err){
                                log.error(appendMsg,"error while delete cpu dump file after sent it to agent: ", err);
                                return
                            } 
                            
                            log.debug(appendMsg,"cpu dump file deleted successfully after sent to the agent path: ", savedPath);
                        })
                    })          

                } else {
                    log.info(appendMsg,"cpu dump file size greater than set size so it cannot send, saved Path", savedPath)
                }

            })


        })

    }, interval)

    this.cpuProfilerHandler.unref();

    //Delete dump files periodically
    if (!this.deleteFilesHandler) {
        this.deleteDumpFiles();
    }

}

InfraManager.prototype.stopCPUProfiling = function () {
    if (this.cpuProfilerHandler) {
        log.debug(appendMsg,'CPU profiler Stopped by the manager...')
        clearInterval(this.cpuProfilerHandler)
        this.cpuProfilerHandler = null;
    }

    if (!this.memoryProfilerHandler) {
        clearInterval(this.deleteFilesHandler)
        this.deleteFilesHandler = null;
    }
}
/**
 * To take memory dump
 */
InfraManager.prototype.startMemoryProfiling = function () {

    if (!profiler.isExists) {
        log.error(appendMsg,"profiler library does not exists...")
        return;
    }

    log.debug(appendMsg,'Memory profiler enabled  by the manager')

    //  if (this.agent.config.inframetrics_interval) {

    var interval = this.agent.config.inframetrics_interval
    //Remove special characters
    var filename = this.agent.config.component_id.replace(/^\[(.+)\]$/, '$1').replace(":", "_");

    var that = this;

    this.memoryProfilerHandler = setInterval(function () {

        profiler.memoryProfile(that.agent.config.cpu_dump_location + pathSepartor + filename, function (type, savedPath) {

            if (type == "error") return;

            fs.stat(savedPath, function (err, data) {

                if (err) {
                    log.error(appendMsg,"memeory profiler dumped file not found ", err)
                    return
                }


                if (bytesToMb(data["size"]) < that.agent.config.max_memory_dump_size) {

                        that.agent.socket.sendFile(type, savedPath, function(){
                            fs.unlink(savedPath , function(err){
                                if (err){
                                    log.error(appendMsg,"error while delete memory dump file after sent it to agent: ", err);
                                    return
                                } 
                                
                                log.debug(appendMsg,"memory dump file deleted successfully after sent to the agent path: ", savedPath);
                            })
                        })          

                } else {
                    log.info(appendMsg,"memory dumped file size greater than set size so it cannot send, saved Path ", savedPath)

                }
            })
        })

    }, interval)

    this.memoryProfilerHandler.unref();

    //Delete dump files periodically
    if (!this.deleteFilesHandler) {
        this.deleteDumpFiles();
    }
}


InfraManager.prototype.stopMemoryProfiling = function () {
    if (this.memoryProfilerHandler) {
        log.debug(appendMsg,'Memory profiler Stopped by the manager...')
        clearInterval(this.memoryProfilerHandler)
        this.memoryProfilerHandler = null;
    }

    if (!this.cpuProfilerHandler) {
        clearInterval(this.deleteFilesHandler)
        this.deleteFilesHandler = null;
    }
}

 var DAY_TO_MILLS = 86400000;

/**
 * To delete the dump files periodically
 */
InfraManager.prototype.deleteDumpFiles = function () {

    log.info(appendMsg,"deleteDumpFiles service activated")
   
    var that = this;

    this.deleteFilesHandler = setInterval(function () {

        deleteFileRecursive(that.agent.config.cpu_dump_location, ".cpuprofile")
        deleteFileRecursive(that.agent.config.memory_dump_location, ".heapsnapshot")

    }, DAY_TO_MILLS)

    this.deleteFilesHandler.unref();
}


/**
 *To Delete cpu and memory dumps 
 * */
function deleteFileRecursive(folder, fileType) {

    fs.readdir(folder, function (err, files) {

        if (err) {
            log.error(appendMsg,"error while readdir for delete files periodically " + folder)
            return;
        }

        files.forEach(function (file, index) {

            if (file.endsWith(fileType)) {
                var filePath = path.join(folder, file);

                fs.stat(filePath, function (err, stat) {

                    var endTime
                    var now;

                    if (err) {
                        return log.error(appendMsg,"error while readdir for to delete a file ", err);
                    }

                    now = new Date().getTime();
					//check if file created one day before
                    endTime = new Date(stat.ctime).getTime() + DAY_TO_MILLS;

                    if (now > endTime) {
                        fs.unlink(filePath, function (err) {

                            if (err) return log.error(appendMsg,"error while unlink the file " + err);

                            log.debug(appendMsg,"peridical file deleted successfully " + filePath)

                        });
                    }
                });
            }
        });
    });
}


function format(num) {
    return (num < 10 ? '0' : '') + num;
};

function getTimeStamp() {
    var now = new Date();

    return format(now.getDate()) +
        '/' + format(now.getMonth() + 1) +
        '/' + now.getFullYear() +
        ' ' + format(now.getHours()) +
        ':' + format(now.getMinutes()) +
        ':' + format(now.getSeconds())
};

function bytesToMb(bytes) {
    //Math.pow(2,20)
    return bytes / 1048576
}




module.exports = InfraManager;