'use strict'

var url = require('url')

var endOfStream = require('end-of-stream')
var log = require('../logger')

var semver = require('semver')
var SUPPORT_PREFINISH = semver.satisfies(process.version, '>=0.12')
var moment = require('moment-timezone')

var tagInbound = 'HttpInbound :'
var appendMsg = 'Http shared :'
var timezone = ''

var eGGUID = "eG-GUID";
var GUID_SEPARATOR = "#~#";

try {
  timezone = moment.tz(moment.tz.guess()).zoneAbbr();
  log.info(appendMsg,'System Timezone:', timezone)
} catch (e) {
  log.error(appendMsg,'timezone not found', e)
}

const transactionForResponse = new WeakMap()
exports.transactionForResponse = transactionForResponse

exports.instrumentRequest = function (agent, moduleName) {
  var ins = agent._instrumentation
  var blackListed = agent.config.black_listed;

  return function (orig) {
    return function (event, req, res) {
      if (event === 'request') {

        log.debug(appendMsg,'New incoming', moduleName, 'request call url:', req.url)

        var startTime = new Date();
        //CHECK url is Blacklisted
        if (isUrlExcluded(req.url)) {
          log.debug(appendMsg,'ignoring blacklisted request to url:', req.url)
          // don't leak previous transaction
          agent._instrumentation.currentTransaction = null
        } else {
          var trans = agent.startTransaction()
          trans.type = 'http:inbound';
          trans.name = req.url;
          trans.method = req.method;
          var startTime = new Date();
          //Add the parameters here
          trans.requestIdentifier = getParams(req, agent.config)
          trans.requestIdentifier.reqTime = getTimeStamp(startTime);


          //Assign component id for every request
          trans.requestIdentifier.component_id = agent.config.component_id

          //Re assign the guid & node order if not exists
          trans.requestIdentifier.eg_guid ? trans.id = trans.requestIdentifier.eg_guid : trans.requestIdentifier.eg_guid = trans.id;
          trans.requestIdentifier.nodeOrder ? trans.nodeOrder = trans.requestIdentifier.nodeOrder : trans.requestIdentifier.nodeOrder = '1';

          transactionForResponse.set(res, trans)

          ins.bindEmitter(req)
          ins.bindEmitter(res)

          //End of stream
          endOfStream(res, function (err) {
            if (!err) {

              //To get the content length
              //Add response values code here
              // if (typeof res.getHeader === 'function') {
              //   trans.requestIdentifier.resContentLength = res.getHeader("Content-Length") || 0;
              // }
              // trans.requestIdentifier.statusMsg = res.statusMessage;

              if (req.route) {
                trans.requestIdentifier.uri = req.route.path || (req.route.regexp && req.route.regexp.source) || ''
              }


              trans.requestIdentifier.statusCode = res.statusCode || '';

              log.debug(appendMsg, "call ended with status code:", res.statusCode, 'id:', trans.id)

              return trans.end()
            }

            //Note we can capture  socket closed with in active HTTP request error

            // Handle case where res.end is called after an error occurred on the
            // stream (e.g. if the underlying socket was prematurely closed)
            res.on('prefinish', function () {
              trans.requestIdentifier.statusCode = res.statusCode || '';
              trans.end()
            })

            // res.on('finish', function () {
            //   //Recently added
            //   trans.requestIdentifier.statusCode = res.statusCode || '';
            //   log.debug(tagInbound + "emitter on finish", res.statusCode)
            //   trans.end()
            // })

          })
        }
      }

      return orig.apply(this, arguments)
    }
  }


  /**
   * Number of options 
   * Filter by status code
   * filter by Filetype
   * filter by regex pattern 
   * filter urls
   * filter by req header contains req.headers['user-agent'] 
   * @param {} url 
   * @param {*} statusCode 
   */

  function isUrlExcluded(url) {
   
    var splittedURL = url.split('?');
    var urlPath = splittedURL[0];

    try {

      if (blackListed.startsWith instanceof RegExp && blackListed.startsWith.test(urlPath)) {
        return true;
      } else {
       // log.debug("Its startWith is not a type of REGEX")
      }

      if (blackListed.endsWith instanceof RegExp && blackListed.endsWith.test(urlPath)) {
        return true;
      } else {
        //log.debug("Its endsWith is not a type of REGEX")
      }


      for (var i = 0; i < blackListed.regexes.length; i++) {

        if (blackListed.regexes[i].test(url)) {
          log.debug(appendMsg,  'ignored url: pattern', url)
          return true;
        }
      }

      /** 
            if (blackListed.urls.indexOf(url) > -1) {
              log.debug(tagInbound + 'ignored url:', url)
              return true;
            }

            if (blackListed.patterns.length > 0) {
              for (var i = 0; i < blackListed.patterns.length; i++) {
                if (blackListed.patterns[i].test(url)) {
                  log.debug(tagInbound + 'ignored url: pattern', url)
                  return true;
                }
              }
            }

            if (blackListed.user_agent.length > 0) {
              for (var i = 0; i < blackListed.user_agent.length; i++) {
                if (blackListed.user_agent[i].test(url)) {
                  debug(tagInbound + 'ignored user agent  pattern', url)
                  return true;
                }
              }
            }

            if (blackListed.file_types instanceof RegExp && blackListed.file_types.test(url)) {
              return true;
            }
      */
    } catch (e) {
      log.error(appendMsg, 'isUrlExcluded fn throw error', e, 'URL:', url)
    }

    return false;
  }

}


//OUT BOUND CALLS
exports.traceOutgoingRequest = function (agent, moduleName) {
  var spanType = 'http:outbound'
  var ins = agent._instrumentation

  return function (orig) {
    return function () {


      var span = agent.buildSpan()
      var id = span && span.transaction.id

      log.debug(appendMsg, 'new', moduleName, 'Oubound call, trans id:',id)


      if (span) {
        /**
         * In Eg to every outbound request 
         * we have to send the current transaction id + node order 
         * So that we can show the topology page by tag and follow method
         */

        var order = span && span.transaction.nodeOrder ? span.transaction.nodeOrder + '.' + (++span.transaction.nodeOrderCounter) : '';
        var eg_guid = id + GUID_SEPARATOR + order;

        //Attach node order in EGUID header

        if (!arguments[0].headers && id) {
          arguments[0].headers = {}
        }

        if (id && order) {
          arguments[0].headers[eGGUID] = eg_guid
        }
      }

      var req = orig.apply(this, arguments)

      if (!span) return req


      /**
            url: 'url with querystring',
            resTime: '',
            method: '',
            statusCode: ''
            nodeOrder: '-'
       */
      //Note:-handle the APM agent url and ignore that request
      var reqInfoArg = arguments[0];
      var options = {}
	  var protocol = 'http';
      if(moduleName !== 'http'){
        protocol = 'https'
      }
      var port = reqInfoArg.port;
      if(port){
        port = ':'+port;
      }
      else{
        port = '';
      }
      options.uri = protocol+'://'+reqInfoArg.host+port+reqInfoArg.path;
      options.method = req.method
      options.nodeOrder = order
      options.resTime = ''
      options.error = ''

      span.options = options

      ins.bindEmitter(req)

      var name = req.method + ' ' + req._headers.host + url.parse(req.path).pathname
      span.start(name, spanType)
      req.on('response', onresponse)

      //to capture external error
      req.on('error', function (err) {

        if (err) {
          span.options.statusCode = 404
          span.options.error = err.toString()
          if (err.stack) span.options.error = JSON.stringify(err.stack)
        }

        span.end()
      })

      return req

      function onresponse(res) {

        log.debug(appendMsg, 'Response came for http oubound request', {
          id: id,
          url: span.options.uri
        })

        ins.bindEmitter(res)

        if (res.prependListener) {
          // Added in Node.js 6.0.0
          res.prependListener('end', onEnd)
        } else {
          var existing = res._events && res._events.end
          if (!existing) {
            res.on('end', onEnd)
          } else {
            if (typeof existing === 'function') {
              res._events.end = [onEnd, existing]
            } else {
              existing.unshift(onEnd)
            }
          }
        }

        function onEnd() {
          log.debug(appendMsg,'http outbound request event is completed', {
            id: id,
            url: span.options.uri
          })
          span.options.statusCode = res.statusCode || ''
          span.end()
        }
      }
    }
  }
}


function getParams(req, config) {

  var requestIdentifier = {};

  try {

    //Note:- This url maybe contains querystring
    requestIdentifier.uri = req.url;
    requestIdentifier.queryString = '-'

    //To remove querystring from the url
    if (requestIdentifier.uri.indexOf('?') > -1) {
      if (req.query) {
        requestIdentifier.queryString = req.query;
      } else {
        var splittedURL = req.url.split('?');
        requestIdentifier.uri = splittedURL[0];
        requestIdentifier.queryString = splittedURL[1];
      }
    }

    //For expressjs pattern path
    if (req.route) {
      //To get regular exp pattern
      //req.route.regexp && req.route.regexp.source
      //To handle ['/api/v1/hello', '/api/v1/hi'] pattern,
      if (typeof req.route.path !== 'string') {
        requestIdentifier.uri = req.route.path.toString();
      } else {
        requestIdentifier.uri = req.route.path;
      }
    }

    //Remote host
    //Refer this npm
    //https://www.npmjs.com/package/request-ip
    var clientIP = req.headers['x-forwarded-for'] || req.connection.remoteAddress;
    //To remove the special chars
    if (clientIP && clientIP.substr(0, 7) == "::ffff:") {
      clientIP = clientIP.substr(7)
    }

    if (clientIP === '::1') {
      clientIP = '127.0.0.1'
    }
    //Add new field
    //Gateway IP for load_balancer
    //
    //HTTP headers are being converted to lower case by http.js
    if (req.headers[eGGUID.toLowerCase()]) {

      log.debug(appendMsg,'http request contains eGGUID found ', req.headers[eGGUID])

      var eg_params = req.headers[eGGUID.toLowerCase()].split(GUID_SEPARATOR)
      requestIdentifier.eg_guid = eg_params[0]
      requestIdentifier.nodeOrder = eg_params[1]

    } else {
      log.debug(appendMsg, 'request not contain eG-GUID')
    }

    // if (req.headers['eg_guid']) {
    //   var eg_params = req.headers['eg_guid'].split('#')
    //   requestIdentifier.eg_guid = eg_params[0]
    //   requestIdentifier.nodeOrder = eg_params[1]
    // }

    requestIdentifier.remote_host = clientIP || '-';
    //req.sessionID  this is not works after
    requestIdentifier.sessid = req.sessionID || '-';
    requestIdentifier.referer = req.headers['Referer'] || req.headers['referer'] || '-';
    requestIdentifier.method = req.method;

    requestIdentifier.headers = config.show_http_headers ? JSON.stringify(req.headers || '-') : "-";
    requestIdentifier.cookies = config.show_http_cookies ? req.headers['cookie'] || '-' : "-";
    requestIdentifier.tz = timezone;
    //Additional:-Status message password doesnt match, invalid input etc...

    requestIdentifier.userAgent = req.headers['user-agent'] || '-'
    //requestIdentifier.host = req.headers['host']

    if (req.headers) {
      //Temp 
      //   requestIdentifier.reqContentLength = req.headers['content-length'] || 0;
    }

    requestIdentifier.appName = '-'
    requestIdentifier.contextName = '-' //identify context name
    requestIdentifier.threadGuid = '-'
    requestIdentifier.resTime = ''
    requestIdentifier.reqType = 'Web'
    requestIdentifier.transType = '-' //Filter by default 

    requestIdentifier.slowThreshOld = 4000
    requestIdentifier.stallThreshOld = 60000
    requestIdentifier.threadId = '-'
    requestIdentifier.device = '-'
    requestIdentifier.os = '-'
    requestIdentifier.browser = '-'
    requestIdentifier.country = '-'
    requestIdentifier.location = '-'
    requestIdentifier.position = '-'
    requestIdentifier.preProcessorTime = '-'
    requestIdentifier.username = '-'
    requestIdentifier.businessContext = '-'
    requestIdentifier.cpuTime = '-5'
    requestIdentifier.waitTime = '-5'
    requestIdentifier.blockTime = '-5'
    requestIdentifier.fastSQL = []
    requestIdentifier.patternName = req.url;
    requestIdentifier.kBusinessTest = false

    /**
     * Note:-
     * Res object collected 
     * Temp commented
     */

    // if (typeof res.getHeader === 'function') {
    //   requestIdentifier.resContentLength = res.getHeader("Content-Length") || 0;
    // }
    // requestIdentifier.statusMsg = res.statusMessage;
    //  requestIdentifier.statusCode = res.statusCode || '';


    /**
     EG_GUID
     NODE_ORDER	
     APPLICATION_NAME
     CONTEXT_NAME
     THREAD_GUID
     RESPONSE_TIME
     REMOTE_HOST
     REQUEST_TYPE	 web
     TRANSACTION_TYPE	 filter
     SLOW_URL_THRESHOLD	 4000
     STALLED_URL_THRESHOLD 600000
     URL_PARAMS -
     THREAD_ID - 
     DEVICE - 	
     OS	-
     BROWSER -
     COUNTRY	-
     LOCATION	 - 
     POSITION	 - 
     TRACE_DETAILS -
     PRE_PROCESSOR_TIME
     USERNAME	
     BUSINESS_CONTEXT - 
    CPU_TIME	 -5
    WAIT_TIME	 -5
    BLOCK_TIME -5
     */

  } catch (e) {
    log.error(appendMsg,'error while collecting http request identifier', e)
  }

  return requestIdentifier;
}


function format(num) {
  return (num < 10 ? '0' : '') + num;
};


//dd/MM/yyyy HH:mm:ss
//2018-04-09 05:05:10.0
//2018-04-12 13:09:37.528
function getTimeStamp(ts) {
  var now = new Date(ts);

  return now.getFullYear() + '-' +
    format(now.getMonth() + 1) + '-' +
    format(now.getDate()) +
    ' ' + format(now.getHours()) +
    ':' + format(now.getMinutes()) +
    ':' + format(now.getSeconds()) +
    '.' + format(now.getMilliseconds());
};